#!/usr/bin/env node
"use strict";
/**
 * automatically creates a module for any CloudFormation namespaces that do not
 * have an AWS construct library.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const pkglint = require("@aws-cdk/pkglint");
const fs = require("fs-extra");
const cfnspec = require("../lib");
// don't be a prude:
/* eslint-disable no-console */
/* eslint-disable quote-props */
async function main() {
    const root = path.join(__dirname, '..', '..');
    if (path.basename(root) !== '@aws-cdk') {
        throw new Error(`Something went wrong. We expected ${root} to be the "packages/@aws-cdk" directory. Did you move me?`);
    }
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const cfnSpecPkgJson = require('../package.json');
    const version = cfnSpecPkgJson.version;
    const jestTypesVersion = cfnSpecPkgJson.devDependencies['@types/jest'];
    // iterate over all cloudformation namespaces
    for (const namespace of cfnspec.namespaces()) {
        const module = pkglint.createModuleDefinitionFromCfnNamespace(namespace);
        const lowcaseModuleName = module.moduleBaseName.toLocaleLowerCase();
        const packagePath = path.join(root, module.moduleName);
        // we already have a module for this namesapce, move on.
        if (await fs.pathExists(packagePath)) {
            const packageJsonPath = path.join(packagePath, 'package.json');
            // eslint-disable-next-line @typescript-eslint/no-require-imports
            const packageJson = require(packageJsonPath);
            let scopes = packageJson['cdk-build'].cloudformation;
            if (typeof scopes === 'string') {
                scopes = [scopes];
            }
            if (scopes.indexOf(namespace) !== -1) {
                // V2-style module is already modeled in the root package, nothing to be done!
                continue;
            }
            else if (await fs.pathExists(path.join(root, `${module.moduleFamily}-${module.moduleBaseName}`.toLocaleLowerCase()))) {
                // V2-style package already has it's own package (legacy behavior), nothing to be done!
                continue;
            }
            else {
                // V2-style package needs to be added to it's "V1" package... Get down to business!
                console.error(`Adding ${namespace} to ${module.packageName}`);
                scopes.push(namespace);
                packageJson['cdk-build'].cloudformation = scopes;
                await fs.writeJson(packageJsonPath, packageJson, { encoding: 'utf-8', spaces: 2 });
                const indexTsPath = path.join(packagePath, 'lib', 'index.ts');
                const indexTs = [
                    (await fs.readFile(indexTsPath, { encoding: 'utf8' })).trimRight(),
                    `// ${namespace} CloudFormation Resources:`,
                    `export * from './${lowcaseModuleName}.generated';`,
                ].join('\n');
                await fs.writeFile(indexTsPath, indexTs, { encoding: 'utf8' });
                continue;
            }
        }
        async function write(relativePath, contents) {
            const fullPath = path.join(packagePath, relativePath);
            const dir = path.dirname(fullPath);
            await fs.mkdirp(dir);
            let data;
            if (typeof contents === 'string') {
                data = contents.trimLeft(); // trim first newline
            }
            else if (Array.isArray(contents)) {
                data = contents.join('\n');
            }
            else if (typeof contents === 'object') {
                data = JSON.stringify(contents, undefined, 2);
            }
            else {
                throw new Error('Invalid type of contents: ' + contents);
            }
            await fs.writeFile(fullPath, data + '\n');
        }
        console.log(`generating module for ${module.packageName}...`);
        const description = `${namespace} Construct Library`;
        await write('package.json', {
            name: module.packageName,
            version,
            description,
            private: true,
            main: 'lib/index.js',
            types: 'lib/index.d.ts',
            jsii: {
                outdir: 'dist',
                projectReferences: true,
                targets: {
                    dotnet: {
                        namespace: module.dotnetPackage,
                        packageId: module.dotnetPackage,
                        signAssembly: true,
                        assemblyOriginatorKeyFile: '../../key.snk',
                        iconUrl: 'https://raw.githubusercontent.com/aws/aws-cdk/main/logo/default-256-dark.png',
                    },
                    java: {
                        package: `${module.javaGroupId}.${module.javaPackage}`,
                        maven: {
                            groupId: module.javaGroupId,
                            artifactId: module.javaArtifactId,
                        },
                    },
                    python: {
                        classifiers: [
                            'Framework :: AWS CDK',
                            'Framework :: AWS CDK :: 2',
                        ],
                        distName: module.pythonDistName,
                        module: module.pythonModuleName,
                    },
                },
                metadata: {
                    jsii: {
                        rosetta: {
                            strict: true,
                        },
                    },
                },
            },
            repository: {
                type: 'git',
                url: 'https://github.com/aws/aws-cdk.git',
                directory: `packages/${module.packageName}`,
            },
            homepage: 'https://github.com/aws/aws-cdk',
            scripts: {
                build: 'cdk-build',
                watch: 'cdk-watch',
                lint: 'cdk-lint',
                test: 'cdk-test',
                integ: 'cdk-integ',
                pkglint: 'pkglint -f',
                package: 'cdk-package',
                awslint: 'cdk-awslint',
                cfn2ts: 'cfn2ts',
                'build+test': 'yarn build && yarn test',
                'build+test+package': 'yarn build+test && yarn package',
                compat: 'cdk-compat',
                gen: 'cfn2ts',
                'rosetta:extract': 'yarn --silent jsii-rosetta extract',
                'build+extract': 'yarn build && yarn rosetta:extract',
                'build+test+extract': 'yarn build+test && yarn rosetta:extract',
            },
            'cdk-build': {
                cloudformation: namespace,
                jest: true,
                env: {
                    AWSLINT_BASE_CONSTRUCT: 'true',
                },
            },
            keywords: [
                'aws',
                'cdk',
                'constructs',
                namespace,
                module.moduleName,
            ],
            author: {
                name: 'Amazon Web Services',
                url: 'https://aws.amazon.com',
                organization: true,
            },
            license: 'Apache-2.0',
            devDependencies: {
                '@aws-cdk/assertions': version,
                '@aws-cdk/cdk-build-tools': version,
                '@aws-cdk/cfn2ts': version,
                '@aws-cdk/pkglint': version,
                '@types/jest': jestTypesVersion,
            },
            dependencies: {
                '@aws-cdk/core': version,
                'constructs': '^10.0.0',
            },
            peerDependencies: {
                '@aws-cdk/core': version,
                'constructs': '^10.0.0',
            },
            engines: {
                node: '>= 14.15.0',
            },
            stability: 'experimental',
            maturity: 'cfn-only',
            awscdkio: {
                announce: false,
            },
            publishConfig: {
                tag: 'latest',
            },
        });
        await write('.gitignore', [
            '*.js',
            '*.js.map',
            '*.d.ts',
            'tsconfig.json',
            'node_modules',
            '*.generated.ts',
            'dist',
            '.jsii',
            '',
            '.LAST_BUILD',
            '.nyc_output',
            'coverage',
            '.nycrc',
            '.LAST_PACKAGE',
            '*.snk',
            'nyc.config.js',
            '!.eslintrc.js',
            '!jest.config.js',
            'junit.xml',
        ]);
        await write('.npmignore', [
            '# Don\'t include original .ts files when doing `npm pack`',
            '*.ts',
            '!*.d.ts',
            'coverage',
            '.nyc_output',
            '*.tgz',
            '',
            'dist',
            '.LAST_PACKAGE',
            '.LAST_BUILD',
            '!*.js',
            '',
            '# Include .jsii',
            '!.jsii',
            '',
            '*.snk',
            '',
            '*.tsbuildinfo',
            '',
            'tsconfig.json',
            '',
            '.eslintrc.js',
            'jest.config.js',
            '',
            '# exclude cdk artifacts',
            '**/cdk.out',
            'junit.xml',
            'test/',
            '!*.lit.ts',
        ]);
        await write('lib/index.ts', [
            `// ${namespace} CloudFormation Resources:`,
            `export * from './${lowcaseModuleName}.generated';`,
        ]);
        await write(`test/${lowcaseModuleName}.test.ts`, [
            "import '@aws-cdk/assertions';",
            "import {} from '../lib';",
            '',
            "test('No tests are specified for this package', () => {",
            '  expect(true).toBe(true);',
            '});',
        ]);
        await pkglint.createLibraryReadme(namespace, path.join(packagePath, 'README.md'));
        await write('.eslintrc.js', [
            "const baseConfig = require('@aws-cdk/cdk-build-tools/config/eslintrc');",
            "baseConfig.parserOptions.project = __dirname + '/tsconfig.json';",
            'module.exports = baseConfig;',
        ]);
        await write('jest.config.js', [
            "const baseConfig = require('@aws-cdk/cdk-build-tools/config/jest.config');",
            'module.exports = baseConfig;',
        ]);
        await write('rosetta/default.ts-fixture', [
            "import { Construct } from 'constructs';",
            "import { Stack } from '@aws-cdk/core';",
            '',
            'class MyStack extends Stack {',
            '  constructor(scope: Construct, id: string) {',
            '    /// here',
            '  }',
            '}',
        ]);
        const templateDir = path.join(__dirname, 'template');
        for (const file of await fs.readdir(templateDir)) {
            await fs.copy(path.join(templateDir, file), path.join(packagePath, file));
        }
        await addDependencyToMegaPackage(path.join('@aws-cdk', 'cloudformation-include'), module.packageName, version, ['dependencies', 'peerDependencies']);
        await addDependencyToMegaPackage('aws-cdk-lib', module.packageName, version, ['devDependencies']);
    }
}
/**
 * A few of our packages (e.g., aws-cdk-lib) require a dependency on every service package.
 * This automates adding the dependency (and peer dependency) to the package.json.
 */
async function addDependencyToMegaPackage(megaPackageName, packageName, version, dependencyTypes) {
    const packageJsonPath = path.join(__dirname, '..', '..', '..', megaPackageName, 'package.json');
    const packageJson = JSON.parse(await fs.readFile(packageJsonPath, 'utf8'));
    dependencyTypes.forEach(dependencyType => {
        const unorderedDeps = {
            ...packageJson[dependencyType],
            [packageName]: version,
        };
        packageJson[dependencyType] = {};
        Object.keys(unorderedDeps).sort().forEach(k => {
            packageJson[dependencyType][k] = unorderedDeps[k];
        });
    });
    await fs.writeFile(packageJsonPath, JSON.stringify(packageJson, null, 2) + '\n');
}
main().catch(e => {
    console.error(e);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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