"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.normalize = exports.massageSpec = void 0;
const scrutiny_1 = require("./scrutiny");
const lib_1 = require("../lib");
function massageSpec(spec) {
    scrutiny_1.detectScrutinyTypes(spec);
    replaceIncompleteTypes(spec);
    dropTypelessAttributes(spec);
}
exports.massageSpec = massageSpec;
/**
 * Fix incomplete type definitions in PropertyTypes
 *
 * Some user-defined types are defined to not have any properties, and not
 * be a collection of other types either. They have no definition at all.
 *
 * Add a property object type with empty properties.
 */
function replaceIncompleteTypes(spec) {
    for (const [name, definition] of Object.entries(spec.PropertyTypes)) {
        if (!lib_1.schema.isRecordType(definition)
            && !lib_1.schema.isCollectionProperty(definition)
            && !lib_1.schema.isScalarProperty(definition)
            && !lib_1.schema.isPrimitiveProperty(definition)) {
            // eslint-disable-next-line no-console
            console.log(`[${name}] Incomplete type, adding empty "Properties" field`);
            definition.Properties = {};
        }
    }
}
/**
 * Drop Attributes specified with the different ResourceTypes that have
 * no type specified.
 */
function dropTypelessAttributes(spec) {
    const resourceTypes = spec.ResourceTypes;
    Object.values(resourceTypes).forEach((resourceType) => {
        const attributes = resourceType.Attributes ?? {};
        Object.keys(attributes).forEach((attrKey) => {
            const attrVal = attributes[attrKey];
            if (Object.keys(attrVal).length === 0) {
                delete attributes[attrKey];
            }
        });
    });
}
/**
 * Modifies the provided specification so that ``ResourceTypes`` and ``PropertyTypes`` are listed in alphabetical order.
 *
 * @param spec an AWS CloudFormation Resource Specification document.
 *
 * @returns ``spec``, after having sorted the ``ResourceTypes`` and ``PropertyTypes`` sections alphabetically.
 */
function normalize(spec) {
    spec.ResourceTypes = normalizeSection(spec.ResourceTypes);
    if (spec.PropertyTypes) {
        spec.PropertyTypes = normalizeSection(spec.PropertyTypes);
    }
    return spec;
    function normalizeSection(section) {
        const result = {};
        for (const key of Object.keys(section).sort()) {
            result[key] = section[key];
        }
        return result;
    }
}
exports.normalize = normalize;
//# sourceMappingURL=data:application/json;base64,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