"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.writeSorted = exports.applyAndWrite = exports.applyPatchSet = exports.evaluatePatchSet = exports.loadPatchSet = void 0;
/**
 * Apply a JSON patch set into the given target file
 *
 * The sources can be taken from one or more directories.
 */
const path = require("path");
const fastJsonPatch = require("fast-json-patch");
const fs = require("fs-extra");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const sortJson = require('sort-json');
async function loadPatchSet(sourceDirectory, relativeTo = process.cwd()) {
    const ret = {};
    const files = await fs.readdir(sourceDirectory);
    for (const file of files) {
        const fullFile = path.join(sourceDirectory, file);
        const relName = path.relative(relativeTo, fullFile);
        if (file.startsWith('.')) {
            // Nothing, ignore
        }
        else if ((await fs.stat(fullFile)).isDirectory()) {
            ret[relName] = {
                type: 'set',
                sources: await loadPatchSet(fullFile, sourceDirectory),
            };
        }
        else if (file.endsWith('.json')) {
            ret[relName] = {
                type: file.indexOf('patch') === -1 ? 'fragment' : 'patch',
                data: await fs.readJson(fullFile),
            };
        }
    }
    return ret;
}
exports.loadPatchSet = loadPatchSet;
function evaluatePatchSet(sources, options = {}) {
    const targetObject = {};
    for (const key of Object.keys(sources).sort()) {
        const value = sources[key];
        switch (value.type) {
            case 'fragment':
                log(key);
                merge(targetObject, value.data, []);
                break;
            case 'patch':
                patch(targetObject, value.data, (m) => log(`${key}: ${m}`));
                break;
            case 'set':
                const evaluated = evaluatePatchSet(value.sources, options);
                log(key);
                merge(targetObject, evaluated, []);
                break;
        }
    }
    return targetObject;
    function log(x) {
        if (!options.quiet) {
            // eslint-disable-next-line no-console
            console.log(x);
        }
    }
}
exports.evaluatePatchSet = evaluatePatchSet;
/**
 * Load a patch set from a directory
 */
async function applyPatchSet(sourceDirectory, options = {}) {
    const patches = await loadPatchSet(sourceDirectory);
    return evaluatePatchSet(patches, options);
}
exports.applyPatchSet = applyPatchSet;
/**
 * Load a patch set and write it out to a file
 */
async function applyAndWrite(targetFile, sourceDirectory, options = {}) {
    const model = await applyPatchSet(sourceDirectory, options);
    await writeSorted(targetFile, model);
}
exports.applyAndWrite = applyAndWrite;
async function writeSorted(targetFile, data) {
    await fs.mkdirp(path.dirname(targetFile));
    await fs.writeJson(targetFile, sortJson(data), { spaces: 2 });
}
exports.writeSorted = writeSorted;
function printSorted(data) {
    process.stdout.write(JSON.stringify(sortJson(data), undefined, 2));
}
function merge(target, fragment, jsonPath) {
    if (!fragment) {
        return;
    }
    if (!target || typeof target !== 'object' || Array.isArray(target)) {
        throw new Error(`Expected object, found: '${target}' at '$.${jsonPath.join('.')}'`);
    }
    for (const key of Object.keys(fragment)) {
        if (key.startsWith('$')) {
            continue;
        }
        if (key in target) {
            const specVal = target[key];
            const fragVal = fragment[key];
            if (typeof specVal !== typeof fragVal) {
                // eslint-disable-next-line max-len
                throw new Error(`Attempted to merge ${JSON.stringify(fragVal)} into incompatible ${JSON.stringify(specVal)} at path ${jsonPath.join('/')}/${key}`);
            }
            if (specVal == fragVal) {
                continue;
            }
            if (typeof specVal !== 'object') {
                // eslint-disable-next-line max-len
                throw new Error(`Conflict when attempting to merge ${JSON.stringify(fragVal)} into ${JSON.stringify(specVal)} at path ${jsonPath.join('/')}/${key}`);
            }
            merge(specVal, fragVal, [...jsonPath, key]);
        }
        else {
            target[key] = fragment[key];
        }
    }
}
function patch(target, fragment, log) {
    if (!fragment) {
        return;
    }
    const patches = findPatches(target, fragment);
    for (const p of patches) {
        log(p.description ?? '');
        try {
            fastJsonPatch.applyPatch(target, p.operations);
        }
        catch (e) {
            throw new Error(`error applying patch: ${JSON.stringify(p, undefined, 2)}: ${e.message}`);
        }
    }
}
/**
 * Find the sets of patches to apply in a document
 *
 * Adjusts paths to be root-relative, which makes it possible to have paths
 * point outside the patch scope.
 */
function findPatches(data, patchSource) {
    const ret = [];
    recurse(data, patchSource, []);
    return ret;
    function recurse(actualData, fragment, jsonPath) {
        if (!fragment) {
            return;
        }
        if ('patch' in fragment) {
            const p = fragment.patch;
            if (!p.operations) {
                throw new Error(`Patch needs 'operations' key, got: ${JSON.stringify(p)}`);
            }
            ret.push({
                description: p.description,
                operations: p.operations.map((op) => adjustPaths(op, jsonPath)),
            });
        }
        else if ('patch:each' in fragment) {
            const p = fragment['patch:each'];
            if (typeof actualData !== 'object') {
                throw new Error(`Patch ${p.description}: expecting object in data, found '${actualData}'`);
            }
            if (!p.operations) {
                throw new Error(`Patch needs 'operations' key, got: ${JSON.stringify(p)}`);
            }
            for (const key in actualData) {
                ret.push({
                    description: `${key}: ${p.description}`,
                    operations: p.operations.map((op) => adjustPaths(op, [...jsonPath, key])),
                });
            }
        }
        else {
            for (const key of Object.keys(fragment)) {
                if (!(key in actualData)) {
                    actualData[key] = {};
                }
                recurse(actualData[key], fragment[key], [...jsonPath, key]);
            }
        }
    }
    function adjustPaths(op, jsonPath) {
        return {
            ...op,
            ...op.path ? { path: adjustPath(op.path, jsonPath) } : undefined,
            ...op.from ? { from: adjustPath(op.from, jsonPath) } : undefined,
        };
    }
    /**
     * Adjust path
     *
     * '$/' means from the root, otherwise interpret as relative path.
     */
    function adjustPath(originalPath, jsonPath) {
        if (typeof originalPath !== 'string') {
            throw new Error(`adjustPath: expected string, got ${JSON.stringify(originalPath)}`);
        }
        if (originalPath.startsWith('$/')) {
            return originalPath.slice(1);
        }
        return jsonPath.map(p => `/${p}`).join('') + originalPath;
    }
}
/**
 * Run this file as a CLI tool, to apply a patch set from the command line
 */
async function main(args) {
    const quiet = eatArg('-q', args) || eatArg('--quiet', args);
    if (args.length < 1) {
        throw new Error('Usage: patch-set <DIR> [<FILE>]');
    }
    const [dir, targetFile] = args;
    const model = await applyPatchSet(dir, { quiet });
    if (targetFile) {
        await writeSorted(targetFile, model);
    }
    else {
        printSorted(model);
    }
}
function eatArg(arg, args) {
    for (let i = 0; i < args.length; i++) {
        if (args[i] === arg) {
            args.splice(i, 1);
            return true;
        }
    }
    return false;
}
if (require.main === module) {
    main(process.argv.slice(2)).catch(e => {
        process.exitCode = 1;
        // eslint-disable-next-line no-console
        console.error(e.message);
    });
}
//# sourceMappingURL=data:application/json;base64,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