"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const util = require("util");
const fs = require("fs-extra");
/* eslint-disable @typescript-eslint/no-require-imports */
const jsonDiff = require('json-diff').diff;
/* eslint-enable */
function line(fmt = '', ...param) {
    process.stdout.write(util.format(fmt, ...param));
    process.stdout.write('\n');
}
async function main() {
    const title = process.argv[2];
    const oldSpecFile = process.argv[3];
    const newSpecFile = process.argv[4];
    const newSpec = await fs.readJSON(newSpecFile);
    const oldSpec = await fs.readJSON(oldSpecFile);
    // Diff operates on PropertyTypes & ResourceTypes
    // Ensure they always exist in the old spec
    if (!oldSpec.PropertyTypes) {
        oldSpec.PropertyTypes = {};
    }
    if (!oldSpec.ResourceTypes) {
        oldSpec.ResourceTypes = {};
    }
    const out = jsonDiff(oldSpec, newSpec);
    // Here's the magic output format of this thing
    // If a key ends in __added, it got added, and the value
    //   is the new value.
    // If a key ends in __deleted, it got deleted, and the value
    //   is the old value.
    // If a value got changed, the value object will look like:
    //   { __old: ..., __new: ... }
    if (!out) {
        return; // no diff
    }
    const resourceTypeAdditions = new Set();
    const resourceTypeDeletions = new Set();
    const attributeChanges = new Array();
    const propertyChanges = new Array();
    const propertyTypeChanges = new Array();
    for (const key of Object.keys(out.ResourceTypes || {})) {
        classifyResourceTypeUpdate(key, out.ResourceTypes[key]);
    }
    for (const key of Object.keys(out.PropertyTypes || {})) {
        classifyPropertyTypeUpdate(key, out.PropertyTypes[key]);
    }
    line(`# ${title} v${newSpec.ResourceSpecificationVersion}`);
    line();
    line('## New Resource Types');
    line();
    resourceTypeAdditions.forEach(type => line(`* ${type}`));
    line();
    if (resourceTypeDeletions.size > 0) {
        line('## Removed Resource Types');
        line();
        resourceTypeDeletions.forEach(type => line(`* ${type}`));
        line();
    }
    line('## Attribute Changes');
    line();
    attributeChanges.forEach(x => line(x));
    line();
    line('## Property Changes');
    line();
    propertyChanges.forEach(x => line(x));
    line();
    line('## Property Type Changes');
    line();
    propertyTypeChanges.forEach(x => line(x));
    function classifyResourceTypeUpdate(resourceType, update) {
        const added = isAdded(resourceType);
        if (added) {
            resourceTypeAdditions.add(added);
            return;
        }
        const deleted = isDeleted(resourceType);
        if (deleted) {
            resourceTypeDeletions.add(deleted);
            return;
        }
        pushDownCompleteChanges(update);
        for (const key of Object.keys(update)) {
            switch (key) {
                case 'Properties':
                    for (const prop of Object.keys(update.Properties)) {
                        describeChanges(resourceType, prop, update.Properties[prop]).forEach(change => {
                            propertyChanges.push(change);
                        });
                    }
                    break;
                case 'Attributes':
                    for (const attr of Object.keys(update.Attributes)) {
                        describeChanges(resourceType, attr, update.Attributes[attr]).forEach(change => {
                            attributeChanges.push(change);
                        });
                    }
                    break;
                case 'Documentation':
                    describeChanges(resourceType, key, update.Documentation).forEach(change => {
                        attributeChanges.push(change);
                    });
                    break;
                default:
                    throw new Error(`Unexpected update to ${resourceType}: ${key}`);
            }
        }
    }
    function classifyPropertyTypeUpdate(propertyType, update) {
        const added = isAdded(propertyType);
        if (added) {
            const resourceType = added.split('.')[0];
            if (resourceTypeAdditions.has(resourceType)) {
                return; // skipping property for added resource types
            }
            propertyTypeChanges.push(`* ${added} (__added__)`);
            return;
        }
        const deleted = isDeleted(propertyType);
        if (deleted) {
            const resourceType = deleted.split('.')[0];
            if (resourceTypeDeletions.has(resourceType)) {
                return; // skipping property for added resource types
            }
            propertyTypeChanges.push(`* ${deleted} (__removed__)`);
            return;
        }
        if (Object.keys(update).length !== 1 && Object.keys(update)[0] === 'Properties') {
            throw new Error('Unexpected update to a resource type. Expecting only "Properties" to change: ' + propertyType);
        }
        for (const prop of Object.keys(update.Properties ?? {})) {
            describeChanges(propertyType, prop, update.Properties[prop]).forEach(change => {
                propertyTypeChanges.push(change);
            });
        }
    }
    /**
     * Push down mass changes to attributes or properties to the individual properties.
     *
     * An example will explain this best. JSON-diff will make the smallest diff, so if there
     * are new properties it will report:
     *
     * "Properties__added": {
     *    "Property1": { ... },
     *    "Property2": { ... },
     * }
     *
     * But we want to see this as:
     *
     * "Properties": {
     *    "Property1__added": { ... },
     *    "Property2__added": { ... },
     * }
     *
     * Same (but in reverse) for deletions.
     */
    function pushDownCompleteChanges(update) {
        for (const [category, entries] of Object.entries(update)) {
            const addedKey = isAdded(category);
            if (addedKey) {
                delete update[category];
                update[addedKey] = suffixKeys('__added', entries);
            }
            const deletedKey = isDeleted(category);
            if (deletedKey) {
                delete update[category];
                update[deletedKey] = suffixKeys('__deleted', entries);
            }
        }
    }
    function isDeleted(key) {
        return isSuffix(key, '__deleted');
    }
    function isAdded(key) {
        return isSuffix(key, '__added');
    }
    function isSuffix(key, suffix) {
        const index = key.indexOf(suffix);
        return index === -1 ? undefined : key.slice(0, index);
    }
    function suffixKeys(suffix, xs) {
        const ret = {};
        for (const [key, value] of Object.entries(xs)) {
            ret[key + suffix] = value;
        }
        return ret;
    }
    function describeChanges(namespace, prefix, update) {
        const changes = new Array();
        const added = isAdded(prefix);
        if (added) {
            changes.push(`* ${namespace} ${added} (__added__)`);
            return changes;
        }
        const deleted = isDeleted(prefix);
        if (deleted) {
            changes.push(`* ${namespace} ${deleted} (__deleted__)`);
            return changes;
        }
        if (typeof (update) !== 'object') {
            throw new Error(`Unexpected change for ${namespace}.${prefix} ${JSON.stringify(update)}`);
        }
        if ('__old' in update && '__new' in update) {
            changes.push(`* ${namespace} ${prefix} (__changed__)`);
            changes.push(`  * Old: ${update.__old}`);
            changes.push(`  * New: ${update.__new}`);
            return changes;
        }
        if (Array.isArray(update)) {
            changes.push(`* ${namespace} ${prefix} (__changed__)`);
            for (const entry of update) {
                if (entry.length === 1 && entry[0] === ' ') {
                    // This means that this element of the array is unchanged
                    continue;
                }
                if (entry.length !== 2) {
                    throw new Error(`Unexpected array diff entry: ${JSON.stringify(entry)}`);
                }
                switch (entry[0]) {
                    case '+':
                        changes.push(`  * Added ${entry[1]}`);
                        break;
                    case '-':
                        throw new Error(`Something awkward happened: ${entry[1]} was deleted from ${namespace} ${prefix}!`);
                    case ' ':
                        // This entry is "context"
                        break;
                    default:
                        throw new Error(`Unexpected array diff entry: ${JSON.stringify(entry)}`);
                }
            }
        }
        else {
            for (const key of Object.keys(update)) {
                for (const change of describeChanges(namespace, `${prefix}.${key}`, update[key])) {
                    changes.push(change);
                }
            }
        }
        return changes;
    }
}
main().catch(e => {
    process.stderr.write(e.stack);
    process.stderr.write('\n');
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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