"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
/**
 * Split the given CloudFormation resource specification up into a patch set, by service
 *
 * After splitting, only write out those fragments that are valid.(*)
 *
 * The reason for this is that some service updates sometimes contain mistakes,
 * and we want to be able to take partial upgrades. By splitting the spec, we can
 * take updates to some services while leaving updates to other services behind.
 *
 * (*) If `NO_VALIDATE` is set, all pieces are written out. In normal operation this
 *     should not be used.
 */
const path = require("path");
const fs = require("fs-extra");
const patch_set_1 = require("./patch-set");
const validate_cfn_1 = require("./validate-cfn");
async function main(args) {
    if (args.length < 3) {
        throw new Error('Usage: split-spec-by-service <SPECFILE> <DIRECTORY> [<SERVICES>]');
    }
    const [specFile, outDir, services] = args;
    const allowedServices = services.trim().split(' ').filter(Boolean);
    log(`Loading specification: ${specFile}`);
    const spec = await fs.readJson(specFile);
    // Split
    log('Splitting');
    const byService = {};
    for (const [propTypeName, propType] of Object.entries(spec.PropertyTypes)) {
        const svcName = serviceName(propTypeName);
        serviceSpec(svcName).PropertyTypes[propTypeName] = propType;
    }
    for (const [resTypeName, resType] of Object.entries(spec.ResourceTypes)) {
        const svcName = serviceName(resTypeName);
        serviceSpec(svcName).ResourceTypes[resTypeName] = resType;
    }
    // Write out
    if (allowedServices.length > 0) {
        log(`Writing: ${allowedServices.join(' ')}`);
    }
    else {
        log('Writing all services');
    }
    for (const [svcName, svcSpec] of Object.entries(byService)) {
        // Skip services that are not explicitly allowed
        if (allowedServices.length > 0 && !allowedServices.includes(svcName)) {
            continue;
        }
        const successTarget = path.join(outDir, `000_${svcName}.json`);
        const rejectedTarget = path.join(outDir, `.000_${svcName}.rejected.json`);
        const errors = !process.env.NO_VALIDATE ? validate_cfn_1.CfnSpecValidator.validate(svcSpec) : [];
        if (errors.length === 0) {
            // Change 'ResourceSpecificationVersion' to '$version', otherwise they will all conflict
            const toWrite = {
                PropertyTypes: svcSpec.PropertyTypes,
                ResourceTypes: svcSpec.ResourceTypes,
                $version: svcSpec.ResourceSpecificationVersion,
            };
            await patch_set_1.writeSorted(successTarget, toWrite);
            await ensureGone(rejectedTarget);
        }
        else {
            console.warn('='.repeat(70));
            console.warn(' '.repeat(Math.floor(35 - svcName.length / 2)) + svcName);
            console.warn('='.repeat(70));
            for (const error of errors) {
                console.warn(validate_cfn_1.formatErrorInContext(error));
            }
            await patch_set_1.writeSorted(rejectedTarget, svcSpec);
            // Make sure that the success file exists. If not, the initial import of a
            // new service failed.
            if (!await fs.pathExists(successTarget)) {
                await patch_set_1.writeSorted(successTarget, {
                    PropertyTypes: {},
                    ResourceTypes: {},
                    $version: '0.0.0',
                });
            }
        }
    }
    await fs.writeJson(path.join(outDir, '001_Version.json'), {
        ResourceSpecificationVersion: spec.ResourceSpecificationVersion,
    }, { spaces: 2 });
    function serviceSpec(svcName) {
        if (!(svcName in byService)) {
            byService[svcName] = {
                PropertyTypes: {},
                ResourceTypes: {},
                ResourceSpecificationVersion: spec.ResourceSpecificationVersion,
            };
        }
        return byService[svcName];
    }
}
async function ensureGone(fileName) {
    try {
        await fs.unlink(fileName);
    }
    catch (e) {
        if (e.code === 'ENOENT') {
            return;
        }
        throw e;
    }
}
function serviceName(x) {
    return x.split('::').slice(0, 2).join('_');
}
function log(x) {
    // eslint-disable-next-line no-console
    console.log(x);
}
main(process.argv.slice(2)).catch(e => {
    // eslint-disable-next-line no-console
    console.error(e);
    process.exitCode = 1;
});
//# sourceMappingURL=data:application/json;base64,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