"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.cannedMetricsForService = void 0;
const canned_metrics_schema_1 = require("./canned-metrics/canned-metrics-schema");
/**
 * Return the list of canned metrics for the given service
 */
function cannedMetricsForService(cloudFormationNamespace) {
    // One metricTemplate has a single set of dimensions, but the same metric NAME
    // may occur in multiple metricTemplates (if it has multiple sets of dimensions)
    const metricTemplates = cannedMetricsIndex()[cloudFormationNamespace] ?? [];
    // First construct almost what we need, but with a single dimension per metric
    const metricsWithDuplicates = flatMap(metricTemplates, metricSet => {
        const dimensions = metricSet.dimensions.map(d => d.dimensionName);
        return metricSet.metrics.map(metric => ({
            namespace: metricSet.namespace,
            dimensions,
            metricName: metric.name,
            defaultStat: metric.defaultStat,
        }));
    });
    // Then combine the dimensions for the same metrics into a single list
    return groupBy(metricsWithDuplicates, m => `${m.namespace}/${m.metricName}`).map(metrics => ({
        namespace: metrics[0].namespace,
        metricName: metrics[0].metricName,
        defaultStat: metrics[0].defaultStat,
        dimensions: Array.from(dedupeStringLists(metrics.map(m => m.dimensions))),
    }));
}
exports.cannedMetricsForService = cannedMetricsForService;
let cannedMetricsCache;
/**
 * Load the canned metrics file and process it into an index, grouped by service namespace
 */
function cannedMetricsIndex() {
    if (cannedMetricsCache === undefined) {
        cannedMetricsCache = {};
        for (const group of canned_metrics_schema_1.loadCannedMetricsFile()) {
            for (const metricTemplate of group.metricTemplates) {
                const [aws, service] = metricTemplate.resourceType.split('::');
                const serviceKey = [aws, service].join('::');
                (cannedMetricsCache[serviceKey] ?? (cannedMetricsCache[serviceKey] = [])).push(metricTemplate);
            }
        }
    }
    return cannedMetricsCache;
}
function flatMap(xs, fn) {
    return Array.prototype.concat.apply([], xs.map(fn));
}
function groupBy(xs, keyFn) {
    const obj = {};
    for (const x of xs) {
        const key = keyFn(x);
        if (key in obj) {
            obj[key].push(x);
        }
        else {
            obj[key] = [x];
        }
    }
    return Object.values(obj);
}
function* dedupeStringLists(xs) {
    const seen = new Set();
    for (const x of xs) {
        x.sort();
        const key = `${x.join(',')}`;
        if (!seen.has(key)) {
            yield x;
        }
        seen.add(key);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2FubmVkLW1ldHJpY3MuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjYW5uZWQtbWV0cmljcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxrRkFBK0Y7QUE0Qy9GOztHQUVHO0FBQ0gsU0FBZ0IsdUJBQXVCLENBQUMsdUJBQStCO0lBQ3JFLDhFQUE4RTtJQUM5RSxnRkFBZ0Y7SUFDaEYsTUFBTSxlQUFlLEdBQUcsa0JBQWtCLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUU1RSw4RUFBOEU7SUFDOUUsTUFBTSxxQkFBcUIsR0FBRyxPQUFPLENBQUMsZUFBZSxFQUFFLFNBQVMsQ0FBQyxFQUFFO1FBQ2pFLE1BQU0sVUFBVSxHQUFHLFNBQVMsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ2xFLE9BQU8sU0FBUyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQ3RDLFNBQVMsRUFBRSxTQUFTLENBQUMsU0FBUztZQUM5QixVQUFVO1lBQ1YsVUFBVSxFQUFFLE1BQU0sQ0FBQyxJQUFJO1lBQ3ZCLFdBQVcsRUFBRSxNQUFNLENBQUMsV0FBVztTQUNoQyxDQUFDLENBQUMsQ0FBQztJQUNOLENBQUMsQ0FBQyxDQUFDO0lBRUgsc0VBQXNFO0lBQ3RFLE9BQU8sT0FBTyxDQUFDLHFCQUFxQixFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUMsU0FBUyxJQUFJLENBQUMsQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDM0YsU0FBUyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1FBQy9CLFVBQVUsRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsVUFBVTtRQUNqQyxXQUFXLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVc7UUFDbkMsVUFBVSxFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFRO0tBQ2pGLENBQUMsQ0FBQyxDQUFDO0FBQ04sQ0FBQztBQXZCRCwwREF1QkM7QUFJRCxJQUFJLGtCQUFrRCxDQUFDO0FBRXZEOztHQUVHO0FBQ0gsU0FBUyxrQkFBa0I7SUFDekIsSUFBSSxrQkFBa0IsS0FBSyxTQUFTLEVBQUU7UUFDcEMsa0JBQWtCLEdBQUcsRUFBRSxDQUFDO1FBQ3hCLEtBQUssTUFBTSxLQUFLLElBQUksNkNBQXFCLEVBQUUsRUFBRTtZQUMzQyxLQUFLLE1BQU0sY0FBYyxJQUFJLEtBQUssQ0FBQyxlQUFlLEVBQUU7Z0JBQ2xELE1BQU0sQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLEdBQUcsY0FBYyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQy9ELE1BQU0sVUFBVSxHQUFHLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDN0MsQ0FBQyxrQkFBa0IsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLFVBQVUsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDO2FBQ2hHO1NBQ0Y7S0FDRjtJQUNELE9BQU8sa0JBQWtCLENBQUM7QUFDNUIsQ0FBQztBQUVELFNBQVMsT0FBTyxDQUFPLEVBQU8sRUFBRSxFQUFpQjtJQUMvQyxPQUFPLEtBQUssQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO0FBQ3RELENBQUM7QUFFRCxTQUFTLE9BQU8sQ0FBSSxFQUFPLEVBQUUsS0FBdUI7SUFDbEQsTUFBTSxHQUFHLEdBQXFDLEVBQUUsQ0FBQztJQUNqRCxLQUFLLE1BQU0sQ0FBQyxJQUFJLEVBQUUsRUFBRTtRQUNsQixNQUFNLEdBQUcsR0FBRyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDckIsSUFBSSxHQUFHLElBQUksR0FBRyxFQUFFO1lBQ2QsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUNsQjthQUFNO1lBQ0wsR0FBRyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDaEI7S0FDRjtJQUNELE9BQU8sTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQztBQUM1QixDQUFDO0FBRUQsUUFBUSxDQUFDLENBQUMsaUJBQWlCLENBQUMsRUFBYztJQUN4QyxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsRUFBVSxDQUFDO0lBQy9CLEtBQUssTUFBTSxDQUFDLElBQUksRUFBRSxFQUFFO1FBQ2xCLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNULE1BQU0sR0FBRyxHQUFHLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDO1FBQzdCLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFO1lBQ2xCLE1BQU0sQ0FBQyxDQUFDO1NBQ1Q7UUFDRCxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0tBQ2Y7QUFDSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgbG9hZENhbm5lZE1ldHJpY3NGaWxlLCBNZXRyaWNUZW1wbGF0ZSB9IGZyb20gJy4vY2FubmVkLW1ldHJpY3MvY2FubmVkLW1ldHJpY3Mtc2NoZW1hJztcblxuZXhwb3J0IHR5cGUgTm9uRW1wdHlBcnJheTxUPiA9IFtULCAuLi5UW11dO1xuXG4vKipcbiAqIEEgc2luZ2xlIGNhbm5lZCBzZXJ2aWNlIG1ldHJpY1xuICpcbiAqIFRoZXNlIGFyZSBraW5kbHkgcHJvdmlkZWQgdG8gdXMgYnkgdGhlIGdvb2QgcGVvcGxlIG9mIENsb3VkV2F0Y2ggRXhwbG9yZXIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ2FubmVkTWV0cmljIHtcbiAgLyoqXG4gICAqIE1ldHJpYyBuYW1lc3BhY2VcbiAgICovXG4gIHJlYWRvbmx5IG5hbWVzcGFjZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBNZXRyaWMgbmFtZVxuICAgKi9cbiAgcmVhZG9ubHkgbWV0cmljTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIGFsbCBwb3NzaWJsZSBkaW1lbnNpb24gcGVybXV0YXRpb25zIGZvciB0aGlzIG1ldHJpY1xuICAgKlxuICAgKiBNb3N0IG1ldHJpY3Mgd2lsbCBoYXZlIGEgc2luZ2xlIGxpc3Qgb2Ygc3RyaW5ncyBhcyB0aGVpciBvbmUgc2V0IG9mXG4gICAqIGFsbG93ZWQgZGltZW5zaW9ucywgYnV0IHNvbWUgbWV0cmljcyBhcmUgZW1pdHRlZCB1bmRlciBtdWx0aXBsZVxuICAgKiBjb21iaW5hdGlvbnMgb2YgZGltZW5zaW9ucy5cbiAgICovXG4gIHJlYWRvbmx5IGRpbWVuc2lvbnM6IE5vbkVtcHR5QXJyYXk8c3RyaW5nW10+O1xuXG4gIC8qKlxuICAgKiBTdWdnZXN0ZWQgZGVmYXVsdCBhZ2dyZWdyYXRpb24gc3RhdGlzdGljXG4gICAqXG4gICAqIE5vdCBhbHdheXMgdGhlIG1vc3QgYXBwcm9wcmlhdGUgb25lIHRvIHVzZSEgVGhlc2UgZGVmYXVsdHMgaGF2ZVxuICAgKiBiZWVuIGNsYXNzaWZpZWQgYnkgcGVvcGxlIGFuZCB0aGV5IGdlbmVyYWxseSBqdXN0IHBpY2sgXCJBdmVyYWdlXCJcbiAgICogYXMgdGhlIGRlZmF1bHQsIGV2ZW4gaWYgaXQgZG9lc24ndCBtYWtlIHNlbnNlLlxuICAgKlxuICAgKiBGb3IgZXhhbXBsZTogZm9yIGV2ZW50LWJhc2VkIG1ldHJpY3MgdGhhdCBvbmx5IGV2ZXIgZW1pdCBgMWBcbiAgICogKGFuZCBuZXZlciBgMGApIHRoZSBiZXR0ZXIgc3RhdGlzdGljIHdvdWxkIGJlIGBTdW1gLlxuICAgKlxuICAgKiBVc2UgeW91ciBqdWRnZW1lbnQgYmFzZWQgb24gdGhlIHR5cGUgb2YgbWV0cmljIHRoaXMgaXMuXG4gICAqL1xuICByZWFkb25seSBkZWZhdWx0U3RhdDogc3RyaW5nO1xufVxuXG4vKipcbiAqIFJldHVybiB0aGUgbGlzdCBvZiBjYW5uZWQgbWV0cmljcyBmb3IgdGhlIGdpdmVuIHNlcnZpY2VcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGNhbm5lZE1ldHJpY3NGb3JTZXJ2aWNlKGNsb3VkRm9ybWF0aW9uTmFtZXNwYWNlOiBzdHJpbmcpOiBDYW5uZWRNZXRyaWNbXSB7XG4gIC8vIE9uZSBtZXRyaWNUZW1wbGF0ZSBoYXMgYSBzaW5nbGUgc2V0IG9mIGRpbWVuc2lvbnMsIGJ1dCB0aGUgc2FtZSBtZXRyaWMgTkFNRVxuICAvLyBtYXkgb2NjdXIgaW4gbXVsdGlwbGUgbWV0cmljVGVtcGxhdGVzIChpZiBpdCBoYXMgbXVsdGlwbGUgc2V0cyBvZiBkaW1lbnNpb25zKVxuICBjb25zdCBtZXRyaWNUZW1wbGF0ZXMgPSBjYW5uZWRNZXRyaWNzSW5kZXgoKVtjbG91ZEZvcm1hdGlvbk5hbWVzcGFjZV0gPz8gW107XG5cbiAgLy8gRmlyc3QgY29uc3RydWN0IGFsbW9zdCB3aGF0IHdlIG5lZWQsIGJ1dCB3aXRoIGEgc2luZ2xlIGRpbWVuc2lvbiBwZXIgbWV0cmljXG4gIGNvbnN0IG1ldHJpY3NXaXRoRHVwbGljYXRlcyA9IGZsYXRNYXAobWV0cmljVGVtcGxhdGVzLCBtZXRyaWNTZXQgPT4ge1xuICAgIGNvbnN0IGRpbWVuc2lvbnMgPSBtZXRyaWNTZXQuZGltZW5zaW9ucy5tYXAoZCA9PiBkLmRpbWVuc2lvbk5hbWUpO1xuICAgIHJldHVybiBtZXRyaWNTZXQubWV0cmljcy5tYXAobWV0cmljID0+ICh7XG4gICAgICBuYW1lc3BhY2U6IG1ldHJpY1NldC5uYW1lc3BhY2UsXG4gICAgICBkaW1lbnNpb25zLFxuICAgICAgbWV0cmljTmFtZTogbWV0cmljLm5hbWUsXG4gICAgICBkZWZhdWx0U3RhdDogbWV0cmljLmRlZmF1bHRTdGF0LFxuICAgIH0pKTtcbiAgfSk7XG5cbiAgLy8gVGhlbiBjb21iaW5lIHRoZSBkaW1lbnNpb25zIGZvciB0aGUgc2FtZSBtZXRyaWNzIGludG8gYSBzaW5nbGUgbGlzdFxuICByZXR1cm4gZ3JvdXBCeShtZXRyaWNzV2l0aER1cGxpY2F0ZXMsIG0gPT4gYCR7bS5uYW1lc3BhY2V9LyR7bS5tZXRyaWNOYW1lfWApLm1hcChtZXRyaWNzID0+ICh7XG4gICAgbmFtZXNwYWNlOiBtZXRyaWNzWzBdLm5hbWVzcGFjZSxcbiAgICBtZXRyaWNOYW1lOiBtZXRyaWNzWzBdLm1ldHJpY05hbWUsXG4gICAgZGVmYXVsdFN0YXQ6IG1ldHJpY3NbMF0uZGVmYXVsdFN0YXQsXG4gICAgZGltZW5zaW9uczogQXJyYXkuZnJvbShkZWR1cGVTdHJpbmdMaXN0cyhtZXRyaWNzLm1hcChtID0+IG0uZGltZW5zaW9ucykpKSBhcyBhbnksXG4gIH0pKTtcbn1cblxudHlwZSBDYW5uZWRNZXRyaWNzSW5kZXggPSBSZWNvcmQ8c3RyaW5nLCBNZXRyaWNUZW1wbGF0ZVtdPjtcblxubGV0IGNhbm5lZE1ldHJpY3NDYWNoZTogQ2FubmVkTWV0cmljc0luZGV4IHwgdW5kZWZpbmVkO1xuXG4vKipcbiAqIExvYWQgdGhlIGNhbm5lZCBtZXRyaWNzIGZpbGUgYW5kIHByb2Nlc3MgaXQgaW50byBhbiBpbmRleCwgZ3JvdXBlZCBieSBzZXJ2aWNlIG5hbWVzcGFjZVxuICovXG5mdW5jdGlvbiBjYW5uZWRNZXRyaWNzSW5kZXgoKSB7XG4gIGlmIChjYW5uZWRNZXRyaWNzQ2FjaGUgPT09IHVuZGVmaW5lZCkge1xuICAgIGNhbm5lZE1ldHJpY3NDYWNoZSA9IHt9O1xuICAgIGZvciAoY29uc3QgZ3JvdXAgb2YgbG9hZENhbm5lZE1ldHJpY3NGaWxlKCkpIHtcbiAgICAgIGZvciAoY29uc3QgbWV0cmljVGVtcGxhdGUgb2YgZ3JvdXAubWV0cmljVGVtcGxhdGVzKSB7XG4gICAgICAgIGNvbnN0IFthd3MsIHNlcnZpY2VdID0gbWV0cmljVGVtcGxhdGUucmVzb3VyY2VUeXBlLnNwbGl0KCc6OicpO1xuICAgICAgICBjb25zdCBzZXJ2aWNlS2V5ID0gW2F3cywgc2VydmljZV0uam9pbignOjonKTtcbiAgICAgICAgKGNhbm5lZE1ldHJpY3NDYWNoZVtzZXJ2aWNlS2V5XSA/PyAoY2FubmVkTWV0cmljc0NhY2hlW3NlcnZpY2VLZXldID0gW10pKS5wdXNoKG1ldHJpY1RlbXBsYXRlKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbiAgcmV0dXJuIGNhbm5lZE1ldHJpY3NDYWNoZTtcbn1cblxuZnVuY3Rpb24gZmxhdE1hcDxBLCBCPih4czogQVtdLCBmbjogKHg6IEEpID0+IEJbXSk6IEJbXSB7XG4gIHJldHVybiBBcnJheS5wcm90b3R5cGUuY29uY2F0LmFwcGx5KFtdLCB4cy5tYXAoZm4pKTtcbn1cblxuZnVuY3Rpb24gZ3JvdXBCeTxBPih4czogQVtdLCBrZXlGbjogKHg6IEEpID0+IHN0cmluZyk6IEFycmF5PE5vbkVtcHR5QXJyYXk8QT4+IHtcbiAgY29uc3Qgb2JqOiBSZWNvcmQ8c3RyaW5nLCBOb25FbXB0eUFycmF5PEE+PiA9IHt9O1xuICBmb3IgKGNvbnN0IHggb2YgeHMpIHtcbiAgICBjb25zdCBrZXkgPSBrZXlGbih4KTtcbiAgICBpZiAoa2V5IGluIG9iaikge1xuICAgICAgb2JqW2tleV0ucHVzaCh4KTtcbiAgICB9IGVsc2Uge1xuICAgICAgb2JqW2tleV0gPSBbeF07XG4gICAgfVxuICB9XG4gIHJldHVybiBPYmplY3QudmFsdWVzKG9iaik7XG59XG5cbmZ1bmN0aW9uKiBkZWR1cGVTdHJpbmdMaXN0cyh4czogc3RyaW5nW11bXSk6IEl0ZXJhYmxlSXRlcmF0b3I8c3RyaW5nW10+IHtcbiAgY29uc3Qgc2VlbiA9IG5ldyBTZXQ8c3RyaW5nPigpO1xuICBmb3IgKGNvbnN0IHggb2YgeHMpIHtcbiAgICB4LnNvcnQoKTtcbiAgICBjb25zdCBrZXkgPSBgJHt4LmpvaW4oJywnKX1gO1xuICAgIGlmICghc2Vlbi5oYXMoa2V5KSkge1xuICAgICAgeWllbGQgeDtcbiAgICB9XG4gICAgc2Vlbi5hZGQoa2V5KTtcbiAgfVxufSJdfQ==