"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isTagPropertyStringMap = exports.isTagPropertyJson = exports.isTagPropertyAutoScalingGroup = exports.isTagPropertyStandard = exports.isTagProperty = exports.isTagPropertyName = exports.isPropertyScrutinyType = exports.PropertyScrutinyType = exports.isUnionProperty = exports.isMapOfListsOfPrimitivesProperty = exports.isMapOfStructsProperty = exports.isPrimitiveMapProperty = exports.isMapProperty = exports.isComplexListProperty = exports.isPrimitiveListProperty = exports.isListProperty = exports.isCollectionProperty = exports.isComplexProperty = exports.isPrimitiveProperty = exports.isScalarProperty = exports.isUpdateType = exports.UpdateType = void 0;
const base_types_1 = require("./base-types");
var UpdateType;
(function (UpdateType) {
    UpdateType["Conditional"] = "Conditional";
    UpdateType["Immutable"] = "Immutable";
    UpdateType["Mutable"] = "Mutable";
})(UpdateType = exports.UpdateType || (exports.UpdateType = {}));
function isUpdateType(str) {
    switch (str) {
        case UpdateType.Conditional:
        case UpdateType.Immutable:
        case UpdateType.Mutable:
            return true;
        default:
            return false;
    }
}
exports.isUpdateType = isUpdateType;
function isScalarProperty(prop) {
    return isPrimitiveProperty(prop)
        || isComplexProperty(prop)
        // A UnionProperty is only Scalar if it defines Types or PrimitiveTypes
        || (isUnionProperty(prop) && !!(prop.Types || prop.PrimitiveTypes));
}
exports.isScalarProperty = isScalarProperty;
function isPrimitiveProperty(prop) {
    return !!prop.PrimitiveType;
}
exports.isPrimitiveProperty = isPrimitiveProperty;
function isComplexProperty(prop) {
    const propType = prop.Type;
    return propType != null && propType !== 'Map' && propType !== 'List';
}
exports.isComplexProperty = isComplexProperty;
function isCollectionProperty(prop) {
    return isListProperty(prop)
        || isMapProperty(prop)
        // A UnionProperty is only Collection if it defines ItemTypes or PrimitiveItemTypes
        || (isUnionProperty(prop) && !!(prop.ItemTypes || prop.PrimitiveItemTypes || prop.InclusiveItemTypes || prop.InclusivePrimitiveItemTypes));
}
exports.isCollectionProperty = isCollectionProperty;
function isListProperty(prop) {
    return prop.Type === 'List';
}
exports.isListProperty = isListProperty;
function isPrimitiveListProperty(prop) {
    return isListProperty(prop) && !!prop.PrimitiveItemType;
}
exports.isPrimitiveListProperty = isPrimitiveListProperty;
function isComplexListProperty(prop) {
    return isListProperty(prop) && !!prop.ItemType;
}
exports.isComplexListProperty = isComplexListProperty;
function isMapProperty(prop) {
    return prop.Type === 'Map';
}
exports.isMapProperty = isMapProperty;
function isPrimitiveMapProperty(prop) {
    return isMapProperty(prop) && !!prop.PrimitiveItemType;
}
exports.isPrimitiveMapProperty = isPrimitiveMapProperty;
function isMapOfStructsProperty(prop) {
    return isMapProperty(prop) &&
        !isPrimitiveMapProperty(prop) &&
        !isMapOfListsOfPrimitivesProperty(prop);
}
exports.isMapOfStructsProperty = isMapOfStructsProperty;
// note: this (and the MapOfListsOfPrimitives type) are not actually valid in the CFN spec!
// they are only here to support our patch of the CFN spec
// to alleviate https://github.com/aws/aws-cdk/issues/3092
function isMapOfListsOfPrimitivesProperty(prop) {
    return isMapProperty(prop) && prop.ItemType === 'List';
}
exports.isMapOfListsOfPrimitivesProperty = isMapOfListsOfPrimitivesProperty;
function isUnionProperty(prop) {
    const castProp = prop;
    return !!(castProp.ItemTypes ||
        castProp.PrimitiveTypes ||
        castProp.Types ||
        castProp.PrimitiveItemTypes ||
        castProp.InclusiveItemTypes ||
        castProp.InclusivePrimitiveItemTypes);
}
exports.isUnionProperty = isUnionProperty;
var PropertyScrutinyType;
(function (PropertyScrutinyType) {
    /**
     * No additional scrutiny
     */
    PropertyScrutinyType["None"] = "None";
    /**
     * This is an IAM policy directly on a resource
     */
    PropertyScrutinyType["InlineResourcePolicy"] = "InlineResourcePolicy";
    /**
     * Either an AssumeRolePolicyDocument or a dictionary of policy documents
     */
    PropertyScrutinyType["InlineIdentityPolicies"] = "InlineIdentityPolicies";
    /**
     * A list of managed policies (on an identity resource)
     */
    PropertyScrutinyType["ManagedPolicies"] = "ManagedPolicies";
    /**
     * A set of ingress rules (on a security group)
     */
    PropertyScrutinyType["IngressRules"] = "IngressRules";
    /**
     * A set of egress rules (on a security group)
     */
    PropertyScrutinyType["EgressRules"] = "EgressRules";
})(PropertyScrutinyType = exports.PropertyScrutinyType || (exports.PropertyScrutinyType = {}));
function isPropertyScrutinyType(str) {
    return PropertyScrutinyType[str] !== undefined;
}
exports.isPropertyScrutinyType = isPropertyScrutinyType;
const tagPropertyNames = {
    FileSystemTags: '',
    HostedZoneTags: '',
    Tags: '',
    UserPoolTags: '',
    AccessPointTags: '',
};
function isTagPropertyName(name) {
    if (undefined === name) {
        return false;
    }
    return tagPropertyNames.hasOwnProperty(name);
}
exports.isTagPropertyName = isTagPropertyName;
/**
 * This function validates that the property **can** be a Tag Property
 *
 * The property is only a Tag if the name of this property is Tags, which is
 * validated using `ResourceType.isTaggable(resource)`.
 */
function isTagProperty(prop) {
    return (isTagPropertyStandard(prop) ||
        isTagPropertyAutoScalingGroup(prop) ||
        isTagPropertyJson(prop) ||
        isTagPropertyStringMap(prop));
}
exports.isTagProperty = isTagProperty;
function isTagPropertyStandard(prop) {
    return (prop.ItemType === 'Tag' ||
        prop.ItemType === 'TagsEntry' ||
        prop.Type === 'Tags' ||
        prop.ItemType === 'TagRef' ||
        prop.ItemType === 'ElasticFileSystemTag' ||
        prop.ItemType === 'HostedZoneTag' ||
        prop.ItemType === 'AccessPointTag');
}
exports.isTagPropertyStandard = isTagPropertyStandard;
function isTagPropertyAutoScalingGroup(prop) {
    return prop.ItemType === 'TagProperty';
}
exports.isTagPropertyAutoScalingGroup = isTagPropertyAutoScalingGroup;
function isTagPropertyJson(prop) {
    return prop.PrimitiveType === base_types_1.PrimitiveType.Json;
}
exports.isTagPropertyJson = isTagPropertyJson;
function isTagPropertyStringMap(prop) {
    return prop.PrimitiveItemType === 'String';
}
exports.isTagPropertyStringMap = isTagPropertyStringMap;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJvcGVydHkuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJwcm9wZXJ0eS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2Q0FBeUQ7QUFvSXpELElBQVksVUFJWDtBQUpELFdBQVksVUFBVTtJQUNwQix5Q0FBMkIsQ0FBQTtJQUMzQixxQ0FBdUIsQ0FBQTtJQUN2QixpQ0FBbUIsQ0FBQTtBQUNyQixDQUFDLEVBSlcsVUFBVSxHQUFWLGtCQUFVLEtBQVYsa0JBQVUsUUFJckI7QUFFRCxTQUFnQixZQUFZLENBQUMsR0FBVztJQUN0QyxRQUFRLEdBQUcsRUFBRTtRQUNYLEtBQUssVUFBVSxDQUFDLFdBQVcsQ0FBQztRQUM1QixLQUFLLFVBQVUsQ0FBQyxTQUFTLENBQUM7UUFDMUIsS0FBSyxVQUFVLENBQUMsT0FBTztZQUNyQixPQUFPLElBQUksQ0FBQztRQUNkO1lBQ0UsT0FBTyxLQUFLLENBQUM7S0FDaEI7QUFDSCxDQUFDO0FBVEQsb0NBU0M7QUFFRCxTQUFnQixnQkFBZ0IsQ0FBQyxJQUFjO0lBQzdDLE9BQU8sbUJBQW1CLENBQUMsSUFBSSxDQUFDO1dBQzNCLGlCQUFpQixDQUFDLElBQUksQ0FBQztRQUMxQix1RUFBdUU7V0FDcEUsQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssSUFBSSxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQztBQUN4RSxDQUFDO0FBTEQsNENBS0M7QUFFRCxTQUFnQixtQkFBbUIsQ0FBQyxJQUFjO0lBQ2hELE9BQU8sQ0FBQyxDQUFFLElBQTBCLENBQUMsYUFBYSxDQUFDO0FBQ3JELENBQUM7QUFGRCxrREFFQztBQUVELFNBQWdCLGlCQUFpQixDQUFDLElBQWM7SUFDOUMsTUFBTSxRQUFRLEdBQUksSUFBd0IsQ0FBQyxJQUFJLENBQUM7SUFDaEQsT0FBTyxRQUFRLElBQUksSUFBSSxJQUFJLFFBQVEsS0FBSyxLQUFLLElBQUksUUFBUSxLQUFLLE1BQU0sQ0FBQztBQUN2RSxDQUFDO0FBSEQsOENBR0M7QUFFRCxTQUFnQixvQkFBb0IsQ0FBQyxJQUFjO0lBQ2pELE9BQU8sY0FBYyxDQUFDLElBQUksQ0FBQztXQUN0QixhQUFhLENBQUMsSUFBSSxDQUFDO1FBQ3RCLG1GQUFtRjtXQUNoRixDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsU0FBUyxJQUFJLElBQUksQ0FBQyxrQkFBa0IsSUFBSSxJQUFJLENBQUMsa0JBQWtCLElBQUksSUFBSSxDQUFDLDJCQUEyQixDQUFDLENBQUMsQ0FBQztBQUMvSSxDQUFDO0FBTEQsb0RBS0M7QUFFRCxTQUFnQixjQUFjLENBQUMsSUFBYztJQUMzQyxPQUFRLElBQXFCLENBQUMsSUFBSSxLQUFLLE1BQU0sQ0FBQztBQUNoRCxDQUFDO0FBRkQsd0NBRUM7QUFFRCxTQUFnQix1QkFBdUIsQ0FBQyxJQUFjO0lBQ3BELE9BQU8sY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBRSxJQUE4QixDQUFDLGlCQUFpQixDQUFDO0FBQ3JGLENBQUM7QUFGRCwwREFFQztBQUVELFNBQWdCLHFCQUFxQixDQUFDLElBQWM7SUFDbEQsT0FBTyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFFLElBQTRCLENBQUMsUUFBUSxDQUFDO0FBQzFFLENBQUM7QUFGRCxzREFFQztBQUVELFNBQWdCLGFBQWEsQ0FBQyxJQUFjO0lBQzFDLE9BQVEsSUFBb0IsQ0FBQyxJQUFJLEtBQUssS0FBSyxDQUFDO0FBQzlDLENBQUM7QUFGRCxzQ0FFQztBQUVELFNBQWdCLHNCQUFzQixDQUFDLElBQWM7SUFDbkQsT0FBTyxhQUFhLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFFLElBQTZCLENBQUMsaUJBQWlCLENBQUM7QUFDbkYsQ0FBQztBQUZELHdEQUVDO0FBRUQsU0FBZ0Isc0JBQXNCLENBQUMsSUFBYztJQUNuRCxPQUFPLGFBQWEsQ0FBQyxJQUFJLENBQUM7UUFDeEIsQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUM7UUFDN0IsQ0FBQyxnQ0FBZ0MsQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUM1QyxDQUFDO0FBSkQsd0RBSUM7QUFFRCwyRkFBMkY7QUFDM0YsMERBQTBEO0FBQzFELDBEQUEwRDtBQUMxRCxTQUFnQixnQ0FBZ0MsQ0FBQyxJQUFjO0lBQzdELE9BQU8sYUFBYSxDQUFDLElBQUksQ0FBQyxJQUFLLElBQTJCLENBQUMsUUFBUSxLQUFLLE1BQU0sQ0FBQztBQUNqRixDQUFDO0FBRkQsNEVBRUM7QUFFRCxTQUFnQixlQUFlLENBQUMsSUFBYztJQUM1QyxNQUFNLFFBQVEsR0FBRyxJQUFxQixDQUFDO0lBQ3ZDLE9BQU8sQ0FBQyxDQUFDLENBQ1AsUUFBUSxDQUFDLFNBQVM7UUFDbEIsUUFBUSxDQUFDLGNBQWM7UUFDdkIsUUFBUSxDQUFDLEtBQUs7UUFDZCxRQUFRLENBQUMsa0JBQWtCO1FBQzNCLFFBQVEsQ0FBQyxrQkFBa0I7UUFDM0IsUUFBUSxDQUFDLDJCQUEyQixDQUNyQyxDQUFDO0FBQ0osQ0FBQztBQVZELDBDQVVDO0FBRUQsSUFBWSxvQkE4Qlg7QUE5QkQsV0FBWSxvQkFBb0I7SUFDOUI7O09BRUc7SUFDSCxxQ0FBYSxDQUFBO0lBRWI7O09BRUc7SUFDSCxxRUFBNkMsQ0FBQTtJQUU3Qzs7T0FFRztJQUNILHlFQUFpRCxDQUFBO0lBRWpEOztPQUVHO0lBQ0gsMkRBQW1DLENBQUE7SUFFbkM7O09BRUc7SUFDSCxxREFBNkIsQ0FBQTtJQUU3Qjs7T0FFRztJQUNILG1EQUEyQixDQUFBO0FBQzdCLENBQUMsRUE5Qlcsb0JBQW9CLEdBQXBCLDRCQUFvQixLQUFwQiw0QkFBb0IsUUE4Qi9CO0FBRUQsU0FBZ0Isc0JBQXNCLENBQUMsR0FBVztJQUNoRCxPQUFRLG9CQUE0QixDQUFDLEdBQUcsQ0FBQyxLQUFLLFNBQVMsQ0FBQztBQUMxRCxDQUFDO0FBRkQsd0RBRUM7QUFFRCxNQUFNLGdCQUFnQixHQUFHO0lBQ3ZCLGNBQWMsRUFBRSxFQUFFO0lBQ2xCLGNBQWMsRUFBRSxFQUFFO0lBQ2xCLElBQUksRUFBRSxFQUFFO0lBQ1IsWUFBWSxFQUFFLEVBQUU7SUFDaEIsZUFBZSxFQUFFLEVBQUU7Q0FDcEIsQ0FBQztBQUlGLFNBQWdCLGlCQUFpQixDQUFDLElBQWE7SUFDN0MsSUFBSSxTQUFTLEtBQUssSUFBSSxFQUFFO1FBQ3RCLE9BQU8sS0FBSyxDQUFDO0tBQ2Q7SUFDRCxPQUFPLGdCQUFnQixDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUMvQyxDQUFDO0FBTEQsOENBS0M7QUFDRDs7Ozs7R0FLRztBQUNILFNBQWdCLGFBQWEsQ0FBQyxJQUFjO0lBQzFDLE9BQU8sQ0FDTCxxQkFBcUIsQ0FBQyxJQUFJLENBQUM7UUFDM0IsNkJBQTZCLENBQUMsSUFBSSxDQUFDO1FBQ25DLGlCQUFpQixDQUFDLElBQUksQ0FBQztRQUN2QixzQkFBc0IsQ0FBQyxJQUFJLENBQUMsQ0FDN0IsQ0FBQztBQUNKLENBQUM7QUFQRCxzQ0FPQztBQUVELFNBQWdCLHFCQUFxQixDQUFDLElBQWM7SUFDbEQsT0FBTyxDQUNKLElBQTRCLENBQUMsUUFBUSxLQUFLLEtBQUs7UUFDL0MsSUFBNEIsQ0FBQyxRQUFRLEtBQUssV0FBVztRQUNyRCxJQUE0QixDQUFDLElBQUksS0FBSyxNQUFNO1FBQzVDLElBQTRCLENBQUMsUUFBUSxLQUFLLFFBQVE7UUFDbEQsSUFBNEIsQ0FBQyxRQUFRLEtBQUssc0JBQXNCO1FBQ2hFLElBQTRCLENBQUMsUUFBUSxLQUFLLGVBQWU7UUFDekQsSUFBNEIsQ0FBQyxRQUFRLEtBQUssZ0JBQWdCLENBQzVELENBQUM7QUFFSixDQUFDO0FBWEQsc0RBV0M7QUFFRCxTQUFnQiw2QkFBNkIsQ0FBQyxJQUFjO0lBQzFELE9BQVEsSUFBb0MsQ0FBQyxRQUFRLEtBQUssYUFBYSxDQUFDO0FBQzFFLENBQUM7QUFGRCxzRUFFQztBQUVELFNBQWdCLGlCQUFpQixDQUFDLElBQWM7SUFDOUMsT0FBUSxJQUF3QixDQUFDLGFBQWEsS0FBSywwQkFBYSxDQUFDLElBQUksQ0FBQztBQUN4RSxDQUFDO0FBRkQsOENBRUM7QUFFRCxTQUFnQixzQkFBc0IsQ0FBQyxJQUFjO0lBQ25ELE9BQVEsSUFBNkIsQ0FBQyxpQkFBaUIsS0FBSyxRQUFRLENBQUM7QUFDdkUsQ0FBQztBQUZELHdEQUVDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgRG9jdW1lbnRlZCwgUHJpbWl0aXZlVHlwZSB9IGZyb20gJy4vYmFzZS10eXBlcyc7XG5cbmV4cG9ydCB0eXBlIFByb3BlcnR5ID0gU2NhbGFyUHJvcGVydHkgfCBDb2xsZWN0aW9uUHJvcGVydHk7XG5leHBvcnQgdHlwZSBTY2FsYXJQcm9wZXJ0eSA9IFByaW1pdGl2ZVByb3BlcnR5IHwgQ29tcGxleFByb3BlcnR5IHwgVW5pb25Qcm9wZXJ0eTtcbmV4cG9ydCB0eXBlIENvbGxlY3Rpb25Qcm9wZXJ0eSA9IExpc3RQcm9wZXJ0eSB8IE1hcFByb3BlcnR5IHwgVW5pb25Qcm9wZXJ0eTtcbmV4cG9ydCB0eXBlIExpc3RQcm9wZXJ0eSA9IFByaW1pdGl2ZUxpc3RQcm9wZXJ0eSB8IENvbXBsZXhMaXN0UHJvcGVydHk7XG5leHBvcnQgdHlwZSBNYXBQcm9wZXJ0eSA9IFByaW1pdGl2ZU1hcFByb3BlcnR5IHwgQ29tcGxleE1hcFByb3BlcnR5O1xuZXhwb3J0IHR5cGUgQ29tcGxleE1hcFByb3BlcnR5ID0gTWFwT2ZTdHJ1Y3RzIHwgTWFwT2ZMaXN0c09mUHJpbWl0aXZlcztcbmV4cG9ydCB0eXBlIFRhZ1Byb3BlcnR5ID0gVGFnUHJvcGVydHlTdGFuZGFyZCB8IFRhZ1Byb3BlcnR5QXV0b1NjYWxpbmdHcm91cCB8IFRhZ1Byb3BlcnR5SnNvbiB8IFRhZ1Byb3BlcnR5U3RyaW5nTWFwO1xuXG5leHBvcnQgaW50ZXJmYWNlIFByb3BlcnR5QmFzZSBleHRlbmRzIERvY3VtZW50ZWQge1xuICAvKipcbiAgICogSW5kaWNhdGVzIHdoZXRoZXIgdGhlIHByb3BlcnR5IGlzIHJlcXVpcmVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgUmVxdWlyZWQ/OiBib29sZWFuO1xuICAvKipcbiAgICogRHVyaW5nIGEgc3RhY2sgdXBkYXRlLCB0aGUgdXBkYXRlIGJlaGF2aW9yIHdoZW4geW91IGFkZCwgcmVtb3ZlLCBvciBtb2RpZnkgdGhlIHByb3BlcnR5LiBBV1MgQ2xvdWRGb3JtYXRpb25cbiAgICogcmVwbGFjZXMgdGhlIHJlc291cmNlIHdoZW4geW91IGNoYW5nZSBgw4xtbXV0YWJsZWBgcHJvcGVydGllcy4gQVdTIENsb3VkRm9ybWF0aW9uIGRvZXNuJ3QgcmVwbGFjZSB0aGUgcmVzb3VyY2VcbiAgICogd2hlbiB5b3UgY2hhbmdlIGBgTXV0YWJsZWBgIHByb3BlcnRpZXMuIGBgQ29uZGl0aW9uYWxgYCB1cGRhdGVzIGNhbiBiZSBtdXRhYmxlIG9yIGltbXV0YWJsZSwgZGVwZW5kaW5nIG9uLCBmb3JcbiAgICogZXhhbXBsZSwgd2hpY2ggb3RoZXIgcHJvcGVydGllcyB5b3UgdXBkYXRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgVXBkYXRlVHlwZS5NdXRhYmxlXG4gICAqL1xuICBVcGRhdGVUeXBlPzogVXBkYXRlVHlwZTtcblxuICAvKipcbiAgICogRHVyaW5nIGEgc3RhY2sgdXBkYXRlLCB3aGF0IGtpbmQgb2YgYWRkaXRpb25hbCBzY3J1dGlueSBjaGFuZ2VzIHRvIHRoaXMgcHJvcGVydHkgc2hvdWxkIGJlIHN1YmplY3RlZCB0b1xuICAgKlxuICAgKiBAZGVmYXVsdCBOb25lXG4gICAqL1xuICBTY3J1dGlueVR5cGU/OiBQcm9wZXJ0eVNjcnV0aW55VHlwZTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBQcmltaXRpdmVQcm9wZXJ0eSBleHRlbmRzIFByb3BlcnR5QmFzZSB7XG4gIC8qKiBUaGUgdmFsaWQgcHJpbWl0aXZlIHR5cGUgZm9yIHRoZSBwcm9wZXJ0eS4gKi9cbiAgUHJpbWl0aXZlVHlwZTogUHJpbWl0aXZlVHlwZTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBDb21wbGV4UHJvcGVydHkgZXh0ZW5kcyBQcm9wZXJ0eUJhc2Uge1xuICAvKiogVGhlIHR5cGUgb2YgdmFsaWQgdmFsdWVzIGZvciB0aGlzIHByb3BlcnR5ICovXG4gIFR5cGU6IHN0cmluZztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBMaXN0UHJvcGVydHlCYXNlIGV4dGVuZHMgUHJvcGVydHlCYXNlIHtcbiAgLyoqXG4gICAqIEEgbGlzdCBpcyBhIGNvbW1hLXNlcGFyYXRlZCBsaXN0IG9mIHZhbHVlcy5cbiAgICovXG4gIFR5cGU6ICdMaXN0JztcbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIEFXUyBDbG91ZEZvcm1hdGlvbiBhbGxvd3MgZHVwbGljYXRlIHZhbHVlcy4gSWYgdGhlIHZhbHVlIGlzIGBgdHJ1ZWBgLCBBV1MgQ2xvdWRGb3JtYXRpb25cbiAgICogaWdub3JlcyBkdXBsaWNhdGUgdmFsdWVzLiBpZiB0aGUgdmFsdWUgaXMgIGBgZmFsc2VgYCwgQVdTIENsb3VkRm9ybWF0aW9uIHJldHVybnMgYW4gYXJyb3IgaWYgeW91IHN1Ym1pdCBkdXBsaWNhdGVcbiAgICogdmFsdWVzLlxuICAgKi9cbiAgRHVwbGljYXRlc0FsbG93ZWQ/OiBib29sZWFuO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFByaW1pdGl2ZUxpc3RQcm9wZXJ0eSBleHRlbmRzIExpc3RQcm9wZXJ0eUJhc2Uge1xuICAvKiogVGhlIHZhbGlkIHByaW1pdGl2ZSB0eXBlIGZvciB0aGUgcHJvcGVydHkuICovXG4gIFByaW1pdGl2ZUl0ZW1UeXBlOiBQcmltaXRpdmVUeXBlO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIENvbXBsZXhMaXN0UHJvcGVydHkgZXh0ZW5kcyBMaXN0UHJvcGVydHlCYXNlIHtcbiAgLyoqIFZhbGlkIHZhbHVlcyBmb3IgdGhlIHByb3BlcnR5ICovXG4gIEl0ZW1UeXBlOiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgTWFwUHJvcGVydHlCYXNlIGV4dGVuZHMgUHJvcGVydHlCYXNlIHtcbiAgLyoqIEEgbWFwIGlzIGEgc2V0IG9mIGtleS12YWx1ZSBwYWlycywgd2hlcmUgdGhlIGtleXMgYXJlIGFsd2F5cyBzdHJpbmdzLiAqL1xuICBUeXBlOiAnTWFwJztcbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIEFXUyBDbG91ZEZvcm1hdGlvbiBhbGxvd3MgZHVwbGljYXRlIHZhbHVlcy4gSWYgdGhlIHZhbHVlIGlzIGBgdHJ1ZWBgLCBBV1MgQ2xvdWRGb3JtYXRpb25cbiAgICogaWdub3JlcyBkdXBsaWNhdGUgdmFsdWVzLiBpZiB0aGUgdmFsdWUgaXMgIGBgZmFsc2VgYCwgQVdTIENsb3VkRm9ybWF0aW9uIHJldHVybnMgYW4gYXJyb3IgaWYgeW91IHN1Ym1pdCBkdXBsaWNhdGVcbiAgICogdmFsdWVzLlxuICAgKi9cbiAgRHVwbGljYXRlc0FsbG93ZWQ/OiBmYWxzZTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBQcmltaXRpdmVNYXBQcm9wZXJ0eSBleHRlbmRzIE1hcFByb3BlcnR5QmFzZSB7XG4gIC8qKiBUaGUgdmFsaWQgcHJpbWl0aXZlIHR5cGUgZm9yIHRoZSBwcm9wZXJ0eS4gKi9cbiAgUHJpbWl0aXZlSXRlbVR5cGU6IFByaW1pdGl2ZVR5cGU7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgTWFwT2ZTdHJ1Y3RzIGV4dGVuZHMgTWFwUHJvcGVydHlCYXNlIHtcbiAgLyoqIFZhbGlkIHZhbHVlcyBmb3IgdGhlIHByb3BlcnR5ICovXG4gIEl0ZW1UeXBlOiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgTWFwT2ZMaXN0c09mUHJpbWl0aXZlcyBleHRlbmRzIE1hcFByb3BlcnR5QmFzZSB7XG4gIC8qKiBUaGUgdHlwZSBvZiB0aGUgbWFwIHZhbHVlcywgd2hpY2ggaW4gdGhpcyBjYXNlIGlzIGFsd2F5cyAnTGlzdCcuICovXG4gIEl0ZW1UeXBlOiBzdHJpbmc7XG5cbiAgLyoqIFRoZSB2YWxpZCBwcmltaXRpdmUgdHlwZSBmb3IgdGhlIGxpc3RzIHRoYXQgYXJlIHRoZSB2YWx1ZXMgb2YgdGhpcyBtYXAuICovXG4gIFByaW1pdGl2ZUl0ZW1JdGVtVHlwZTogUHJpbWl0aXZlVHlwZTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBUYWdQcm9wZXJ0eVN0YW5kYXJkIGV4dGVuZHMgUHJvcGVydHlCYXNlIHtcbiAgSXRlbVR5cGU6ICdUYWcnIHwgJ1RhZ3NFbnRyeScgfCAnVGFnUmVmJyB8ICdFbGFzdGljRmlsZVN5c3RlbVRhZycgfCAnSG9zdGVkWm9uZVRhZycgfCAnQWNjZXNzUG9pbnRUYWcnO1xuICBUeXBlOiAnVGFncyc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgVGFnUHJvcGVydHlBdXRvU2NhbGluZ0dyb3VwIGV4dGVuZHMgUHJvcGVydHlCYXNlIHtcbiAgSXRlbVR5cGU6ICdUYWdQcm9wZXJ0eSc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgVGFnUHJvcGVydHlKc29uIGV4dGVuZHMgUHJvcGVydHlCYXNlIHtcbiAgUHJpbWl0aXZlVHlwZTogUHJpbWl0aXZlVHlwZS5Kc29uO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFRhZ1Byb3BlcnR5U3RyaW5nTWFwIGV4dGVuZHMgUHJvcGVydHlCYXNlIHtcbiAgUHJpbWl0aXZlSXRlbVR5cGU6ICdTdHJpbmcnO1xufVxuXG4vKipcbiAqIEEgcHJvcGVydHkgdHlwZSB0aGF0IGNhbiBiZSBvbmUgb2Ygc2V2ZXJhbCB0eXBlcy4gQ3VycmVudGx5IHVzZWQgb25seSBpbiBTQU0uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVW5pb25Qcm9wZXJ0eSBleHRlbmRzIFByb3BlcnR5QmFzZSB7XG4gIC8qKiBWYWxpZCBwcmltaXRpdmUgdHlwZXMgZm9yIHRoZSBwcm9wZXJ0eSAqL1xuICBQcmltaXRpdmVUeXBlcz86IFByaW1pdGl2ZVR5cGVbXTtcbiAgLyoqIFZhbGlkIGNvbXBsZXggdHlwZXMgZm9yIHRoZSBwcm9wZXJ0eSAqL1xuICBUeXBlcz86IHN0cmluZ1tdXG4gIC8qKiBWYWxpZCBwcmltaXRpdmUgaXRlbSB0eXBlcyBmb3IgdGhpcyBwcm9wZXJ0eSAqL1xuICBQcmltaXRpdmVJdGVtVHlwZXM/OiBQcmltaXRpdmVUeXBlW107XG4gIC8qKiBWYWxpZCBsaXN0IGl0ZW0gdHlwZXMgZm9yIHRoZSBwcm9wZXJ0eSAqL1xuICBJdGVtVHlwZXM/OiBzdHJpbmdbXTtcbiAgLyoqIFZhbGlkIGNvbXBsZXggdHlwZXMgZm9yIHRoaXMgcHJvcGVydHkgKi9cbiAgSW5jbHVzaXZlSXRlbVR5cGVzPzogc3RyaW5nW107XG4gIC8qKiBWYWxpZCBwcmltaXRpdmUgaXRlbSB0eXBlcyBmb3IgdGhpcyBwcm9wZXJ0eSAqL1xuICBJbmNsdXNpdmVQcmltaXRpdmVJdGVtVHlwZXM/OiBQcmltaXRpdmVUeXBlW107XG59XG5cbmV4cG9ydCBlbnVtIFVwZGF0ZVR5cGUge1xuICBDb25kaXRpb25hbCA9ICdDb25kaXRpb25hbCcsXG4gIEltbXV0YWJsZSA9ICdJbW11dGFibGUnLFxuICBNdXRhYmxlID0gJ011dGFibGUnXG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpc1VwZGF0ZVR5cGUoc3RyOiBzdHJpbmcpOiBzdHIgaXMgVXBkYXRlVHlwZSB7XG4gIHN3aXRjaCAoc3RyKSB7XG4gICAgY2FzZSBVcGRhdGVUeXBlLkNvbmRpdGlvbmFsOlxuICAgIGNhc2UgVXBkYXRlVHlwZS5JbW11dGFibGU6XG4gICAgY2FzZSBVcGRhdGVUeXBlLk11dGFibGU6XG4gICAgICByZXR1cm4gdHJ1ZTtcbiAgICBkZWZhdWx0OlxuICAgICAgcmV0dXJuIGZhbHNlO1xuICB9XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpc1NjYWxhclByb3BlcnR5KHByb3A6IFByb3BlcnR5KTogcHJvcCBpcyBTY2FsYXJQcm9wZXJ0eSB7XG4gIHJldHVybiBpc1ByaW1pdGl2ZVByb3BlcnR5KHByb3ApXG4gICAgfHwgaXNDb21wbGV4UHJvcGVydHkocHJvcClcbiAgICAvLyBBIFVuaW9uUHJvcGVydHkgaXMgb25seSBTY2FsYXIgaWYgaXQgZGVmaW5lcyBUeXBlcyBvciBQcmltaXRpdmVUeXBlc1xuICAgIHx8IChpc1VuaW9uUHJvcGVydHkocHJvcCkgJiYgISEocHJvcC5UeXBlcyB8fCBwcm9wLlByaW1pdGl2ZVR5cGVzKSk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpc1ByaW1pdGl2ZVByb3BlcnR5KHByb3A6IFByb3BlcnR5KTogcHJvcCBpcyBQcmltaXRpdmVQcm9wZXJ0eSB7XG4gIHJldHVybiAhIShwcm9wIGFzIFByaW1pdGl2ZVByb3BlcnR5KS5QcmltaXRpdmVUeXBlO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gaXNDb21wbGV4UHJvcGVydHkocHJvcDogUHJvcGVydHkpOiBwcm9wIGlzIENvbXBsZXhQcm9wZXJ0eSB7XG4gIGNvbnN0IHByb3BUeXBlID0gKHByb3AgYXMgQ29tcGxleFByb3BlcnR5KS5UeXBlO1xuICByZXR1cm4gcHJvcFR5cGUgIT0gbnVsbCAmJiBwcm9wVHlwZSAhPT0gJ01hcCcgJiYgcHJvcFR5cGUgIT09ICdMaXN0Jztcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGlzQ29sbGVjdGlvblByb3BlcnR5KHByb3A6IFByb3BlcnR5KTogcHJvcCBpcyBDb2xsZWN0aW9uUHJvcGVydHkge1xuICByZXR1cm4gaXNMaXN0UHJvcGVydHkocHJvcClcbiAgICB8fCBpc01hcFByb3BlcnR5KHByb3ApXG4gICAgLy8gQSBVbmlvblByb3BlcnR5IGlzIG9ubHkgQ29sbGVjdGlvbiBpZiBpdCBkZWZpbmVzIEl0ZW1UeXBlcyBvciBQcmltaXRpdmVJdGVtVHlwZXNcbiAgICB8fCAoaXNVbmlvblByb3BlcnR5KHByb3ApICYmICEhKHByb3AuSXRlbVR5cGVzIHx8IHByb3AuUHJpbWl0aXZlSXRlbVR5cGVzIHx8IHByb3AuSW5jbHVzaXZlSXRlbVR5cGVzIHx8IHByb3AuSW5jbHVzaXZlUHJpbWl0aXZlSXRlbVR5cGVzKSk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpc0xpc3RQcm9wZXJ0eShwcm9wOiBQcm9wZXJ0eSk6IHByb3AgaXMgTGlzdFByb3BlcnR5IHtcbiAgcmV0dXJuIChwcm9wIGFzIExpc3RQcm9wZXJ0eSkuVHlwZSA9PT0gJ0xpc3QnO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gaXNQcmltaXRpdmVMaXN0UHJvcGVydHkocHJvcDogUHJvcGVydHkpOiBwcm9wIGlzIFByaW1pdGl2ZUxpc3RQcm9wZXJ0eSB7XG4gIHJldHVybiBpc0xpc3RQcm9wZXJ0eShwcm9wKSAmJiAhIShwcm9wIGFzIFByaW1pdGl2ZUxpc3RQcm9wZXJ0eSkuUHJpbWl0aXZlSXRlbVR5cGU7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpc0NvbXBsZXhMaXN0UHJvcGVydHkocHJvcDogUHJvcGVydHkpOiBwcm9wIGlzIENvbXBsZXhMaXN0UHJvcGVydHkge1xuICByZXR1cm4gaXNMaXN0UHJvcGVydHkocHJvcCkgJiYgISEocHJvcCBhcyBDb21wbGV4TGlzdFByb3BlcnR5KS5JdGVtVHlwZTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGlzTWFwUHJvcGVydHkocHJvcDogUHJvcGVydHkpOiBwcm9wIGlzIE1hcFByb3BlcnR5IHtcbiAgcmV0dXJuIChwcm9wIGFzIE1hcFByb3BlcnR5KS5UeXBlID09PSAnTWFwJztcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGlzUHJpbWl0aXZlTWFwUHJvcGVydHkocHJvcDogUHJvcGVydHkpOiBwcm9wIGlzIFByaW1pdGl2ZU1hcFByb3BlcnR5IHtcbiAgcmV0dXJuIGlzTWFwUHJvcGVydHkocHJvcCkgJiYgISEocHJvcCBhcyBQcmltaXRpdmVNYXBQcm9wZXJ0eSkuUHJpbWl0aXZlSXRlbVR5cGU7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpc01hcE9mU3RydWN0c1Byb3BlcnR5KHByb3A6IFByb3BlcnR5KTogcHJvcCBpcyBNYXBPZlN0cnVjdHMge1xuICByZXR1cm4gaXNNYXBQcm9wZXJ0eShwcm9wKSAmJlxuICAgICFpc1ByaW1pdGl2ZU1hcFByb3BlcnR5KHByb3ApICYmXG4gICAgIWlzTWFwT2ZMaXN0c09mUHJpbWl0aXZlc1Byb3BlcnR5KHByb3ApO1xufVxuXG4vLyBub3RlOiB0aGlzIChhbmQgdGhlIE1hcE9mTGlzdHNPZlByaW1pdGl2ZXMgdHlwZSkgYXJlIG5vdCBhY3R1YWxseSB2YWxpZCBpbiB0aGUgQ0ZOIHNwZWMhXG4vLyB0aGV5IGFyZSBvbmx5IGhlcmUgdG8gc3VwcG9ydCBvdXIgcGF0Y2ggb2YgdGhlIENGTiBzcGVjXG4vLyB0byBhbGxldmlhdGUgaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL2lzc3Vlcy8zMDkyXG5leHBvcnQgZnVuY3Rpb24gaXNNYXBPZkxpc3RzT2ZQcmltaXRpdmVzUHJvcGVydHkocHJvcDogUHJvcGVydHkpOiBwcm9wIGlzIE1hcE9mTGlzdHNPZlByaW1pdGl2ZXMge1xuICByZXR1cm4gaXNNYXBQcm9wZXJ0eShwcm9wKSAmJiAocHJvcCBhcyBDb21wbGV4TWFwUHJvcGVydHkpLkl0ZW1UeXBlID09PSAnTGlzdCc7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpc1VuaW9uUHJvcGVydHkocHJvcDogUHJvcGVydHkpOiBwcm9wIGlzIFVuaW9uUHJvcGVydHkge1xuICBjb25zdCBjYXN0UHJvcCA9IHByb3AgYXMgVW5pb25Qcm9wZXJ0eTtcbiAgcmV0dXJuICEhKFxuICAgIGNhc3RQcm9wLkl0ZW1UeXBlcyB8fFxuICAgIGNhc3RQcm9wLlByaW1pdGl2ZVR5cGVzIHx8XG4gICAgY2FzdFByb3AuVHlwZXMgfHxcbiAgICBjYXN0UHJvcC5QcmltaXRpdmVJdGVtVHlwZXMgfHxcbiAgICBjYXN0UHJvcC5JbmNsdXNpdmVJdGVtVHlwZXMgfHxcbiAgICBjYXN0UHJvcC5JbmNsdXNpdmVQcmltaXRpdmVJdGVtVHlwZXNcbiAgKTtcbn1cblxuZXhwb3J0IGVudW0gUHJvcGVydHlTY3J1dGlueVR5cGUge1xuICAvKipcbiAgICogTm8gYWRkaXRpb25hbCBzY3J1dGlueVxuICAgKi9cbiAgTm9uZSA9ICdOb25lJyxcblxuICAvKipcbiAgICogVGhpcyBpcyBhbiBJQU0gcG9saWN5IGRpcmVjdGx5IG9uIGEgcmVzb3VyY2VcbiAgICovXG4gIElubGluZVJlc291cmNlUG9saWN5ID0gJ0lubGluZVJlc291cmNlUG9saWN5JyxcblxuICAvKipcbiAgICogRWl0aGVyIGFuIEFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudCBvciBhIGRpY3Rpb25hcnkgb2YgcG9saWN5IGRvY3VtZW50c1xuICAgKi9cbiAgSW5saW5lSWRlbnRpdHlQb2xpY2llcyA9ICdJbmxpbmVJZGVudGl0eVBvbGljaWVzJyxcblxuICAvKipcbiAgICogQSBsaXN0IG9mIG1hbmFnZWQgcG9saWNpZXMgKG9uIGFuIGlkZW50aXR5IHJlc291cmNlKVxuICAgKi9cbiAgTWFuYWdlZFBvbGljaWVzID0gJ01hbmFnZWRQb2xpY2llcycsXG5cbiAgLyoqXG4gICAqIEEgc2V0IG9mIGluZ3Jlc3MgcnVsZXMgKG9uIGEgc2VjdXJpdHkgZ3JvdXApXG4gICAqL1xuICBJbmdyZXNzUnVsZXMgPSAnSW5ncmVzc1J1bGVzJyxcblxuICAvKipcbiAgICogQSBzZXQgb2YgZWdyZXNzIHJ1bGVzIChvbiBhIHNlY3VyaXR5IGdyb3VwKVxuICAgKi9cbiAgRWdyZXNzUnVsZXMgPSAnRWdyZXNzUnVsZXMnLFxufVxuXG5leHBvcnQgZnVuY3Rpb24gaXNQcm9wZXJ0eVNjcnV0aW55VHlwZShzdHI6IHN0cmluZyk6IHN0ciBpcyBQcm9wZXJ0eVNjcnV0aW55VHlwZSB7XG4gIHJldHVybiAoUHJvcGVydHlTY3J1dGlueVR5cGUgYXMgYW55KVtzdHJdICE9PSB1bmRlZmluZWQ7XG59XG5cbmNvbnN0IHRhZ1Byb3BlcnR5TmFtZXMgPSB7XG4gIEZpbGVTeXN0ZW1UYWdzOiAnJyxcbiAgSG9zdGVkWm9uZVRhZ3M6ICcnLFxuICBUYWdzOiAnJyxcbiAgVXNlclBvb2xUYWdzOiAnJyxcbiAgQWNjZXNzUG9pbnRUYWdzOiAnJyxcbn07XG5cbmV4cG9ydCB0eXBlIFRhZ1Byb3BlcnR5TmFtZSA9IGtleW9mIHR5cGVvZiB0YWdQcm9wZXJ0eU5hbWVzO1xuXG5leHBvcnQgZnVuY3Rpb24gaXNUYWdQcm9wZXJ0eU5hbWUobmFtZT86IHN0cmluZyk6IG5hbWUgaXMgVGFnUHJvcGVydHlOYW1lIHtcbiAgaWYgKHVuZGVmaW5lZCA9PT0gbmFtZSkge1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuICByZXR1cm4gdGFnUHJvcGVydHlOYW1lcy5oYXNPd25Qcm9wZXJ0eShuYW1lKTtcbn1cbi8qKlxuICogVGhpcyBmdW5jdGlvbiB2YWxpZGF0ZXMgdGhhdCB0aGUgcHJvcGVydHkgKipjYW4qKiBiZSBhIFRhZyBQcm9wZXJ0eVxuICpcbiAqIFRoZSBwcm9wZXJ0eSBpcyBvbmx5IGEgVGFnIGlmIHRoZSBuYW1lIG9mIHRoaXMgcHJvcGVydHkgaXMgVGFncywgd2hpY2ggaXNcbiAqIHZhbGlkYXRlZCB1c2luZyBgUmVzb3VyY2VUeXBlLmlzVGFnZ2FibGUocmVzb3VyY2UpYC5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGlzVGFnUHJvcGVydHkocHJvcDogUHJvcGVydHkpOiBwcm9wIGlzIFRhZ1Byb3BlcnR5IHtcbiAgcmV0dXJuIChcbiAgICBpc1RhZ1Byb3BlcnR5U3RhbmRhcmQocHJvcCkgfHxcbiAgICBpc1RhZ1Byb3BlcnR5QXV0b1NjYWxpbmdHcm91cChwcm9wKSB8fFxuICAgIGlzVGFnUHJvcGVydHlKc29uKHByb3ApIHx8XG4gICAgaXNUYWdQcm9wZXJ0eVN0cmluZ01hcChwcm9wKVxuICApO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gaXNUYWdQcm9wZXJ0eVN0YW5kYXJkKHByb3A6IFByb3BlcnR5KTogcHJvcCBpcyBUYWdQcm9wZXJ0eVN0YW5kYXJkIHtcbiAgcmV0dXJuIChcbiAgICAocHJvcCBhcyBUYWdQcm9wZXJ0eVN0YW5kYXJkKS5JdGVtVHlwZSA9PT0gJ1RhZycgfHxcbiAgICAocHJvcCBhcyBUYWdQcm9wZXJ0eVN0YW5kYXJkKS5JdGVtVHlwZSA9PT0gJ1RhZ3NFbnRyeScgfHxcbiAgICAocHJvcCBhcyBUYWdQcm9wZXJ0eVN0YW5kYXJkKS5UeXBlID09PSAnVGFncycgfHxcbiAgICAocHJvcCBhcyBUYWdQcm9wZXJ0eVN0YW5kYXJkKS5JdGVtVHlwZSA9PT0gJ1RhZ1JlZicgfHxcbiAgICAocHJvcCBhcyBUYWdQcm9wZXJ0eVN0YW5kYXJkKS5JdGVtVHlwZSA9PT0gJ0VsYXN0aWNGaWxlU3lzdGVtVGFnJyB8fFxuICAgIChwcm9wIGFzIFRhZ1Byb3BlcnR5U3RhbmRhcmQpLkl0ZW1UeXBlID09PSAnSG9zdGVkWm9uZVRhZycgfHxcbiAgICAocHJvcCBhcyBUYWdQcm9wZXJ0eVN0YW5kYXJkKS5JdGVtVHlwZSA9PT0gJ0FjY2Vzc1BvaW50VGFnJ1xuICApO1xuXG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpc1RhZ1Byb3BlcnR5QXV0b1NjYWxpbmdHcm91cChwcm9wOiBQcm9wZXJ0eSk6IHByb3AgaXMgVGFnUHJvcGVydHlBdXRvU2NhbGluZ0dyb3VwIHtcbiAgcmV0dXJuIChwcm9wIGFzIFRhZ1Byb3BlcnR5QXV0b1NjYWxpbmdHcm91cCkuSXRlbVR5cGUgPT09ICdUYWdQcm9wZXJ0eSc7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpc1RhZ1Byb3BlcnR5SnNvbihwcm9wOiBQcm9wZXJ0eSk6IHByb3AgaXMgVGFnUHJvcGVydHlKc29uIHtcbiAgcmV0dXJuIChwcm9wIGFzIFRhZ1Byb3BlcnR5SnNvbikuUHJpbWl0aXZlVHlwZSA9PT0gUHJpbWl0aXZlVHlwZS5Kc29uO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gaXNUYWdQcm9wZXJ0eVN0cmluZ01hcChwcm9wOiBQcm9wZXJ0eSk6IHByb3AgaXMgVGFnUHJvcGVydHlTdHJpbmdNYXAge1xuICByZXR1cm4gKHByb3AgYXMgVGFnUHJvcGVydHlTdHJpbmdNYXApLlByaW1pdGl2ZUl0ZW1UeXBlID09PSAnU3RyaW5nJztcbn1cbiJdfQ==