"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityGroupChanges = void 0;
const chalk = require("chalk");
const security_group_rule_1 = require("./security-group-rule");
const diffable_1 = require("../diffable");
const render_intrinsics_1 = require("../render-intrinsics");
const util_1 = require("../util");
/**
 * Changes to IAM statements
 */
class SecurityGroupChanges {
    constructor(props) {
        this.ingress = new diffable_1.DiffableCollection();
        this.egress = new diffable_1.DiffableCollection();
        // Group rules
        for (const ingressProp of props.ingressRulePropertyChanges) {
            this.ingress.addOld(...this.readInlineRules(ingressProp.oldValue, ingressProp.resourceLogicalId));
            this.ingress.addNew(...this.readInlineRules(ingressProp.newValue, ingressProp.resourceLogicalId));
        }
        for (const egressProp of props.egressRulePropertyChanges) {
            this.egress.addOld(...this.readInlineRules(egressProp.oldValue, egressProp.resourceLogicalId));
            this.egress.addNew(...this.readInlineRules(egressProp.newValue, egressProp.resourceLogicalId));
        }
        // Rule resources
        for (const ingressRes of props.ingressRuleResourceChanges) {
            this.ingress.addOld(...this.readRuleResource(ingressRes.oldProperties));
            this.ingress.addNew(...this.readRuleResource(ingressRes.newProperties));
        }
        for (const egressRes of props.egressRuleResourceChanges) {
            this.egress.addOld(...this.readRuleResource(egressRes.oldProperties));
            this.egress.addNew(...this.readRuleResource(egressRes.newProperties));
        }
        this.ingress.calculateDiff();
        this.egress.calculateDiff();
    }
    get hasChanges() {
        return this.ingress.hasChanges || this.egress.hasChanges;
    }
    /**
     * Return a summary table of changes
     */
    summarize() {
        const ret = [];
        const header = ['', 'Group', 'Dir', 'Protocol', 'Peer'];
        const inWord = 'In';
        const outWord = 'Out';
        // Render a single rule to the table (curried function so we can map it across rules easily--thank you JavaScript!)
        const renderRule = (plusMin, inOut) => (rule) => [
            plusMin,
            rule.groupId,
            inOut,
            rule.describeProtocol(),
            rule.describePeer(),
        ].map(s => plusMin === '+' ? chalk.green(s) : chalk.red(s));
        // First generate all lines, sort later
        ret.push(...this.ingress.additions.map(renderRule('+', inWord)));
        ret.push(...this.egress.additions.map(renderRule('+', outWord)));
        ret.push(...this.ingress.removals.map(renderRule('-', inWord)));
        ret.push(...this.egress.removals.map(renderRule('-', outWord)));
        // Sort by group name then ingress/egress (ingress first)
        ret.sort(util_1.makeComparator((row) => [row[1], row[2].indexOf(inWord) > -1 ? 0 : 1]));
        ret.splice(0, 0, header);
        return ret;
    }
    toJson() {
        return util_1.deepRemoveUndefined({
            ingressRuleAdditions: util_1.dropIfEmpty(this.ingress.additions.map(s => s.toJson())),
            ingressRuleRemovals: util_1.dropIfEmpty(this.ingress.removals.map(s => s.toJson())),
            egressRuleAdditions: util_1.dropIfEmpty(this.egress.additions.map(s => s.toJson())),
            egressRuleRemovals: util_1.dropIfEmpty(this.egress.removals.map(s => s.toJson())),
        });
    }
    get rulesAdded() {
        return this.ingress.hasAdditions
            || this.egress.hasAdditions;
    }
    readInlineRules(rules, logicalId) {
        if (!rules) {
            return [];
        }
        // UnCloudFormation so the parser works in an easier domain
        const ref = '${' + logicalId + '.GroupId}';
        return rules.map((r) => new security_group_rule_1.SecurityGroupRule(render_intrinsics_1.renderIntrinsics(r), ref));
    }
    readRuleResource(resource) {
        if (!resource) {
            return [];
        }
        // UnCloudFormation so the parser works in an easier domain
        return [new security_group_rule_1.SecurityGroupRule(render_intrinsics_1.renderIntrinsics(resource))];
    }
}
exports.SecurityGroupChanges = SecurityGroupChanges;
//# sourceMappingURL=data:application/json;base64,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