# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

**sv2svg** converts structural SystemVerilog modules into readable logic-diagram SVGs using Schemdraw. It parses HDL, assigns gates to levels via topological sorting, and renders them left-to-right (or top-to-bottom) with automatic routing and collision avoidance.

## Key Development Commands

### Environment Setup
```bash
# Create and activate virtual environment
python3 -m venv .venv
source .venv/bin/activate  # macOS/Linux
# .venv\Scripts\activate   # Windows

# Install in editable mode
pip install --upgrade pip
pip install -e .
```

### Running the CLI
```bash
# Basic usage
sv2svg input.sv -o output.svg

# With visual enhancements
sv2svg input.sv --fill-gates --signal-styles --fanout-wires --style vibrant -o output.svg

# Full-featured diagram
sv2svg input.sv --style vibrant --fill-gates --signal-styles --fanout-wires --table --no-caption -o output.svg

# Vertical layout with dark theme
sv2svg input.sv --orientation vertical --style dark -o output.svg

# Output to stdout (useful for pipelines)
sv2svg input.sv -o - > output.svg

# Check version
sv2svg --version
```

**CLI Options**:

*Style & Layout*:
- `--style {classic,blueprint,midnight,mono,vibrant,dark}` — Color scheme (default: classic)
- `--orientation {horizontal,vertical}` — Layout direction (default: horizontal)
- `--grid-x FLOAT` — Horizontal grid snapping (default: 0, disabled by default)
- `--grid-y FLOAT` — Vertical grid snapping (default: 0, disabled by default)
- `--no-symmetry` — Disable symmetric gate placement
- `--input-order {alpha,ports,auto}` — Input port ordering (default: alpha)

*Visualization Enhancements*:
- `--fill-gates` — Enable fill colors for logic gates (subtle pastels by default, vibrant colors with `--style vibrant`)
- `--signal-styles` — Use different line styles (solid=primary I/O, dashed=intermediate signals)
- `--fanout-wires` — Use thicker lines for signals with higher fan-out (1.0→2.0 based on load count)
- `--table` — Include truth table in diagram (max 5 inputs)
- `--no-caption` — Suppress "Module: modulename" caption

### Testing
```bash
# Install test dependencies
pip install -e ".[test]"

# Run all tests (70 tests covering parser, circuit, and CLI)
pytest tests/ -v

# Run with coverage
pytest tests/ --cov=sv2svg --cov-report=html
```

The test suite includes:
- **Parser tests** (22 tests): Expression tokenization, parsing, AST to gates conversion
- **Circuit tests** (21 tests): Module parsing, level assignment, diagram generation
- **CLI tests** (27 tests): Command-line options, file operations, output formats

See `TESTING.md` for detailed testing guide.

### Release Process
- **Commits**: Follow [Conventional Commits](https://www.conventionalcommits.org) (e.g., `feat:`, `fix:`, `docs:`) for semantic versioning
- **Versioning**: Managed by `hatch-vcs` from git tags (dev builds show `0.x.x.dev0+g<hash>`)
- **Automation**: Merge to `main` triggers semantic-release, which updates `CHANGELOG.md`, tags releases, and publishes to PyPI via trusted publishing

## Architecture

### Core Components

**`src/sv2svg/core.py`** — Main logic engine:
- **`SVCircuit`**: Parses SystemVerilog, builds connectivity graph, assigns gate levels, generates Schemdraw drawing
- **Expression Parser**: Full expression parser with tokenizer, AST builder, and gate converter supporting complex expressions with operator precedence and parentheses
- **Parser**: Regex-based extraction of module ports, gate instantiations, and complex `assign` statements
- **Level Assignment**: Topological sort to place gates in left-to-right layers
- **Layout Algorithm**:
  - Barycenter-based gate ordering within each level (reduces wire crossings)
  - Optional symmetry mode: mirrors sibling gates around shared driver centerlines
  - Grid snapping for alignment
- **Routing**: Wire routing with bounding-box collision avoidance (detours around gates)
- **Labeling**: Automatic labeling of intermediate signals (declared as `logic`)
- **Truth Table**: Circuit simulator generates truth tables for verification (max 5 inputs)
- **Vertical Orientation**: SVG post-processing rotates horizontal layout 90° clockwise

**`src/sv2svg/layout.py`** — Layout engine:
- Gate positioning and level assignment
- Barycenter-based reordering for wire crossing minimization
- Routing command generation for optimized wire placement

**`src/sv2svg/cli.py`** — Command-line interface:
- Argument parsing via `argparse`
- Delegates to `SVCircuit.parse_file()` and `SVCircuit.generate_diagram()`
- Stdout support for pipeline integration (`-o -` without creating files)

**`src/sv2svg/__init__.py`** — Package entry point exposing `__version__` and public API

**`src/sv2svg/_version.py`** — Auto-generated by `hatch-vcs` during build (git-based version)

### Data Flow
1. **Parse** SystemVerilog → extract module name, ports, gates, internal signals
2. **Build Connectivity** → map signal drivers/sinks, construct dependency graph
3. **Assign Levels** → topological sort for left-to-right placement
4. **Reorder by Barycenter** → minimize crossings via iterative weighted positioning
5. **Generate Schemdraw Drawing** → place gates, route wires with collision avoidance
6. **Rotate (optional)** → apply SVG transformation for vertical orientation
7. **Output** → write SVG file or stdout

### Parser Capabilities

The parser supports **structural SystemVerilog** (explicit gate instantiations) with full expression parsing for `assign` statements:

#### ✅ Fully Supported
- **Gate instantiations**: `AND u1(a, b, y);`
- **All common logic gates**: AND, OR, NAND, NOR, XOR, XNOR, NOT/INV, BUF/BUFFER
- **Complex assign expressions** with full operator precedence and parentheses:
  - Simple: `assign y = a & b;` → AND gate
  - Complex: `assign y = a | b & c;` → OR(a, AND(b,c)) with proper precedence
  - Parentheses: `assign y = (a | b) & c;` → AND(OR(a,b), c)
  - Multi-operator: `assign y = a & b | c & d;` → OR(AND(a,b), AND(c,d))
  - Negation patterns: `assign y = ~(a & b | c);` → NOR gate with intermediate gates
  - Multi-input: `assign y = a & b & c;` → cascaded AND gates with auto-generated intermediate signals

**Expression Parser** (`core.py:26-303`):
- **Tokenizer**: Converts expressions to tokens (identifiers, operators, parentheses)
- **Parser**: Builds AST with correct operator precedence (NOT > AND > OR/XOR)
- **AST Converter**: Generates gate instances with intermediate signals (`_expr_*`)

#### ❌ Not Supported
- Ternary operators: `assign y = sel ? a : b;`
- Bit operations: `assign y = a[0] & b[1];`
- Arithmetic: `assign y = a + b;`
- For these, use explicit gate instantiations or different HDL constructs

### Routing and Collision Avoidance

The routing algorithm (`core.py:612-925`) uses two key strategies:

1. **Horizontal line avoidance** (`hline_avoid`): Detours around gate bounding boxes
2. **Vertical trunk allocation** (`used_verticals`): Tracks occupied vertical bus lines to prevent overlap

When modifying routing logic:
- **Primary inputs** use a vertical "trunk" that branches to multiple gates
- **Internal signals** use a midpoint vertical with horizontal branches, auto-labeled if declared as `logic` (`core.py:908-913`)
- **Commutative gates** (AND, OR, XOR, etc.) have their inputs auto-sorted by Y-coordinate to minimize crossings
- Grid snapping (`--grid-x`, `--grid-y`) happens **after** coordinate calculation

### Truth Table Generation

When `--table` is specified, the circuit simulator (`core.py:535-606`) generates a truth table:
- **Simulation**: Evaluates circuit for all input combinations (max 2^5 = 32 rows for 5 inputs)
- **Gate evaluation**: Level-ordered evaluation with support for all standard logic gates
- **Display**: Table positioned to the right of circuit with sorted inputs and outputs
- **Limitation**: Only circuits with ≤5 inputs generate tables (larger would be too large)

### Style System

`STYLE_PRESETS` in `core.py:306-348` defines color schemes with font size tuning and optional gate fill colors:

**Available Styles**:
- `classic` — Dark blue-gray (default)
- `blueprint` — NASA blue for technical documentation
- `midnight` — Cyan on dark background
- `mono` — Grayscale for print-friendly output
- `vibrant` — Bright gate colors (red, blue, orange, purple, teal) with `--fill-gates`
- `dark` — Light-on-dark theme for dark mode documentation

Each preset configures:
- **Base settings**: `color`, `lw` (line width), `fontsize` (10pt for labels)
- **Module label color**: Title text color
- **Gate label fontsize**: Smaller font (9pt) for gate names to fit better inside gates
- **Gate fills** (vibrant only): Color-coded gate fills (AND=red, OR=blue, XOR=orange, NOT=purple, BUF=teal)
- **Background** (dark only): Dark background color for light-on-dark theme

**Font Sizes**:
- Module labels: 10pt
- Port labels: 10pt
- Gate labels: 9pt (configurable per style)
- Wire labels (intermediate signals): 8pt
- Truth table: 9pt headers, 8pt values

### Visualization Features

**Gate Fill Colors** (`--fill-gates`):
- Default fills: Subtle pastel colors (light blue, alice blue, light yellow, lavender, mint)
- Vibrant fills: Bright colors from `vibrant` style preset
- Implementation: `_add_gate()` method applies `fill()` to gate elements based on gate type
- Location: `core.py:1077-1206`

**Signal Line Styles** (`--signal-styles`):
- **Solid lines** (`-`): Primary inputs and outputs
- **Dashed lines** (`--`): Intermediate signals (declared as `logic`)
- Implementation: `get_line_style()` helper function determines style based on signal type
- Applied to all `elm.Line()` calls via `ls` parameter
- Location: `core.py:746-754`

**Fan-out Wire Thickness** (`--fanout-wires`):
- **1.0**: Single load (1 gate)
- **1.3**: Dual load (2 gates)
- **1.6**: Moderate fan-out (3-4 gates)
- **2.0**: High fan-out (5+ gates)
- Implementation: `get_line_width()` calculates thickness based on signal fan-out count
- Applied to all `elm.Line()` calls via `lw` parameter
- Location: `core.py:756-770`

**Caption Control** (`--no-caption`):
- Suppresses "Module: modulename" label when enabled
- Location: `core.py:689-695`

## Common Development Patterns

### Adding a New Gate Type
1. Update `_add_gate()` in `core.py:1077-1206` to handle the new gate type string
2. Map to appropriate Schemdraw `logic.*` class or fallback to `elm.Box`
3. Apply gate label fontsize parameter for consistent styling
4. Use `maybe_fill()` helper to support `--fill-gates` feature
5. If adding assign statement support, update the expression parser's gate type mapping

### Adding a New Style Preset
1. Add entry to `STYLE_PRESETS` dictionary in `core.py:306-348`
2. Configure `config` dict with `color`, `lw`, `fontsize`
3. Set `module_label_color` and `gate_label_fontsize`
4. Optionally add `gate_fills` dict for `--fill-gates` integration
5. Optionally add `background` color for dark themes
6. Update `available_styles()` list in CLI help (`cli.py:17`)

### Adding Visualization Features
1. **Add CLI flag**: Add argument to `cli.py` parser (lines 25-30)
2. **Update signature**: Add parameter to `generate_diagram()` in `core.py` (line 689)
3. **Pass parameter**: Update both stdout and file generation calls in `cli.py` (lines 47-67)
4. **Implement feature**: Add logic in `generate_diagram()` method
5. **Apply styling**: Use helper functions or modify element creation
6. **Test**: Create test cases and verify with existing fixtures

**Example pattern** (signal styling):
- Helper function: `get_line_style(signal)` returns style based on signal type
- Apply to lines: `elm.Line(**line_kwargs)` where `line_kwargs = {'ls': style, 'lw': width}`
- Thread signal name through routing functions for context-aware styling

### Modifying Layout Algorithm
- **Level assignment**: See `layout.py:LayoutEngine.assign_levels()`
- **Barycenter ordering**: See `layout.py:LayoutEngine.reorder_by_barycenter()`
- **Symmetry logic**: See `generate_diagram()` symmetry block (`core.py:~817-860`)

### Debugging Routing Issues
- Inspect `bboxes` list for gate collision regions
- Add temporary `print()` statements in `hline_avoid()` or `vline_avoid()`
- Use `--grid-x 0 --grid-y 0` to disable snapping and see raw coordinates
- Check intermediate signal labels to verify connectivity
- Verify line styling with `--signal-styles` and `--fanout-wires` to visualize signal paths

## Project Constraints

- **Python 3.9+** required (`pyproject.toml:14`)
- **Runtime dependency**: `schemdraw>=0.16`
- **Test dependencies**: `pytest>=7.0`, `pytest-cov>=4.0` (install with `pip install -e ".[test]"`)
- **No linting/formatting tools configured** — maintain existing code style (4-space indents, snake_case)
- **Expression parser** — full tokenizer/parser/AST for complex expressions, but limited to boolean logic
- **CI workflow** — runs full test suite (70 tests) on push/PR

## Version Management

- **Source of truth**: Git tags via `hatch-vcs`
- **Fallback version**: `0.0.0` (when `.git` not present)
- **Dev builds**: Auto-suffixed with `.dev0+g<hash>.d<date>`
- **Version file**: Auto-generated at `src/sv2svg/_version.py` during build (gitignored)

Do not manually edit `_version.py` or hardcode versions elsewhere.