"""
Wrapper Builder
Constructs the wrapper module content
"""
import re
from typing import List
from .parser import HwifSignal
from .template_generator import generate_flat_assignments


class WrapperBuilder:
    """Builds the wrapper module content"""

    def __init__(
        self,
        module_name: str,
        package_name: str,
        inst_name: str,
        module_content: str,
        input_signals: List[HwifSignal],
        output_signals: List[HwifSignal],
    ):
        self.module_name = module_name
        self.package_name = package_name
        self.inst_name = inst_name
        self.module_content = module_content
        self.input_signals = input_signals
        self.output_signals = output_signals

        # Extract info from module
        self._parse_module()

    def _parse_module(self):
        """Parse module content to extract key information"""
        # Check for parameters
        param_match = re.search(
            r"module\s+\w+\s*#\s*\((.*?)\)\s*\(", self.module_content, re.DOTALL
        )
        self.has_params = param_match is not None
        self.params = []
        if self.has_params:
            param_text = param_match.group(1).strip()  # type: ignore[union-attr]
            for param_line in param_text.split("\n"):
                param_line = param_line.strip().rstrip(",")
                if param_line:
                    self.params.append(param_line)

        # Extract all non-hwif ports from the module
        self.non_hwif_ports = self._extract_non_hwif_ports()

    def _extract_non_hwif_ports(self) -> List[str]:
        """Extract all ports except hwif_in and hwif_out"""
        ports: List[str] = []

        # Find port section
        if self.has_params:
            match = re.search(
                r"module\s+\w+\s*#\s*\([^)]+\)\s*\((.*?)\);",
                self.module_content,
                re.DOTALL,
            )
        else:
            match = re.search(
                r"module\s+\w+\s*\((.*?)\);", self.module_content, re.DOTALL
            )

        if not match:
            return ports

        port_section = match.group(1)

        # Split by comma, but be careful of commas inside brackets
        for line in port_section.split("\n"):
            line = line.strip()
            if not line or line.startswith("//"):
                continue

            # Remove trailing comma
            line = line.rstrip(",")

            # Skip hwif ports (they contain ::)
            if "::" in line and ("hwif_in" in line or "hwif_out" in line):
                continue

            ports.append(line)

        return ports

    def generate(self) -> str:
        """Generate the complete wrapper module content"""
        lines = []

        # Header
        lines.append("// Generated by PeakRDL-etana HWIF Wrapper Tool")
        lines.append("//  https://github.com/daxzio/PeakRDL-etana")
        lines.append("//")
        lines.append(
            "// This is a wrapper module that flattens the hwif structs into individual signals"
        )
        lines.append("")

        # Module declaration
        lines.append(self._generate_module_declaration())
        lines.append("")

        # Internal struct signals
        lines.append(
            "    //--------------------------------------------------------------------------"
        )
        lines.append("    // Internal hwif struct signals")
        lines.append(
            "    //--------------------------------------------------------------------------"
        )

        if self.input_signals:
            lines.append(f"    {self.package_name}::{self.inst_name}__in_t hwif_in;")
        if self.output_signals:
            lines.append(f"    {self.package_name}::{self.inst_name}__out_t hwif_out;")
        lines.append("")

        # Assignments
        lines.append(
            "    //--------------------------------------------------------------------------"
        )
        lines.append("    // Flatten struct to individual signals")
        lines.append(
            "    //--------------------------------------------------------------------------"
        )
        lines.append(generate_flat_assignments(self.input_signals, is_input=True))
        if self.output_signals:
            lines.append("")
            lines.append(generate_flat_assignments(self.output_signals, is_input=False))
        lines.append("")

        # Module instantiation
        lines.append(
            "    //--------------------------------------------------------------------------"
        )
        lines.append("    // Instantiate the main regblock module")
        lines.append(
            "    //--------------------------------------------------------------------------"
        )
        lines.append(self._generate_instance())
        lines.append("")
        lines.append("endmodule")
        lines.append("")

        return "\n".join(lines)

    def _generate_module_declaration(self) -> str:
        """Generate the module declaration with ports"""
        lines = []

        if self.has_params:
            lines.append(f"module {self.module_name}_wrapper #(")
            lines.append("        " + ",\n        ".join(self.params))
            lines.append("    ) (")
        else:
            lines.append(f"module {self.module_name}_wrapper (")

        # Build port list
        ports: List[str] = []

        # Copy all non-hwif ports from original module
        for port in self.non_hwif_ports:
            ports.append(f"        {port}")

        # Add flat hwif ports if they don't conflict
        for sig in self.input_signals:
            ports.append(f"        {sig.get_port_declaration()}")
        for sig in self.output_signals:
            ports.append(f"        {sig.get_port_declaration()}")

        lines.append(",\n".join(ports))
        lines.append("    );")

        return "\n".join(lines)

    def _generate_instance(self) -> str:
        """Generate module instantiation"""
        lines = []

        if self.has_params:
            lines.append(f"    {self.module_name} #(")
            # Convert parameter declarations to instantiation syntax
            param_insts = []
            for param_decl in self.params:
                # Extract parameter name from declaration like "parameter ID_WIDTH = 1"
                param_match = re.search(r"parameter\s+(\w+)\s*=", param_decl)
                if param_match:
                    param_name = param_match.group(1)
                    param_insts.append(f".{param_name}({param_name})")
                else:
                    # Fallback: use the parameter declaration as-is (shouldn't happen)
                    param_insts.append(param_decl)
            lines.append("        " + ",\n        ".join(param_insts))
            lines.append(f"    ) i_{self.module_name} (")
        else:
            lines.append(f"    {self.module_name} i_{self.module_name} (")

        # Build port connections
        ports: List[str] = []

        # Connect all non-hwif ports (pass-through)
        for port_decl in self.non_hwif_ports:
            # Extract port name, handling edge case where bit width has no space
            # e.g., "input wire [3:0]s_axil_wstrb" -> "s_axil_wstrb"
            port_match = re.search(r"\[[\d:]+\](\w+)|\b(\w+)$", port_decl)
            if port_match:
                port_name = (
                    port_match.group(1) if port_match.group(1) else port_match.group(2)
                )
            else:
                port_name = port_decl.split()[-1]
            ports.append(f"        .{port_name}({port_name})")

        # Connect hwif ports to internal structs
        if self.input_signals:
            ports.append("        .hwif_in(hwif_in)")
        if self.output_signals:
            ports.append("        .hwif_out(hwif_out)")

        lines.append(",\n".join(ports))
        lines.append("    );")

        return "\n".join(lines)
