"""
VHDL Wrapper Builder
Constructs the wrapper entity content
"""
import re
from typing import List, Tuple, Optional
from .vhdl_parser import VhdlPackageParser


class VhdlWrapperBuilder:
    """Builds the VHDL wrapper entity content"""

    def __init__(
        self,
        module_name: str,
        package_name: str,
        entity_content: str,
        parser: VhdlPackageParser,
        in_record_type: Optional[str],
        out_record_type: Optional[str],
    ):
        self.module_name = module_name
        self.package_name = package_name
        self.entity_content = entity_content
        self.parser = parser
        self.in_record_type = in_record_type
        self.out_record_type = out_record_type

        # Flatten the records
        self.in_signals = []
        self.out_signals = []

        if in_record_type:
            self.in_signals = parser.flatten_record(
                in_record_type, "hwif_in", "hwif_in"
            )
        if out_record_type:
            self.out_signals = parser.flatten_record(
                out_record_type, "hwif_out", "hwif_out"
            )

        # Post-process signals to concatenate arrays of simple types
        self.in_signals = self._concatenate_array_signals(self.in_signals)
        self.out_signals = self._concatenate_array_signals(self.out_signals)

        # Extract non-hwif ports
        self.non_hwif_ports = self._extract_non_hwif_ports()

        # Detect required additional packages
        self.required_packages = self._detect_required_packages()

    def _detect_required_packages(self) -> List[str]:
        """Detect which additional IEEE packages are needed based on signal types"""
        required = []

        # Collect all VHDL types from signals
        all_types = []
        for signal_name, vhdl_type, direction, record_path in self.in_signals:
            all_types.append(vhdl_type)
        for signal_name, vhdl_type, direction, record_path in self.out_signals:
            all_types.append(vhdl_type)

        # Check for ufixed or sfixed (require fixed_pkg)
        for vhdl_type in all_types:
            if "ufixed" in vhdl_type or "sfixed" in vhdl_type:
                if "ieee.fixed_pkg.all" not in required:
                    required.append("ieee.fixed_pkg.all")
                break

        return required

    def _extract_non_hwif_ports(self) -> List[str]:
        """Extract all ports except hwif_in and hwif_out"""
        ports = []

        # Find the port section in the entity - need to handle nested parens
        # Find 'port (' and then manually find the matching ')'
        port_start_match = re.search(
            r"entity\s+\w+\s+is\s+port\s*\(",
            self.entity_content,
            re.DOTALL | re.IGNORECASE,
        )

        if not port_start_match:
            return ports

        # Start after 'port ('
        start_pos = port_start_match.end()
        paren_depth = 1
        end_pos = start_pos

        # Find the matching closing parenthesis
        for i in range(start_pos, len(self.entity_content)):
            if self.entity_content[i] == "(":
                paren_depth += 1
            elif self.entity_content[i] == ")":
                paren_depth -= 1
                if paren_depth == 0:
                    end_pos = i
                    break

        if end_pos == start_pos:
            return ports

        port_section = self.entity_content[start_pos:end_pos]

        # Split by semicolon, keeping track of nested parentheses
        current_port = []
        paren_depth = 0

        for char in port_section:
            if char == "(":
                paren_depth += 1
                current_port.append(char)
            elif char == ")":
                paren_depth -= 1
                current_port.append(char)
            elif char == ";" and paren_depth == 0:
                # Found a port delimiter at the top level
                port_line = "".join(current_port).strip()
                if port_line and not port_line.startswith("--"):
                    # Skip hwif ports
                    if "hwif_in" not in port_line and "hwif_out" not in port_line:
                        ports.append(port_line)
                current_port = []
            else:
                current_port.append(char)

        # Handle last port (might not have trailing semicolon)
        if current_port:
            port_line = "".join(current_port).strip()
            if port_line and not port_line.startswith("--"):
                if "hwif_in" not in port_line and "hwif_out" not in port_line:
                    ports.append(port_line)

        return ports

    def generate(self) -> str:
        """Generate the complete wrapper entity content"""
        lines = []

        # Header
        lines.append("-- Generated by PeakRDL-regblock-vhdl HWIF Wrapper Tool")
        lines.append("--  https://github.com/SystemRDL/PeakRDL-regblock-vhdl")
        lines.append("--")
        lines.append(
            "-- This is a wrapper entity that flattens the hwif records into individual signals"
        )
        lines.append("")

        # Library and use clauses
        lines.append("library ieee;")
        lines.append("use ieee.std_logic_1164.all;")
        lines.append("use ieee.numeric_std.all;")
        # Add any additional required packages (e.g., fixed_pkg)
        for pkg in self.required_packages:
            lines.append(f"use {pkg};")
        lines.append("")
        lines.append(f"use work.{self.package_name}.all;")
        lines.append("")

        # Entity declaration
        lines.append(self._generate_entity_declaration())
        lines.append("")

        # Architecture
        lines.append(f"architecture wrapper of {self.module_name}_wrapper is")
        lines.append("")

        # Internal record signals
        lines.append(
            "    ----------------------------------------------------------------------------"
        )
        lines.append("    -- Internal hwif record signals")
        lines.append(
            "    ----------------------------------------------------------------------------"
        )

        if self.in_signals:
            lines.append(f"    signal hwif_in : {self.in_record_type};")
        if self.out_signals:
            lines.append(f"    signal hwif_out : {self.out_record_type};")
        lines.append("")

        lines.append("begin")
        lines.append("")

        # Flatten assignments
        lines.append(
            "    ----------------------------------------------------------------------------"
        )
        lines.append("    -- Flatten record to individual signals")
        lines.append(
            "    ----------------------------------------------------------------------------"
        )

        if self.in_signals:
            lines.extend(self._generate_input_assignments())
            lines.append("")

        if self.out_signals:
            lines.extend(self._generate_output_assignments())
            lines.append("")

        # Entity instantiation
        lines.append(
            "    ----------------------------------------------------------------------------"
        )
        lines.append("    -- Instantiate the main entity")
        lines.append(
            "    ----------------------------------------------------------------------------"
        )
        lines.append(self._generate_instance())
        lines.append("")

        lines.append(f"end architecture wrapper;")
        lines.append("")

        return "\n".join(lines)

    def _generate_entity_declaration(self) -> str:
        """Generate the entity declaration with flattened ports"""
        lines = []
        lines.append(f"entity {self.module_name}_wrapper is")

        # Build port list
        ports = []

        # Copy all non-hwif ports
        for port in self.non_hwif_ports:
            ports.append(f"        {port};")

        # Add flattened hwif ports
        for signal_name, vhdl_type, direction, record_path in self.in_signals:
            port_decl = f"        {signal_name} : in {vhdl_type};"
            ports.append(port_decl)

        for signal_name, vhdl_type, direction, record_path in self.out_signals:
            port_decl = f"        {signal_name} : out {vhdl_type};"
            ports.append(port_decl)

        # If we have ports, add port declaration
        if ports:
            # Remove semicolon from last port
            ports[-1] = ports[-1].rstrip(";")
            lines.append("    port (")
            lines.extend(ports)
            lines.append("    );")

        lines.append(f"end entity {self.module_name}_wrapper;")

        return "\n".join(lines)

    def _generate_input_assignments(self) -> List[str]:
        """Generate assignments from flattened inputs to record"""
        lines = []

        if not hasattr(self, "_array_signal_mappings"):
            self._array_signal_mappings = {}

        for signal_name, vhdl_type, direction, record_path in self.in_signals:
            if signal_name in self._array_signal_mappings:
                # This is a concatenated array signal - need to slice and assign to individual elements
                mapping = self._array_signal_mappings[signal_name]
                items = mapping["items"]
                elem_width = mapping["elem_width"]

                # Generate assignments for each array element
                # Need to slice the concatenated signal in reverse order (highest index first)
                for idx, orig_sig_name, _, _, orig_record_path in items:
                    # Reverse the index for slicing (highest index is MSB)
                    rev_idx = len(items) - 1 - idx
                    if elem_width == 1:
                        # Single bit - use direct indexing, not slice
                        lines.append(
                            f"    {orig_record_path} <= {signal_name}({rev_idx});"
                        )
                    else:
                        slice_high = (rev_idx + 1) * elem_width - 1
                        slice_low = rev_idx * elem_width
                        lines.append(
                            f"    {orig_record_path} <= {signal_name}({slice_high} downto {slice_low});"
                        )
            else:
                # Regular assignment: hwif_in.path.to.field <= signal_name;
                lines.append(f"    {record_path} <= {signal_name};")

        return lines

    def _generate_output_assignments(self) -> List[str]:
        """Generate assignments from record to flattened outputs"""
        lines = []

        if not hasattr(self, "_array_signal_mappings"):
            self._array_signal_mappings = {}

        for signal_name, vhdl_type, direction, record_path in self.out_signals:
            if signal_name in self._array_signal_mappings:
                # This is a concatenated array signal - need to build concatenation
                mapping = self._array_signal_mappings[signal_name]
                items = mapping["items"]
                elem_width = mapping["elem_width"]

                # Build concatenation: highest index first (MSB)
                # e.g., x(63).x.value & x(62).x.value & ... & x(1).x.value & x(0).x.value
                concat_parts = []
                for idx, orig_sig_name, _, _, orig_record_path in reversed(items):
                    # Just concatenate the record paths directly (each is already the right width)
                    concat_parts.append(orig_record_path)

                # Concatenate all parts
                assignment = f"    {signal_name} <= " + " & ".join(concat_parts) + ";"
                lines.append(assignment)
            else:
                # Regular assignment: signal_name <= hwif_out.path.to.field;
                lines.append(f"    {signal_name} <= {record_path};")

        return lines

    def _concatenate_array_signals(
        self, signals: List[Tuple[str, str, str, str]]
    ) -> List[Tuple[str, str, str, str]]:
        """
        Concatenate array signals into single vectors for cocotb compatibility.
        Detects signals with numeric indices (e.g., hwif_out_x_0_x, hwif_out_x_1_x)
        and concatenates them into a single vector (e.g., hwif_out_x).
        """
        import re
        from collections import defaultdict

        # Pattern to match signals with numeric indices: name_<idx>_field
        array_pattern = re.compile(r"^(.+)_(\d+)_(.+)$")

        # Group signals by base name (without index)
        array_groups = defaultdict(list)
        simple_signals = []
        # Track base_name usage to detect when suffix can be dropped
        base_name_groups = defaultdict(list)

        for signal_name, vhdl_type, direction, record_path in signals:
            match = array_pattern.match(signal_name)
            if match:
                base_name = match.group(1)
                idx = int(match.group(2))
                suffix = match.group(3)
                key = (base_name, suffix, vhdl_type)
                array_groups[key].append(
                    (idx, signal_name, vhdl_type, direction, record_path)
                )
                base_name_groups[base_name].append(key)
            else:
                simple_signals.append((signal_name, vhdl_type, direction, record_path))

        result = list(simple_signals)

        # Process each array group
        for (base_name, suffix, vhdl_type), items in array_groups.items():
            # Sort by index
            items.sort(key=lambda x: x[0])
            indices = [x[0] for x in items]

            # Check if indices are consecutive starting from 0
            if indices == list(range(len(indices))):
                # Extract width from vhdl_type
                # For std_logic_vector: "std_logic_vector(31 downto 0)" -> 32
                # For std_logic: "std_logic" -> 1
                width_match = re.search(r"\((\d+)\s+downto\s+(\d+)\)", vhdl_type)
                if width_match:
                    high = int(width_match.group(1))
                    low = int(width_match.group(2))
                    elem_width = high - low + 1
                    total_width = elem_width * len(indices)
                elif "std_logic" in vhdl_type and "vector" not in vhdl_type:
                    # Single std_logic - width is 1
                    elem_width = 1
                    total_width = len(indices)
                else:
                    # Can't determine width, keep individual signals
                    result.extend([(x[1], x[2], x[3], x[4]) for x in items])
                    continue

                # Create concatenated signal name
                # If there are multiple groups with same base_name but different suffixes, use base_name_suffix
                # If there's only one group for this base_name, we can drop the suffix (like test_pipelined_cpuif)
                # This matches SystemVerilog wrapper behavior which removes redundant names (e.g., x_x -> x)
                groups_for_base = base_name_groups[base_name]
                # Count unique suffixes for this base_name
                unique_suffixes = set(key[1] for key in groups_for_base)
                if len(unique_suffixes) > 1:
                    # Multiple fields in the array - include suffix to avoid collisions
                    concat_name = f"{base_name}_{suffix}"
                else:
                    # Single field - check if suffix is redundant (matches last part of base_name)
                    # e.g., hwif_out_x with suffix x -> hwif_out_x (not hwif_out_x_x)
                    if base_name.endswith(f"_{suffix}"):
                        # Redundant suffix - drop it (matches SystemVerilog wrapper behavior)
                        concat_name = base_name
                    else:
                        # Suffix is different - keep it
                        concat_name = f"{base_name}_{suffix}"
                concat_type = f"std_logic_vector({total_width - 1} downto 0)"

                # Store original signals for assignment generation
                concat_record_path = (
                    items[0][4].rsplit(f"({items[0][0]})", 1)[0]
                    if f"({items[0][0]})" in items[0][4]
                    else items[0][4]
                )

                # Add concatenated signal
                result.append(
                    (concat_name, concat_type, items[0][3], concat_record_path)
                )

                # Store mapping for assignment generation (we'll handle this in assignment methods)
                if not hasattr(self, "_array_signal_mappings"):
                    self._array_signal_mappings = {}
                self._array_signal_mappings[concat_name] = {
                    "items": items,
                    "elem_width": elem_width,
                    "count": len(indices),
                }
            else:
                # Not consecutive, keep individual signals
                result.extend([(x[1], x[2], x[3], x[4]) for x in items])

        return result

    def _generate_instance(self) -> str:
        """Generate entity instantiation"""
        lines = []
        lines.append(f"    i_{self.module_name} : entity work.{self.module_name}")
        lines.append("        port map (")

        # Build port connections
        port_connections = []

        # Connect all non-hwif ports (pass-through)
        for port_decl in self.non_hwif_ports:
            # Extract port name from declaration
            # Format: <name> : in/out <type>
            # The name is the first word before the colon
            colon_pos = port_decl.find(":")
            if colon_pos > 0:
                port_name = port_decl[:colon_pos].strip()
                port_connections.append(f"            {port_name} => {port_name},")

        # Connect hwif ports to internal records
        if self.in_signals:
            port_connections.append("            hwif_in => hwif_in,")
        if self.out_signals:
            port_connections.append("            hwif_out => hwif_out,")

        # Remove comma from last connection
        if port_connections:
            port_connections[-1] = port_connections[-1].rstrip(",")

        lines.extend(port_connections)
        lines.append("        );")

        return "\n".join(lines)
