"""
🚀 Writable Batch Mode 使用示例

展示如何使用零内存开销的writable_batch_mode进行高性能数组操作
"""
import random
import time
import numpy as np
from numpack import NumPack


def example_basic_usage():
    """示例1: 基本用法"""
    print("=" * 60)
    print("Example 1: Basic Usage")
    print("=" * 60)
    
    # 创建NumPack文件
    npk = NumPack('example_writable.npk', drop_if_exists=True)
    npk.open()
    
    # 保存一些数组
    arrays = {
        'array1': np.random.rand(1, 1000000),
        'array2': np.random.rand(1, 1000000),
        'array3': np.random.rand(1, 1000000),
    }
    npk.save(arrays)
    
    # 使用writable_batch_mode进行批量修改
    start = time.perf_counter()
    with npk.writable_batch_mode() as wb:
        # 加载数组（返回mmap视图，零拷贝）
        arr1 = wb.load('array1')
        arr1 *= 2.0  # 直接在文件上修改
        
        arr2 = wb.load('array2')
        arr2 += 1.0
        
        arr3 = wb.load('array3')
        arr3 /= 2.0
        
        # save是可选的（保持API一致性）
        wb.save({'array1': arr1, 'array2': arr2, 'array3': arr3})
    
    elapsed = time.perf_counter() - start
    print(f"\n✅ Modifying 3 arrays took: {elapsed * 1000:.2f} ms")
    print(f"💾 Memory overhead: Nearly 0 MB (virtual memory only)")
    
    # 验证修改已持久化
    result = npk.load('array1', lazy=False)
    print(f"📊 Verification: array1 mean = {result.mean():.4f}")
    
    npk.close()
    print()


def example_high_performance_loop():
    """示例2: 高性能循环操作"""
    print("=" * 60)
    print("Example 2: High-performance loop operations (original test case)")
    print("=" * 60)
    
    # 创建测试数据
    npk = NumPack('example_loop.npk', drop_if_exists=True)
    npk.open()
    
    arrays = {
        'a1': np.random.rand(1, 1000000),
        'a2': np.random.rand(1, 1000000),
        'a3': np.random.rand(1, 1000000),
    }
    npk.save(arrays)
    
    # 高性能循环
    foo = ['a1', 'a2', 'a3']
    random.seed(42)
    
    start = time.perf_counter()
    with npk.writable_batch_mode() as wb:
        for i in range(100):
            c = random.choice(foo)
            a = wb.load(c)    # mmap视图（零拷贝）
            a *= 4.1          # 直接在文件上修改
            wb.save({c: a})   # 可选
    
    elapsed = time.perf_counter() - start
    
    print(f"\n✅ 100 random operations took: {elapsed * 1000:.2f} ms")
    print(f"📈 Average per operation: {elapsed * 10:.3f} ms")
    print(f"🎯 Performance target: < 100 ms (< 1 ms/op)")
    
    if elapsed * 1000 <= 100:
        print(f"🎉 Target met! Speedup ~18-20x")
    
    npk.close()
    print()


def example_large_array():
    """示例3: 超大数组（内存装不下）"""
    print("=" * 60)
    print("Example 3: Very large array scenario")
    print("=" * 60)
    
    # 创建大数组（~80MB每个）
    npk = NumPack('example_large.npk', drop_if_exists=True)
    npk.open()
    
    print("Creating large arrays (each ~80MB)...")
    large_arrays = {
        'big1': np.random.rand(1, 10000000).astype(np.float64),
        'big2': np.random.rand(1, 10000000).astype(np.float64),
        'big3': np.random.rand(1, 10000000).astype(np.float64),
    }
    npk.save(large_arrays)
    
    print("Processing with writable_batch_mode...")
    start = time.perf_counter()
    with npk.writable_batch_mode() as wb:
        for name in ['big1', 'big2', 'big3']:
            arr = wb.load(name)
            arr *= 1.5  # 直接在文件上修改
            wb.save({name: arr})
    
    elapsed = time.perf_counter() - start
    
    print(f"\n✅ Processing ~240MB data took: {elapsed * 1000:.2f} ms")
    print(f"💾 Memory overhead: Nearly 0 MB")
    print(f"🌟 Advantage: Supports TB-scale data (disk limited, not memory)")
    
    npk.close()
    print()


def example_comparison():
    """示例4: batch_mode vs writable_batch_mode对比"""
    print("=" * 60)
    print("Example 4: Performance comparison")
    print("=" * 60)
    
    # 准备数据
    npk = NumPack('example_compare.npk', drop_if_exists=True)
    npk.open()
    
    test_array = np.random.rand(1, 1000000)
    npk.save({'test': test_array})
    
    # 测试batch_mode
    print("\nTesting batch_mode (memory cache)...")
    start = time.perf_counter()
    with npk.batch_mode():
        for i in range(50):
            a = npk.load('test')
            a *= 1.1
            npk.save({'test': a})
    batch_time = time.perf_counter() - start
    
    # 恢复数据
    npk.save({'test': test_array})
    
    # 测试writable_batch_mode
    print("Testing writable_batch_mode (zero memory)...")
    start = time.perf_counter()
    with npk.writable_batch_mode() as wb:
        for i in range(50):
            a = wb.load('test')
            a *= 1.1
            wb.save({'test': a})
    writable_time = time.perf_counter() - start
    
    print("\n" + "=" * 60)
    print("📊 Performance comparison results:")
    print("=" * 60)
    print(f"{'Mode':<25} {'Time(ms)':<12} {'Memory overhead'}")
    print("-" * 60)
    print(f"{'batch_mode':<25} {batch_time*1000:<12.2f} ~8 MB")
    print(f"{'writable_batch_mode':<25} {writable_time*1000:<12.2f} ~0 MB")
    print("=" * 60)
    print("\n✅ Conclusion:")
    print("  • batch_mode: Small arrays, pursue extreme speed")
    print("  • writable_batch_mode: Large arrays, zero memory overhead")
    
    npk.close()
    print()


def example_best_practices():
    """示例5: 最佳实践"""
    print("=" * 60)
    print("Example 5: Best practices")
    print("=" * 60)
    
    npk = NumPack('example_practices.npk', drop_if_exists=True)
    npk.open()
    
    # 创建测试数据
    npk.save({
        'data1': np.random.rand(100, 1000),
        'data2': np.random.rand(100, 1000),
    })
    
    print("\n✅ Recommended practices:")
    print()
    
    # 1. 使用context manager
    print("1. Always use context manager:")
    print("```python")
    print("with npk.writable_batch_mode() as wb:")
    print("    arr = wb.load('data')")
    print("    arr *= 2.0")
    print("    # Auto flush on exit")
    print("```")
    print()
    
    # 2. 缓存array引用
    print("2. Cache frequently accessed arrays:")
    with npk.writable_batch_mode() as wb:
        # 第一次load会创建mmap
        arr1 = wb.load('data1')
        arr2 = wb.load('data2')
        
        # 后续直接使用缓存的引用
        for i in range(10):
            arr1 *= 1.1
            arr2 += 0.1
    print("✅ Avoid repeatedly loading the same array")
    print()
    
    # 3. 异常处理
    print("3. Exception handling (automatic):")
    try:
        with npk.writable_batch_mode() as wb:
            arr = wb.load('data1')
            arr *= 2.0
            # 即使抛出异常，也会自动flush和清理
    except Exception as e:
        pass
    print("✅ Context manager automatically cleans up resources")
    print()
    
    npk.close()


if __name__ == '__main__':
    print("\n" + "🚀" * 30)
    print("Writable Batch Mode - Zero memory overhead high performance solution")
    print("🚀" * 30 + "\n")
    
    # 运行所有示例
    example_basic_usage()
    example_high_performance_loop()
    example_large_array()
    example_comparison()
    example_best_practices()
    
    print("\n" + "=" * 60)
    print("🎉 All examples completed!")
    print("=" * 60)
    print("\n💡 Usage recommendations:")
    print("  • Small arrays (< 100MB): Use batch_mode()")
    print("  • Large arrays (> 100MB): Use writable_batch_mode()")
    print("  • Memory-constrained environments: Always use writable_batch_mode()")
    print("  • TB-scale data: writable_batch_mode is the only choice")
    print()

