//! NumPack SIMD单元测试
//! 
//! 验证SIMD实现的正确性和性能

#[cfg(test)]
mod tests {
    use super::*;
    use crate::metadata::DataType;

    #[test]
    fn test_cpu_capabilities_detection() {
        let caps = CpuCapabilities::detect();
        
        // 验证CPU能力检测
        println!("CPU Capabilities:");
        println!("  SSE2: {}", caps.sse2);
        println!("  AVX2: {}", caps.avx2);
        println!("  AVX512F: {}", caps.avx512f);
        println!("  NEON: {}", caps.neon);
        println!("  Cache Line Size: {}", caps.cache_line_size);
        println!("  Preferred Vector Width: {}", caps.preferred_vector_width);
        
        // 基本检查
        assert!(caps.cache_line_size > 0);
        assert!(caps.preferred_vector_width > 0);
    }

    #[test]
    fn test_simd_strategy_selection() {
        let simd = NumPackSIMD::new();
        
        // 测试不同数据类型的策略选择
        let strategies = vec![
            (DataType::Float32, 1024),
            (DataType::Int64, 2048),
            (DataType::Uint8, 512),
            (DataType::Bool, 128),
            (DataType::Complex64, 256),
        ];
        
        for (dtype, size) in strategies {
            let strategy = simd.get_optimal_strategy(dtype, size);
            println!("DataType: {:?}, Size: {}, Strategy: {:?}", dtype, size, strategy);
            
            // 验证策略不为空
            assert!(matches!(strategy, SIMDStrategy::Scalar | 
                SIMDStrategy::AVX512Bool | SIMDStrategy::AVX2Bool | SIMDStrategy::PackedBool |
                SIMDStrategy::AVX512Byte | SIMDStrategy::AVX2Byte | SIMDStrategy::SSE2Byte | SIMDStrategy::NEONByte |
                SIMDStrategy::AVX512Word | SIMDStrategy::AVX2Word | SIMDStrategy::SSE2Word | SIMDStrategy::NEONWord |
                SIMDStrategy::AVX512DWord | SIMDStrategy::AVX2DWord | SIMDStrategy::SSE2DWord | SIMDStrategy::NEONDWord |
                SIMDStrategy::AVX512QWord | SIMDStrategy::AVX2QWord | SIMDStrategy::SSE2QWord | SIMDStrategy::NEONQWord |
                SIMDStrategy::AVX2Complex64 | SIMDStrategy::SSE2Complex64 |
                SIMDStrategy::AVX2Complex128 | SIMDStrategy::SSE2Complex128));
        }
    }

    #[test]
    fn test_scalar_row_copy() {
        let simd = NumPackSIMD::new();
        
        // 创建测试数据
        let src_data = (0..1024u8).cycle().take(4096).collect::<Vec<_>>();
        let indices = vec![0, 2, 4, 6, 8];
        let row_size = 256;
        let mut dst = vec![0u8; indices.len() * row_size];
        
        // 测试标量行拷贝
        let result = simd.scalar_copy_rows(&src_data, &mut dst, &indices, row_size);
        assert!(result.is_ok());
        
        // 验证拷贝结果
        for (i, &idx) in indices.iter().enumerate() {
            let src_offset = idx * row_size;
            let dst_offset = i * row_size;
            
            for j in 0..row_size {
                assert_eq!(dst[dst_offset + j], src_data[src_offset + j]);
            }
        }
    }

    #[test]
    fn test_access_pattern_detection() {
        let pattern_simd = AccessPatternSIMD::new();
        
        // 测试不同的访问模式检测
        let test_cases = vec![
            (vec![100], AccessPattern::SingleRandom),
            (vec![0, 1, 2, 3, 4], AccessPattern::Sequential),
            (vec![0, 10, 20, 30, 40], AccessPattern::Strided),
            ((0..100).step_by(17).collect::<Vec<_>>(), AccessPattern::BatchRandom),
        ];
        
        for (indices, expected_pattern) in test_cases {
            let detected = pattern_simd.detect_access_pattern(&indices, 1024);
            println!("Indices: {:?} -> Detected: {:?}, Expected: {:?}", 
                     &indices[..std::cmp::min(5, indices.len())], detected, expected_pattern);
            
            // 注：由于检测算法的复杂性，这里只验证函数能正常运行
            assert!(matches!(detected, AccessPattern::SingleRandom | AccessPattern::BatchRandom | 
                AccessPattern::Sequential | AccessPattern::Strided | 
                AccessPattern::Clustered | AccessPattern::Streaming));
        }
    }

    #[test]
    fn test_data_type_conversion() {
        let simd = NumPackSIMD::new();
        
        // 创建i32测试数据
        let src_i32 = vec![1i32, 2, 3, 4, 5, 6, 7, 8];
        let src_bytes = unsafe {
            std::slice::from_raw_parts(src_i32.as_ptr() as *const u8, src_i32.len() * 4)
        };
        
        // 转换为f32
        let mut dst_bytes = vec![0u8; src_i32.len() * 4];
        let result = simd.scalar_convert(src_bytes, &mut dst_bytes, DataType::Int32, DataType::Float32);
        assert!(result.is_ok());
        
        // 验证转换结果
        let dst_f32 = unsafe {
            std::slice::from_raw_parts(dst_bytes.as_ptr() as *const f32, src_i32.len())
        };
        
        for i in 0..src_i32.len() {
            assert_eq!(dst_f32[i], src_i32[i] as f32);
        }
    }

    #[test]
    fn test_simd_error_handling() {
        let simd = NumPackSIMD::new();
        
        // 测试索引越界错误
        let src_data = vec![0u8; 100];
        let mut dst = vec![0u8; 200];
        let bad_indices = vec![0, 1, 50]; // 最后一个索引会导致越界
        let row_size = 50;
        
        let result = simd.scalar_copy_rows(&src_data, &mut dst, &bad_indices, row_size);
        assert!(result.is_err());
        assert!(matches!(result.unwrap_err(), SIMDError::IndexOutOfBounds));
    }

    #[test]
    fn test_benchmark_basic_functionality() {
        let mut benchmark = SIMDBenchmark::new();
        
        // 测试基本功能
        let test_data = benchmark.generate_test_data(DataType::Float32, 1024);
        assert_eq!(test_data.len(), 1024 * 4); // f32 = 4 bytes
        
        let random_indices = benchmark.generate_random_indices(1000, 10);
        assert_eq!(random_indices.len(), 10);
        
        // 验证所有索引都在范围内
        for &idx in &random_indices {
            assert!(idx < 1000);
        }
    }

    #[test]
    fn test_simd_vs_scalar_correctness() {
        let simd = NumPackSIMD::new();
        
        // 创建测试数据
        let src_data = (0..2048u8).collect::<Vec<_>>();
        let indices = vec![0, 4, 8, 12, 16];
        let row_size = 64;
        
        // SIMD和标量版本的结果应该相同
        let mut simd_dst = vec![0u8; indices.len() * row_size];
        let mut scalar_dst = vec![0u8; indices.len() * row_size];
        
        let simd_result = simd.copy_rows(&src_data, &mut simd_dst, &indices, row_size, DataType::Uint8);
        let scalar_result = simd.scalar_copy_rows(&src_data, &mut scalar_dst, &indices, row_size);
        
        assert!(simd_result.is_ok());
        assert!(scalar_result.is_ok());
        
        // 验证结果一致性
        assert_eq!(simd_dst, scalar_dst);
    }

    #[test]
    fn test_performance_record_and_cleanup() {
        let mut pattern_simd = AccessPatternSIMD::new();
        
        // 记录一些性能数据
        pattern_simd.record_performance(
            AccessPattern::BatchRandom,
            DataType::Float32,
            1024,
            SIMDStrategy::AVX2DWord,
            std::time::Duration::from_millis(10),
            100.0
        );
        
        // 验证性能记录功能不会崩溃
        let stats = pattern_simd.get_performance_stats(AccessPattern::BatchRandom);
        // stats可能为None，这是正常的
        
        // 测试清理功能
        pattern_simd.cleanup_expired_records(std::time::Duration::from_secs(1));
        // 清理功能应该正常运行
    }
} 