"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolve = void 0;
const util_1 = require("util");
function isResolvable(obj) {
    return obj.toJSON !== undefined;
}
function resolve(value, options = {}) {
    const args = options.args ?? [];
    const omitEmpty = options.omitEmpty ?? false;
    if (value == null) {
        return value;
    }
    if (isResolvable(value)) {
        const resolved = value.toJSON();
        return resolve(resolved, options);
    }
    // Special resolution for few JavaScript built-in types
    // that by default would be stringified as empty objects ('{}')
    // as they are missing a `toJSON` implementation.
    switch (true) {
        case util_1.types.isRegExp(value):
            if (value.flags)
                throw new Error("RegExp with flags should be explicitly converted to a string");
            return value.source;
        case util_1.types.isSet(value):
            return resolve(Array.from(value), options);
        case util_1.types.isMap(value):
            return resolve(Object.fromEntries(value), options);
        case util_1.types.isBoxedPrimitive(value):
            return resolve(value.valueOf(), options);
    }
    // if value is a function, call it and resolve the result.
    if (typeof value === "function") {
        const resolved = value.apply(undefined, args);
        return resolve(resolved, options);
    }
    if (typeof value !== "object") {
        return value;
    }
    if (Array.isArray(value)) {
        if (omitEmpty && value.length === 0) {
            return undefined;
        }
        return value.map((x) => resolve(x, options)).filter((x) => x != null); // filter undefined/null/omitted
    }
    // only allow data types (i.e. objects without constructors)
    if (value.constructor && value.constructor?.name !== "Object") {
        throw new Error(`only data types can be resolved. trying to resolve object of type ${value.constructor?.name}`);
    }
    const result = {};
    for (const [k, v] of Object.entries(value)) {
        const resolved = resolve(v, options);
        // skip undefined values
        if (resolved === undefined) {
            continue;
        }
        result[k] = resolved;
    }
    if (omitEmpty && Object.keys(result).length === 0) {
        return undefined;
    }
    return result;
}
exports.resolve = resolve;
//# sourceMappingURL=data:application/json;base64,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