"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCdkTypeScriptApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const auto_discover_1 = require("./auto-discover");
const awscdk_deps_js_1 = require("./awscdk-deps-js");
const cdk_config_1 = require("./cdk-config");
const cdk_tasks_1 = require("./cdk-tasks");
const integ_runner_1 = require("./integ-runner");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const javascript_1 = require("../javascript");
const typescript_1 = require("../typescript");
/**
 * AWS CDK app in TypeScript
 *
 * @pjid awscdk-app-ts
 */
class AwsCdkTypeScriptApp extends typescript_1.TypeScriptAppProject {
    /**
     * The CDK version this app is using.
     */
    get cdkVersion() {
        return this.cdkDeps.cdkVersion;
    }
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            bundlerOptions: {
                ...options.bundlerOptions,
                // we invoke the "bundle" task as part of the build step in cdk.json so
                // we don't want it to be added to the pre-compile phase.
                runBundleTask: javascript_1.RunBundleTask.MANUAL,
            },
        });
        this.cdkDeps = new awscdk_deps_js_1.AwsCdkDepsJs(this, {
            dependencyType: dependencies_1.DependencyType.RUNTIME,
            ...options,
        });
        this.appEntrypoint = options.appEntrypoint ?? "main.ts";
        // CLI
        this.addDevDeps(`aws-cdk@${this.cdkDeps.cdkVersion}`);
        // no compile step because we do all of it in typescript directly
        this.compileTask.reset();
        this.cdkTasks = new cdk_tasks_1.CdkTasks(this);
        // add synth to the build
        this.postCompileTask.spawn(this.cdkTasks.synthSilent);
        const tsConfigFile = this.tsconfig?.fileName;
        if (!tsConfigFile) {
            throw new Error("Expecting tsconfig.json");
        }
        this.cdkConfig = new cdk_config_1.CdkConfig(this, {
            app: `npx ts-node -P ${tsConfigFile} --prefer-ts-exts ${path.posix.join(this.srcdir, this.appEntrypoint)}`,
            featureFlags: this.cdkDeps.cdkMajorVersion < 2,
            buildCommand: this.runTaskCommand(this.bundler.bundleTask),
            watchIncludes: [`${this.srcdir}/**/*.ts`, `${this.testdir}/**/*.ts`],
            watchExcludes: [
                "README.md",
                "cdk*.json",
                "**/*.d.ts",
                "**/*.js",
                "tsconfig.json",
                "package*.json",
                "yarn.lock",
                "node_modules",
            ],
            ...options,
        });
        this.gitignore.exclude(".parcel-cache/");
        this.npmignore?.exclude(`${this.cdkConfig.cdkout}/`);
        this.npmignore?.exclude(".cdk.staging/");
        if (this.tsconfig) {
            this.tsconfig.exclude.push(this.cdkConfig.cdkout);
        }
        this.addDevDeps("ts-node");
        if (options.sampleCode ?? true) {
            new SampleCode(this, this.cdkDeps.cdkMajorVersion);
        }
        new auto_discover_1.AutoDiscover(this, {
            srcdir: this.srcdir,
            testdir: this.testdir,
            lambdaOptions: options.lambdaOptions,
            tsconfigPath: this.tsconfigDev.fileName,
            cdkDeps: this.cdkDeps,
            lambdaAutoDiscover: options.lambdaAutoDiscover ?? true,
            edgeLambdaAutoDiscover: options.edgeLambdaAutoDiscover ?? true,
            lambdaExtensionAutoDiscover: options.lambdaExtensionAutoDiscover ?? true,
            integrationTestAutoDiscover: options.integrationTestAutoDiscover ?? true,
        });
        if (options.experimentalIntegRunner) {
            new integ_runner_1.IntegRunner(this);
        }
    }
    /**
     * Adds an AWS CDK module dependencies
     * @param modules The list of modules to depend on
     */
    addCdkDependency(...modules) {
        return this.cdkDeps.addV1Dependencies(...modules);
    }
}
exports.AwsCdkTypeScriptApp = AwsCdkTypeScriptApp;
_a = JSII_RTTI_SYMBOL_1;
AwsCdkTypeScriptApp[_a] = { fqn: "projen.awscdk.AwsCdkTypeScriptApp", version: "0.79.27" };
class SampleCode extends component_1.Component {
    constructor(project, cdkMajorVersion) {
        super(project);
        this.cdkMajorVersion = cdkMajorVersion;
        this.appProject = project;
    }
    synthesize() {
        const outdir = this.project.outdir;
        const srcdir = path.join(outdir, this.appProject.srcdir);
        if (fs.existsSync(srcdir) &&
            fs.readdirSync(srcdir).filter((x) => x.endsWith(".ts"))) {
            return;
        }
        const srcImports = new Array();
        if (this.cdkMajorVersion < 2) {
            srcImports.push("import { App, Construct, Stack, StackProps } from '@aws-cdk/core';");
        }
        else {
            srcImports.push("import { App, Stack, StackProps } from 'aws-cdk-lib';");
            srcImports.push("import { Construct } from 'constructs';");
        }
        const srcCode = `${srcImports.join("\n")}

export class MyStack extends Stack {
  constructor(scope: Construct, id: string, props: StackProps = {}) {
    super(scope, id, props);

    // define resources here...
  }
}

// for development, use account/region from cdk cli
const devEnv = {
  account: process.env.CDK_DEFAULT_ACCOUNT,
  region: process.env.CDK_DEFAULT_REGION,
};

const app = new App();

new MyStack(app, '${this.project.name}-dev', { env: devEnv });
// new MyStack(app, '${this.project.name}-prod', { env: prodEnv });

app.synth();`;
        fs.mkdirSync(srcdir, { recursive: true });
        fs.writeFileSync(path.join(srcdir, this.appProject.appEntrypoint), srcCode);
        const testdir = path.join(outdir, this.appProject.testdir);
        if (fs.existsSync(testdir) &&
            fs.readdirSync(testdir).filter((x) => x.endsWith(".ts"))) {
            return;
        }
        const testImports = new Array();
        if (this.cdkMajorVersion < 2) {
            testImports.push("import { App } from '@aws-cdk/core';");
            testImports.push("import { Template } from '@aws-cdk/assertions';");
        }
        else {
            testImports.push("import { App } from 'aws-cdk-lib';");
            testImports.push("import { Template } from 'aws-cdk-lib/assertions';");
        }
        const appEntrypointName = path.basename(this.appProject.appEntrypoint, ".ts");
        const testCode = `${testImports.join("\n")}
import { MyStack } from '../${this.appProject.srcdir}/${appEntrypointName}';

test('Snapshot', () => {
  const app = new App();
  const stack = new MyStack(app, 'test');

  const template = Template.fromStack(stack);
  expect(template.toJSON()).toMatchSnapshot();
});`;
        fs.mkdirSync(testdir, { recursive: true });
        fs.writeFileSync(path.join(testdir, `${appEntrypointName}.test.ts`), testCode);
    }
}
//# sourceMappingURL=data:application/json;base64,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