"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaExtension = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const case_1 = require("case");
const internal_1 = require("./internal");
const lambda_function_1 = require("./lambda-function");
const component_1 = require("../component");
const javascript_1 = require("../javascript");
const source_code_1 = require("../source-code");
/**
 * Create a Lambda Extension
 */
class LambdaExtension extends component_1.Component {
    constructor(project, options) {
        super(project);
        const basePath = (0, path_1.join)((0, path_1.dirname)(options.entrypoint), (0, path_1.basename)(options.entrypoint, internal_1.TYPESCRIPT_LAMBDA_EXTENSION_EXT));
        const name = options.name ?? (0, path_1.basename)(basePath);
        const bundler = javascript_1.Bundler.of(project);
        if (!bundler) {
            throw new Error("No bundler found. Please add a Bundler component to your project.");
        }
        const compatibleRuntimes = options.compatibleRuntimes ?? [
            lambda_function_1.LambdaRuntime.NODEJS_18_X,
            lambda_function_1.LambdaRuntime.NODEJS_16_X,
            lambda_function_1.LambdaRuntime.NODEJS_14_X,
            lambda_function_1.LambdaRuntime.NODEJS_12_X,
        ];
        if (compatibleRuntimes.length === 0) {
            throw new Error("Compatible runtimes must include at least one runtime");
        }
        // Use the lowest runtime version to bundle
        const [bundlerRuntime] = compatibleRuntimes.sort((a, b) => a.functionRuntime.localeCompare(b.functionRuntime));
        // Allow extension code to import dev-deps since they are only needed
        // during bundling
        const eslint = javascript_1.Eslint.of(project);
        eslint?.allowDevDeps(options.entrypoint);
        const bundle = bundler.addBundle(options.entrypoint, {
            platform: bundlerRuntime.esbuildPlatform,
            target: bundlerRuntime.esbuildTarget,
            externals: bundlerRuntime.defaultExternals,
            outfile: `extensions/${name}`,
            // Make the output executable because Lambda expects to run
            // extensions as stand-alone programs alongside the main lambda
            // process.
            executable: true,
            ...options.bundlingOptions,
        });
        const constructFile = options.constructFile ?? `${basePath}-layer-version.ts`;
        new LambdaLayerConstruct(project, {
            constructFile: constructFile,
            constructName: options.constructName,
            assetDir: bundle.outdir,
            compatibleRuntimes: compatibleRuntimes,
            description: `Provides a Lambda Extension \`${name}\` from ${(0, internal_1.convertToPosixPath)(options.entrypoint)}`,
            cdkDeps: options.cdkDeps,
        });
    }
}
exports.LambdaExtension = LambdaExtension;
_a = JSII_RTTI_SYMBOL_1;
LambdaExtension[_a] = { fqn: "projen.awscdk.LambdaExtension", version: "0.79.27" };
class LambdaLayerConstruct extends source_code_1.SourceCode {
    constructor(project, options) {
        super(project, options.constructFile);
        const src = this;
        const cdkDeps = options.cdkDeps;
        const constructName = options.constructName ?? (0, case_1.pascal)((0, path_1.basename)(options.constructFile, ".ts"));
        const propsType = `${constructName}Props`;
        const assetDir = (0, path_1.relative)((0, path_1.dirname)(options.constructFile), options.assetDir);
        if (src.marker) {
            src.line(`// ${src.marker}`);
        }
        src.line("import * as path from 'path';");
        if (cdkDeps.cdkMajorVersion === 1) {
            src.line("import * as lambda from '@aws-cdk/aws-lambda';");
            src.line("import { Construct } from '@aws-cdk/core';");
            cdkDeps.addV1Dependencies("@aws-cdk/aws-lambda");
            cdkDeps.addV1Dependencies("@aws-cdk/core");
        }
        else {
            src.line("import * as lambda from 'aws-cdk-lib/aws-lambda';");
            src.line("import { Construct } from 'constructs';");
        }
        src.line();
        src.line("/**");
        src.line(` * Props for ${constructName}`);
        src.line(" */");
        src.open(`export interface ${propsType} extends lambda.LayerVersionOptions {`);
        src.close("}");
        src.line();
        src.line("/**");
        src.line(` * ${options.description}`);
        src.line(" */");
        src.open(`export class ${constructName} extends lambda.LayerVersion {`);
        src.open(`constructor(scope: Construct, id: string, props?: ${propsType}) {`);
        src.open("super(scope, id, {");
        src.line(`description: ${encodeCodeString(options.description)},`);
        src.line("...props,");
        src.open("compatibleRuntimes: [");
        for (const runtime of options.compatibleRuntimes) {
            src.line(`new lambda.Runtime('${runtime.functionRuntime}', lambda.RuntimeFamily.NODEJS),`);
        }
        src.close("],");
        src.open(`code: lambda.Code.fromAsset(path.join(__dirname,`);
        src.line(`${encodeCodeString((0, internal_1.convertToPosixPath)(assetDir))})),`);
        src.close();
        src.close("});");
        src.close("}");
        src.close("}");
    }
}
/**
 * Encodes a string for embedding in source code.
 */
function encodeCodeString(value) {
    const json = JSON.stringify(value);
    const escapedString = json.substring(1, json.length - 1).replace(/'/g, "\\'");
    return `'${escapedString}'`;
}
//# sourceMappingURL=data:application/json;base64,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