"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuildWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflow_actions_1 = require("../github/workflow-actions");
const workflows_model_1 = require("../github/workflows-model");
const javascript_1 = require("../javascript");
const runner_options_1 = require("../runner-options");
const PULL_REQUEST_REF = "${{ github.event.pull_request.head.ref }}";
const PULL_REQUEST_REPOSITORY = "${{ github.event.pull_request.head.repo.full_name }}";
const BUILD_JOBID = "build";
const SELF_MUTATION_STEP = "self_mutation";
const SELF_MUTATION_HAPPENED_OUTPUT = "self_mutation_happened";
const IS_FORK = "github.event.pull_request.head.repo.full_name != github.repository";
const NOT_FORK = `!(${IS_FORK})`;
const SELF_MUTATION_CONDITION = `needs.${BUILD_JOBID}.outputs.${SELF_MUTATION_HAPPENED_OUTPUT}`;
class BuildWorkflow extends component_1.Component {
    constructor(project, options) {
        super(project);
        this._postBuildJobs = [];
        const github = github_1.GitHub.of(project);
        if (!github) {
            throw new Error("BuildWorkflow is currently only supported for GitHub projects");
        }
        this.github = github;
        this.preBuildSteps = options.preBuildSteps ?? [];
        this.postBuildSteps = options.postBuildSteps ?? [];
        this.gitIdentity = options.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.buildTask = options.buildTask;
        this.artifactsDirectory = options.artifactsDirectory;
        this.name = options.name ?? "build";
        const mutableBuilds = options.mutableBuild ?? true;
        this.workflow = new github_1.GithubWorkflow(github, this.name);
        this.workflow.on(options.workflowTriggers ?? {
            pullRequest: {},
            workflowDispatch: {}, // allow manual triggering
        });
        this.addBuildJob(options);
        if (mutableBuilds) {
            this.addSelfMutationJob(options);
        }
        if (project instanceof javascript_1.NodeProject) {
            project.addPackageIgnore(constants_1.PERMISSION_BACKUP_FILE);
        }
    }
    addBuildJob(options) {
        const jobConfig = {
            ...(0, runner_options_1.filteredRunsOnOptions)(options.runsOn, options.runsOnGroup),
            container: options.containerImage
                ? { image: options.containerImage }
                : undefined,
            env: {
                CI: "true",
                ...options.env,
            },
            permissions: {
                contents: workflows_model_1.JobPermission.WRITE,
                ...options.permissions,
            },
            steps: (() => this.renderBuildSteps()),
            outputs: {
                [SELF_MUTATION_HAPPENED_OUTPUT]: {
                    stepId: SELF_MUTATION_STEP,
                    outputName: SELF_MUTATION_HAPPENED_OUTPUT,
                },
            },
        };
        this.workflow.addJob(BUILD_JOBID, jobConfig);
    }
    /**
     * Returns a list of job IDs that are part of the build.
     */
    get buildJobIds() {
        return [BUILD_JOBID, ...this._postBuildJobs];
    }
    /**
     * Adds steps that are executed after the build.
     * @param steps The job steps
     */
    addPostBuildSteps(...steps) {
        this.postBuildSteps.push(...steps);
    }
    /**
     * Adds another job to the build workflow which is executed after the build
     * job succeeded.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param id The id of the new job
     * @param job The job specification
     */
    addPostBuildJob(id, job) {
        const steps = [];
        if (this.artifactsDirectory) {
            steps.push({
                name: "Download build artifacts",
                uses: "actions/download-artifact@v4",
                with: {
                    name: constants_1.BUILD_ARTIFACT_NAME,
                    path: this.artifactsDirectory,
                },
            }, {
                name: "Restore build artifact permissions",
                continueOnError: true,
                run: [
                    `cd ${this.artifactsDirectory} && setfacl --restore=${constants_1.PERMISSION_BACKUP_FILE}`,
                ].join("\n"),
            });
        }
        steps.push(...(job.steps ?? []));
        this.workflow.addJob(id, {
            needs: [BUILD_JOBID],
            // only run if build did not self-mutate
            if: `! ${SELF_MUTATION_CONDITION}`,
            ...job,
            steps: steps,
        });
        // add to the list of build job IDs
        this._postBuildJobs.push(id);
    }
    /**
     * Run a task as a job within the build workflow which is executed after the
     * build job succeeded.
     *
     * The job will have access to build artifacts and will install project
     * dependencies in order to be able to run any commands used in the tasks.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param options Specify tools and other options
     */
    addPostBuildJobTask(task, options = {}) {
        this.addPostBuildJobCommands(`post-build-${task.name}`, [`${this.project.projenCommand} ${task.name}`], {
            checkoutRepo: true,
            installDeps: true,
            tools: options.tools,
            ...(0, runner_options_1.filteredRunsOnOptions)(options.runsOn, options.runsOnGroup),
        });
    }
    /**
     * Run a sequence of commands as a job within the build workflow which is
     * executed after the build job succeeded.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param options Specify tools and other options
     */
    addPostBuildJobCommands(id, commands, options) {
        const steps = [];
        if (options?.checkoutRepo) {
            steps.push(github_1.WorkflowSteps.checkout({
                with: {
                    ref: PULL_REQUEST_REF,
                    repository: PULL_REQUEST_REPOSITORY,
                    ...(this.github.downloadLfs ? { lfs: true } : {}),
                },
            }));
        }
        if (options?.checkoutRepo &&
            options?.installDeps &&
            this.project instanceof javascript_1.NodeProject) {
            steps.push({
                name: "Install dependencies",
                run: `${this.project.package.installCommand}`,
            });
        }
        steps.push({ run: commands.join("\n") });
        this.addPostBuildJob(id, {
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
            },
            tools: options?.tools,
            ...(0, runner_options_1.filteredRunsOnOptions)(options?.runsOn, options?.runsOnGroup),
            steps,
        });
    }
    addSelfMutationJob(options) {
        this.workflow.addJob("self-mutation", {
            ...(0, runner_options_1.filteredRunsOnOptions)(options.runsOn, options.runsOnGroup),
            permissions: {
                contents: workflows_model_1.JobPermission.WRITE,
            },
            needs: [BUILD_JOBID],
            if: `always() && ${SELF_MUTATION_CONDITION} && ${NOT_FORK}`,
            steps: [
                ...this.workflow.projenCredentials.setupSteps,
                ...workflow_actions_1.WorkflowActions.checkoutWithPatch({
                    // we need to use a PAT so that our push will trigger the build workflow
                    token: this.workflow.projenCredentials.tokenRef,
                    ref: PULL_REQUEST_REF,
                    repository: PULL_REQUEST_REPOSITORY,
                    lfs: this.github.downloadLfs,
                }),
                github_1.WorkflowSteps.setupGitIdentity({ gitIdentity: this.gitIdentity }),
                {
                    name: "Push changes",
                    env: {
                        PULL_REQUEST_REF,
                    },
                    run: [
                        "git add .",
                        'git commit -s -m "chore: self mutation"',
                        `git push origin HEAD:$PULL_REQUEST_REF`,
                    ].join("\n"),
                },
            ],
        });
    }
    /**
     * Called (lazily) during synth to render the build job steps.
     */
    renderBuildSteps() {
        return [
            github_1.WorkflowSteps.checkout({
                with: {
                    ref: PULL_REQUEST_REF,
                    repository: PULL_REQUEST_REPOSITORY,
                    ...(this.github.downloadLfs ? { lfs: true } : {}),
                },
            }),
            ...this.preBuildSteps,
            {
                name: this.buildTask.name,
                run: this.github.project.runTaskCommand(this.buildTask),
            },
            ...this.postBuildSteps,
            // check for mutations and upload a git patch file as an artifact
            ...workflow_actions_1.WorkflowActions.uploadGitPatch({
                stepId: SELF_MUTATION_STEP,
                outputName: SELF_MUTATION_HAPPENED_OUTPUT,
                mutationError: "Files were changed during build (see build log). If this was triggered from a fork, you will need to update your branch.",
            }),
            // upload the build artifact only if we have post-build jobs (otherwise, there's no point)
            ...(this._postBuildJobs.length == 0
                ? []
                : [
                    {
                        name: "Backup artifact permissions",
                        continueOnError: true,
                        run: `cd ${this.artifactsDirectory} && getfacl -R . > ${constants_1.PERMISSION_BACKUP_FILE}`,
                    },
                    github_1.WorkflowSteps.uploadArtifact({
                        with: {
                            name: constants_1.BUILD_ARTIFACT_NAME,
                            path: this.artifactsDirectory,
                        },
                    }),
                ]),
        ];
    }
}
exports.BuildWorkflow = BuildWorkflow;
_a = JSII_RTTI_SYMBOL_1;
BuildWorkflow[_a] = { fqn: "projen.build.BuildWorkflow", version: "0.79.27" };
//# sourceMappingURL=data:application/json;base64,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