"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsiiProject = exports.Stability = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("node:path");
const semver_1 = require("semver");
const consts_1 = require("./consts");
const jsii_docgen_1 = require("./jsii-docgen");
const javascript_1 = require("../javascript");
const runner_options_1 = require("../runner-options");
const typescript_1 = require("../typescript");
const util_1 = require("../util");
const EMAIL_REGEX = /(?:[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*|"(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21\x23-\x5b\x5d-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])*")@(?:(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?|\[(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?|[a-z0-9-]*[a-z0-9]:(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21-\x5a\x53-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])+)\])/;
const URL_REGEX = /((([A-Za-z]{3,9}:(?:\/\/)?)(?:[\-;:&=\+\$,\w]+@)?[A-Za-z0-9\.\-]+|(?:www\.|[\-;:&=\+\$,\w]+@)[A-Za-z0-9\.\-]+)((?:\/[\+~%\/\.\w\-_]*)?\??(?:[\-\+=&;%@\.\w_]*)#?(?:[\.\!\/\\\w]*))?)/;
const REPO_TEMP_DIRECTORY = ".repo";
var Stability;
(function (Stability) {
    Stability["EXPERIMENTAL"] = "experimental";
    Stability["STABLE"] = "stable";
    Stability["DEPRECATED"] = "deprecated";
})(Stability || (exports.Stability = Stability = {}));
/**
 * Multi-language jsii library project
 *
 * @pjid jsii
 */
class JsiiProject extends typescript_1.TypeScriptProject {
    // This project copies to whole repo into the artifactsDirectory.
    // Which causes the release tag file to be in a path like ./dist/dist/releasetag.txt
    // Use a TS hack to allow the release component to get the correct path from the project
    // @ts-ignore
    get releaseTagFilePath() {
        return path.posix.normalize(path.posix.join(this.artifactsDirectory, this.artifactsDirectory));
    }
    constructor(options) {
        const { authorEmail, authorUrl } = parseAuthorAddress(options);
        // True if jsii version 1.x is compatible with the requested version range.
        const usesLegacyJsii = options.jsiiVersion == null ||
            (options.jsiiVersion !== "*" &&
                new semver_1.Range(options.jsiiVersion).intersects(new semver_1.Range("1.x")));
        const defaultOptions = {
            repository: options.repositoryUrl,
            authorName: options.author,
            authorEmail,
            authorUrl,
            jestOptions: usesLegacyJsii ? { jestVersion: "^27" } : undefined,
        };
        const forcedOptions = {
            releaseToNpm: false,
            disableTsconfig: true,
            docgen: false, // we use jsii-docgen here so disable typescript docgen
        };
        const mergedOptions = (0, util_1.deepMerge)([
            defaultOptions,
            options,
            forcedOptions,
        ]);
        super(mergedOptions);
        const srcdir = this.srcdir;
        const libdir = this.libdir;
        this.addFields({ types: `${libdir}/index.d.ts` });
        const compressAssembly = options.compressAssembly ?? false;
        // this is an unhelpful warning
        const jsiiFlags = ["--silence-warnings=reserved-word"];
        if (compressAssembly) {
            jsiiFlags.push("--compress-assembly");
        }
        const compatIgnore = options.compatIgnore ?? ".compatignore";
        this.addFields({ stability: options.stability ?? Stability.STABLE });
        if (options.stability === Stability.DEPRECATED) {
            this.addFields({ deprecated: true });
        }
        const compatTask = this.addTask("compat", {
            description: "Perform API compatibility check against latest version",
            exec: `jsii-diff npm:$(node -p "require(\'./package.json\').name") -k --ignore-file ${compatIgnore} || (echo "\nUNEXPECTED BREAKING CHANGES: add keys such as \'removed:constructs.Node.of\' to ${compatIgnore} to skip.\n" && exit 1)`,
        });
        const compat = options.compat ?? false;
        if (compat) {
            this.compileTask.spawn(compatTask);
        }
        this.compileTask.reset(["jsii", ...jsiiFlags].join(" "));
        this.watchTask.reset(["jsii", "-w", ...jsiiFlags].join(" "));
        this.packageAllTask = this.addTask("package-all", {
            description: "Packages artifacts for all target languages",
        });
        // in jsii we consider the entire repo (post build) as the build artifact
        // which is then used to create the language bindings in separate jobs.
        const prepareRepoForCI = [
            `rsync -a . .repo --exclude .git --exclude node_modules`,
            `rm -rf ${this.artifactsDirectory}`,
            `mv .repo ${this.artifactsDirectory}`,
        ].join(" && ");
        // when running inside CI we just prepare the repo for packaging, which
        // takes place in separate tasks.
        // outside of CI (i.e locally) we simply package all targets.
        this.packageTask.reset(`if [ ! -z \${CI} ]; then ${prepareRepoForCI}; else ${this.runTaskCommand(this.packageAllTask)}; fi`);
        const targets = {};
        const jsii = {
            outdir: this.artifactsDirectory,
            targets,
            tsc: {
                outDir: libdir,
                rootDir: srcdir,
            },
        };
        if (options.excludeTypescript) {
            jsii.excludeTypescript = options.excludeTypescript;
        }
        this.addFields({ jsii });
        this.release?.publisher.addGitHubPrePublishingSteps({
            name: "Prepare Repository",
            run: `mv ${this.artifactsDirectory} ${REPO_TEMP_DIRECTORY}`,
        }, {
            name: "Collect GitHub Metadata",
            run: `mv ${REPO_TEMP_DIRECTORY}/${this.artifactsDirectory} ${this.artifactsDirectory}`,
        });
        const extraJobOptions = {
            ...this.getJobRunsOnConfig(options),
            ...(options.workflowContainerImage
                ? { container: { image: options.workflowContainerImage } }
                : {}),
        };
        if (options.releaseToNpm != false) {
            const task = this.addPackagingTask("js");
            this.release?.publisher.publishToNpm({
                ...this.pacmakForLanguage("js", task),
                registry: this.package.npmRegistry,
                npmTokenSecret: this.package.npmTokenSecret,
                npmProvenance: this.package.npmProvenance,
                codeArtifactOptions: options.codeArtifactOptions,
            });
            this.addPackagingTarget("js", task, extraJobOptions);
        }
        // we cannot call an option `java` because the java code generated by jsii
        // does not compile due to a conflict between this option name and the `java`
        // package (e.g. when `java.util.Objects` is referenced).
        if ("java" in options) {
            throw new Error('the "java" option is now called "publishToMaven"');
        }
        if (options.publishToMaven) {
            targets.java = {
                package: options.publishToMaven.javaPackage,
                maven: {
                    groupId: options.publishToMaven.mavenGroupId,
                    artifactId: options.publishToMaven.mavenArtifactId,
                },
            };
            const task = this.addPackagingTask("java");
            this.release?.publisher.publishToMaven({
                ...this.pacmakForLanguage("java", task),
                ...options.publishToMaven,
            });
            this.addPackagingTarget("java", task, extraJobOptions);
        }
        const pypi = options.publishToPypi ?? options.python;
        if (pypi) {
            targets.python = {
                distName: pypi.distName,
                module: pypi.module,
            };
            const task = this.addPackagingTask("python");
            this.release?.publisher.publishToPyPi({
                ...this.pacmakForLanguage("python", task),
                ...pypi,
            });
            this.addPackagingTarget("python", task, extraJobOptions);
        }
        const nuget = options.publishToNuget ?? options.dotnet;
        if (nuget) {
            targets.dotnet = {
                namespace: nuget.dotNetNamespace,
                packageId: nuget.packageId,
                iconUrl: nuget.iconUrl,
            };
            const task = this.addPackagingTask("dotnet");
            this.release?.publisher.publishToNuget({
                ...this.pacmakForLanguage("dotnet", task),
                ...nuget,
            });
            this.addPackagingTarget("dotnet", task, extraJobOptions);
        }
        const golang = options.publishToGo;
        if (golang) {
            targets.go = {
                moduleName: golang.moduleName,
                packageName: golang.packageName,
            };
            const task = this.addPackagingTask("go");
            this.release?.publisher.publishToGo({
                ...this.pacmakForLanguage("go", task),
                ...golang,
            });
            this.addPackagingTarget("go", task, extraJobOptions);
        }
        const jsiiSuffix = options.jsiiVersion === "*"
            ? // If jsiiVersion is "*", don't specify anything so the user can manage.
                ""
            : // Otherwise, use `jsiiVersion` or fall back to `1.x`.
                `@${options.jsiiVersion ?? "1.x"}`;
        this.addDevDeps(`jsii${jsiiSuffix}`, `jsii-rosetta${jsiiSuffix}`, "jsii-diff", "jsii-pacmak");
        this.gitignore.exclude(".jsii", "tsconfig.json");
        this.npmignore?.include(".jsii");
        if (options.docgen ?? true) {
            new jsii_docgen_1.JsiiDocgen(this, { filePath: options.docgenFilePath });
        }
        // jsii updates .npmignore, so we make it writable
        if (this.npmignore) {
            this.npmignore.readonly = false;
        }
        // When using jsii@1.x, we need to add some resolutions to avoid including
        // TypeScript-3.9-incompatible dependencies that break the compiler.
        if (usesLegacyJsii) {
            // https://github.com/projen/projen/issues/2165
            this.package.addPackageResolutions("@types/prettier@2.6.0");
            // https://github.com/projen/projen/issues/2264
            this.package.addPackageResolutions("@types/babel__traverse@7.18.2");
            const jsiiVersion = options.jsiiVersion ?? "1.x";
            if (jsiiVersion.startsWith("1.")) {
                const majorNodeVersion = (0, semver_1.major)(this.package.minNodeVersion ?? "16.0.0");
                // see https://github.com/projen/projen/issues/3324
                const nodeTypesVersionRange = (majorVersion) => {
                    switch (majorVersion) {
                        case 16:
                            return `^16 <= 16.18.78`;
                        case 18:
                            return `^18 <= 18.11.19`;
                        case 19:
                        case 20:
                        case 21:
                        case 22:
                            this.logger.warn(`jsii@${jsiiVersion} and @types/node@^${majorVersion} are incompatible. Falling back to @types/node@^18.`, "Please upgrade to a modern version of jsii.");
                            return `^18 <= 18.11.19`;
                        default:
                            return `^${majorVersion}`;
                    }
                };
                this.addDevDeps(
                // https://github.com/projen/projen/pull/3076
                `@types/node@${nodeTypesVersionRange(majorNodeVersion)}`);
            }
        }
    }
    /**
     * Adds a target language to the build workflow and creates a package task.
     * @param language
     * @returns
     */
    addPackagingTarget(language, packTask, extraJobOptions) {
        if (!this.buildWorkflow) {
            return;
        }
        const pacmak = this.pacmakForLanguage(language, packTask);
        this.buildWorkflow.addPostBuildJob(`package-${language}`, {
            ...(0, runner_options_1.filteredRunsOnOptions)(extraJobOptions.runsOn, extraJobOptions.runsOnGroup),
            permissions: {},
            tools: {
                node: { version: this.nodeVersion ?? "18.x" },
                ...pacmak.publishTools,
            },
            steps: pacmak.prePublishSteps ?? [],
            ...extraJobOptions,
        });
    }
    addPackagingTask(language) {
        const packageTask = this.tasks.addTask(`package:${language}`, {
            description: `Create ${language} language bindings`,
        });
        const commandParts = ["jsii-pacmak", "-v"];
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            commandParts.push("--pack-command 'pnpm pack'");
        }
        commandParts.push(`--target ${language}`);
        packageTask.exec(commandParts.join(" "));
        this.packageAllTask.spawn(packageTask);
        return packageTask;
    }
    pacmakForLanguage(target, packTask) {
        // at this stage, `artifactsDirectory` contains the prebuilt repository.
        // for the publishing to work seamlessely, that directory needs to contain the actual artifact.
        // so we move the repo, create the artifact, and put it in the expected place.
        const prePublishSteps = [];
        prePublishSteps.push(...this.workflowBootstrapSteps);
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            prePublishSteps.push({
                name: "Setup pnpm",
                uses: "pnpm/action-setup@v3",
                with: { version: this.package.pnpmVersion },
            });
        }
        prePublishSteps.push({
            name: "Prepare Repository",
            run: `mv ${this.artifactsDirectory} ${REPO_TEMP_DIRECTORY}`,
        }, {
            name: "Install Dependencies",
            run: `cd ${REPO_TEMP_DIRECTORY} && ${this.package.installCommand}`,
        }, {
            name: `Create ${target} artifact`,
            run: `cd ${REPO_TEMP_DIRECTORY} && npx projen ${packTask.name}`,
        }, {
            name: `Collect ${target} Artifact`,
            run: `mv ${REPO_TEMP_DIRECTORY}/${this.artifactsDirectory} ${this.artifactsDirectory}`,
        });
        return {
            publishTools: consts_1.JSII_TOOLCHAIN[target],
            prePublishSteps,
        };
    }
    /**
     * Generates the runs-on config for Jobs.
     * Throws error if 'runsOn' and 'runsOnGroup' are both set.
     *
     * @param options - 'runsOn' or 'runsOnGroup'.
     */
    getJobRunsOnConfig(options) {
        return options.workflowRunsOnGroup
            ? { runsOnGroup: options.workflowRunsOnGroup }
            : options.workflowRunsOn
                ? { runsOn: options.workflowRunsOn }
                : {};
    }
}
exports.JsiiProject = JsiiProject;
_a = JSII_RTTI_SYMBOL_1;
JsiiProject[_a] = { fqn: "projen.cdk.JsiiProject", version: "0.79.27" };
function parseAuthorAddress(options) {
    let authorEmail = options.authorEmail;
    let authorUrl = options.authorUrl;
    if (options.authorAddress) {
        if (options.authorEmail && options.authorEmail !== options.authorAddress) {
            throw new Error("authorEmail is deprecated and cannot be used in conjunction with authorAddress");
        }
        if (options.authorUrl && options.authorUrl !== options.authorAddress) {
            throw new Error("authorUrl is deprecated and cannot be used in conjunction with authorAddress.");
        }
        if (EMAIL_REGEX.test(options.authorAddress)) {
            authorEmail = options.authorAddress;
        }
        else if (URL_REGEX.test(options.authorAddress)) {
            authorUrl = options.authorAddress;
        }
        else {
            throw new Error(`authorAddress must be either an email address or a URL: ${options.authorAddress}`);
        }
    }
    return { authorEmail, authorUrl };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoianNpaS1wcm9qZWN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2Nkay9qc2lpLXByb2plY3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxrQ0FBa0M7QUFDbEMsbUNBQXNDO0FBQ3RDLHFDQUE0RDtBQUM1RCwrQ0FBMkM7QUFHM0MsOENBQTJEO0FBUTNELHNEQUEwRDtBQUMxRCw4Q0FBNEU7QUFDNUUsa0NBQW9DO0FBRXBDLE1BQU0sV0FBVyxHQUNmLDRhQUE0YSxDQUFDO0FBQy9hLE1BQU0sU0FBUyxHQUNiLHNMQUFzTCxDQUFDO0FBQ3pMLE1BQU0sbUJBQW1CLEdBQUcsT0FBTyxDQUFDO0FBa0hwQyxJQUFZLFNBSVg7QUFKRCxXQUFZLFNBQVM7SUFDbkIsMENBQTZCLENBQUE7SUFDN0IsOEJBQWlCLENBQUE7SUFDakIsc0NBQXlCLENBQUE7QUFDM0IsQ0FBQyxFQUpXLFNBQVMseUJBQVQsU0FBUyxRQUlwQjtBQXVDRDs7OztHQUlHO0FBQ0gsTUFBYSxXQUFZLFNBQVEsOEJBQWlCO0lBS2hELGlFQUFpRTtJQUNqRSxvRkFBb0Y7SUFDcEYsd0ZBQXdGO0lBQ3hGLGFBQWE7SUFDYixJQUFZLGtCQUFrQjtRQUM1QixPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUN6QixJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLENBQ2xFLENBQUM7SUFDSixDQUFDO0lBRUQsWUFBWSxPQUEyQjtRQUNyQyxNQUFNLEVBQUUsV0FBVyxFQUFFLFNBQVMsRUFBRSxHQUFHLGtCQUFrQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRS9ELDJFQUEyRTtRQUMzRSxNQUFNLGNBQWMsR0FDbEIsT0FBTyxDQUFDLFdBQVcsSUFBSSxJQUFJO1lBQzNCLENBQUMsT0FBTyxDQUFDLFdBQVcsS0FBSyxHQUFHO2dCQUMxQixJQUFJLGNBQUssQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLENBQUMsVUFBVSxDQUFDLElBQUksY0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUVqRSxNQUFNLGNBQWMsR0FBc0M7WUFDeEQsVUFBVSxFQUFFLE9BQU8sQ0FBQyxhQUFhO1lBQ2pDLFVBQVUsRUFBRSxPQUFPLENBQUMsTUFBTTtZQUMxQixXQUFXO1lBQ1gsU0FBUztZQUNULFdBQVcsRUFBRSxjQUFjLENBQUMsQ0FBQyxDQUFDLEVBQUUsV0FBVyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO1NBQ2pFLENBQUM7UUFFRixNQUFNLGFBQWEsR0FBRztZQUNwQixZQUFZLEVBQUUsS0FBSztZQUNuQixlQUFlLEVBQUUsSUFBSTtZQUNyQixNQUFNLEVBQUUsS0FBSyxFQUFFLHVEQUF1RDtTQUN2RSxDQUFDO1FBRUYsTUFBTSxhQUFhLEdBQUcsSUFBQSxnQkFBUyxFQUFDO1lBQzlCLGNBQWM7WUFDZCxPQUFPO1lBQ1AsYUFBYTtTQUNkLENBQTZCLENBQUM7UUFFL0IsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBRXJCLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUM7UUFDM0IsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQztRQUUzQixJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsTUFBTSxhQUFhLEVBQUUsQ0FBQyxDQUFDO1FBRWxELE1BQU0sZ0JBQWdCLEdBQUcsT0FBTyxDQUFDLGdCQUFnQixJQUFJLEtBQUssQ0FBQztRQUUzRCwrQkFBK0I7UUFDL0IsTUFBTSxTQUFTLEdBQUcsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQ3ZELElBQUksZ0JBQWdCLEVBQUU7WUFDcEIsU0FBUyxDQUFDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1NBQ3ZDO1FBRUQsTUFBTSxZQUFZLEdBQUcsT0FBTyxDQUFDLFlBQVksSUFBSSxlQUFlLENBQUM7UUFFN0QsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLFNBQVMsRUFBRSxPQUFPLENBQUMsU0FBUyxJQUFJLFNBQVMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1FBRXJFLElBQUksT0FBTyxDQUFDLFNBQVMsS0FBSyxTQUFTLENBQUMsVUFBVSxFQUFFO1lBQzlDLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxVQUFVLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztTQUN0QztRQUVELE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxFQUFFO1lBQ3hDLFdBQVcsRUFBRSx3REFBd0Q7WUFDckUsSUFBSSxFQUFFLGdGQUFnRixZQUFZLGdHQUFnRyxZQUFZLHlCQUF5QjtTQUN4TyxDQUFDLENBQUM7UUFFSCxNQUFNLE1BQU0sR0FBRyxPQUFPLENBQUMsTUFBTSxJQUFJLEtBQUssQ0FBQztRQUN2QyxJQUFJLE1BQU0sRUFBRTtZQUNWLElBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQ3BDO1FBRUQsSUFBSSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxNQUFNLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUN6RCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUM3RCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsYUFBYSxFQUFFO1lBQ2hELFdBQVcsRUFBRSw2Q0FBNkM7U0FDM0QsQ0FBQyxDQUFDO1FBRUgseUVBQXlFO1FBQ3pFLHVFQUF1RTtRQUN2RSxNQUFNLGdCQUFnQixHQUFHO1lBQ3ZCLHdEQUF3RDtZQUN4RCxVQUFVLElBQUksQ0FBQyxrQkFBa0IsRUFBRTtZQUNuQyxZQUFZLElBQUksQ0FBQyxrQkFBa0IsRUFBRTtTQUN0QyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUVmLHVFQUF1RTtRQUN2RSxpQ0FBaUM7UUFDakMsNkRBQTZEO1FBQzdELElBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUNwQiw0QkFBNEIsZ0JBQWdCLFVBQVUsSUFBSSxDQUFDLGNBQWMsQ0FDdkUsSUFBSSxDQUFDLGNBQWMsQ0FDcEIsTUFBTSxDQUNSLENBQUM7UUFFRixNQUFNLE9BQU8sR0FBd0IsRUFBRSxDQUFDO1FBRXhDLE1BQU0sSUFBSSxHQUFRO1lBQ2hCLE1BQU0sRUFBRSxJQUFJLENBQUMsa0JBQWtCO1lBQy9CLE9BQU87WUFDUCxHQUFHLEVBQUU7Z0JBQ0gsTUFBTSxFQUFFLE1BQU07Z0JBQ2QsT0FBTyxFQUFFLE1BQU07YUFDaEI7U0FDRixDQUFDO1FBRUYsSUFBSSxPQUFPLENBQUMsaUJBQWlCLEVBQUU7WUFDN0IsSUFBSSxDQUFDLGlCQUFpQixHQUFHLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQztTQUNwRDtRQUVELElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBRXpCLElBQUksQ0FBQyxPQUFPLEVBQUUsU0FBUyxDQUFDLDJCQUEyQixDQUNqRDtZQUNFLElBQUksRUFBRSxvQkFBb0I7WUFDMUIsR0FBRyxFQUFFLE1BQU0sSUFBSSxDQUFDLGtCQUFrQixJQUFJLG1CQUFtQixFQUFFO1NBQzVELEVBQ0Q7WUFDRSxJQUFJLEVBQUUseUJBQXlCO1lBQy9CLEdBQUcsRUFBRSxNQUFNLG1CQUFtQixJQUFJLElBQUksQ0FBQyxrQkFBa0IsSUFBSSxJQUFJLENBQUMsa0JBQWtCLEVBQUU7U0FDdkYsQ0FDRixDQUFDO1FBRUYsTUFBTSxlQUFlLEdBQWlCO1lBQ3BDLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUFDLE9BQU8sQ0FBQztZQUNuQyxHQUFHLENBQUMsT0FBTyxDQUFDLHNCQUFzQjtnQkFDaEMsQ0FBQyxDQUFDLEVBQUUsU0FBUyxFQUFFLEVBQUUsS0FBSyxFQUFFLE9BQU8sQ0FBQyxzQkFBc0IsRUFBRSxFQUFFO2dCQUMxRCxDQUFDLENBQUMsRUFBRSxDQUFDO1NBQ1IsQ0FBQztRQUVGLElBQUksT0FBTyxDQUFDLFlBQVksSUFBSSxLQUFLLEVBQUU7WUFDakMsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3pDLElBQUksQ0FBQyxPQUFPLEVBQUUsU0FBUyxDQUFDLFlBQVksQ0FBQztnQkFDbkMsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQztnQkFDckMsUUFBUSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVztnQkFDbEMsY0FBYyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYztnQkFDM0MsYUFBYSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsYUFBYTtnQkFDekMsbUJBQW1CLEVBQUUsT0FBTyxDQUFDLG1CQUFtQjthQUNqRCxDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxlQUFlLENBQUMsQ0FBQztTQUN0RDtRQUVELDBFQUEwRTtRQUMxRSw2RUFBNkU7UUFDN0UseURBQXlEO1FBQ3pELElBQUksTUFBTSxJQUFJLE9BQU8sRUFBRTtZQUNyQixNQUFNLElBQUksS0FBSyxDQUFDLGtEQUFrRCxDQUFDLENBQUM7U0FDckU7UUFFRCxJQUFJLE9BQU8sQ0FBQyxjQUFjLEVBQUU7WUFDMUIsT0FBTyxDQUFDLElBQUksR0FBRztnQkFDYixPQUFPLEVBQUUsT0FBTyxDQUFDLGNBQWMsQ0FBQyxXQUFXO2dCQUMzQyxLQUFLLEVBQUU7b0JBQ0wsT0FBTyxFQUFFLE9BQU8sQ0FBQyxjQUFjLENBQUMsWUFBWTtvQkFDNUMsVUFBVSxFQUFFLE9BQU8sQ0FBQyxjQUFjLENBQUMsZUFBZTtpQkFDbkQ7YUFDRixDQUFDO1lBRUYsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBRTNDLElBQUksQ0FBQyxPQUFPLEVBQUUsU0FBUyxDQUFDLGNBQWMsQ0FBQztnQkFDckMsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQztnQkFDdkMsR0FBRyxPQUFPLENBQUMsY0FBYzthQUMxQixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsa0JBQWtCLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxlQUFlLENBQUMsQ0FBQztTQUN4RDtRQUVELE1BQU0sSUFBSSxHQUFHLE9BQU8sQ0FBQyxhQUFhLElBQUksT0FBTyxDQUFDLE1BQU0sQ0FBQztRQUNyRCxJQUFJLElBQUksRUFBRTtZQUNSLE9BQU8sQ0FBQyxNQUFNLEdBQUc7Z0JBQ2YsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO2dCQUN2QixNQUFNLEVBQUUsSUFBSSxDQUFDLE1BQU07YUFDcEIsQ0FBQztZQUVGLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUM3QyxJQUFJLENBQUMsT0FBTyxFQUFFLFNBQVMsQ0FBQyxhQUFhLENBQUM7Z0JBQ3BDLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUM7Z0JBQ3pDLEdBQUcsSUFBSTthQUNSLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLEVBQUUsSUFBSSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1NBQzFEO1FBRUQsTUFBTSxLQUFLLEdBQUcsT0FBTyxDQUFDLGNBQWMsSUFBSSxPQUFPLENBQUMsTUFBTSxDQUFDO1FBQ3ZELElBQUksS0FBSyxFQUFFO1lBQ1QsT0FBTyxDQUFDLE1BQU0sR0FBRztnQkFDZixTQUFTLEVBQUUsS0FBSyxDQUFDLGVBQWU7Z0JBQ2hDLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztnQkFDMUIsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO2FBQ3ZCLENBQUM7WUFFRixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDN0MsSUFBSSxDQUFDLE9BQU8sRUFBRSxTQUFTLENBQUMsY0FBYyxDQUFDO2dCQUNyQyxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDO2dCQUN6QyxHQUFHLEtBQUs7YUFDVCxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsa0JBQWtCLENBQUMsUUFBUSxFQUFFLElBQUksRUFBRSxlQUFlLENBQUMsQ0FBQztTQUMxRDtRQUVELE1BQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUM7UUFDbkMsSUFBSSxNQUFNLEVBQUU7WUFDVixPQUFPLENBQUMsRUFBRSxHQUFHO2dCQUNYLFVBQVUsRUFBRSxNQUFNLENBQUMsVUFBVTtnQkFDN0IsV0FBVyxFQUFFLE1BQU0sQ0FBQyxXQUFXO2FBQ2hDLENBQUM7WUFFRixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDekMsSUFBSSxDQUFDLE9BQU8sRUFBRSxTQUFTLENBQUMsV0FBVyxDQUFDO2dCQUNsQyxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDO2dCQUNyQyxHQUFHLE1BQU07YUFDVixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxlQUFlLENBQUMsQ0FBQztTQUN0RDtRQUVELE1BQU0sVUFBVSxHQUNkLE9BQU8sQ0FBQyxXQUFXLEtBQUssR0FBRztZQUN6QixDQUFDLENBQUMsd0VBQXdFO2dCQUN4RSxFQUFFO1lBQ0osQ0FBQyxDQUFDLHNEQUFzRDtnQkFDdEQsSUFBSSxPQUFPLENBQUMsV0FBVyxJQUFJLEtBQUssRUFBRSxDQUFDO1FBQ3pDLElBQUksQ0FBQyxVQUFVLENBQ2IsT0FBTyxVQUFVLEVBQUUsRUFDbkIsZUFBZSxVQUFVLEVBQUUsRUFDM0IsV0FBVyxFQUNYLGFBQWEsQ0FDZCxDQUFDO1FBRUYsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBQ2pELElBQUksQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRWpDLElBQUksT0FBTyxDQUFDLE1BQU0sSUFBSSxJQUFJLEVBQUU7WUFDMUIsSUFBSSx3QkFBVSxDQUFDLElBQUksRUFBRSxFQUFFLFFBQVEsRUFBRSxPQUFPLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQztTQUM1RDtRQUVELGtEQUFrRDtRQUNsRCxJQUFJLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDbEIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDO1NBQ2pDO1FBRUQsMEVBQTBFO1FBQzFFLG9FQUFvRTtRQUNwRSxJQUFJLGNBQWMsRUFBRTtZQUNsQiwrQ0FBK0M7WUFDL0MsSUFBSSxDQUFDLE9BQU8sQ0FBQyxxQkFBcUIsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1lBRTVELCtDQUErQztZQUMvQyxJQUFJLENBQUMsT0FBTyxDQUFDLHFCQUFxQixDQUFDLCtCQUErQixDQUFDLENBQUM7WUFFcEUsTUFBTSxXQUFXLEdBQUcsT0FBTyxDQUFDLFdBQVcsSUFBSSxLQUFLLENBQUM7WUFDakQsSUFBSSxXQUFXLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxFQUFFO2dCQUNoQyxNQUFNLGdCQUFnQixHQUFHLElBQUEsY0FBSyxFQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxJQUFJLFFBQVEsQ0FBQyxDQUFDO2dCQUV4RSxtREFBbUQ7Z0JBQ25ELE1BQU0scUJBQXFCLEdBQUcsQ0FBQyxZQUFvQixFQUFVLEVBQUU7b0JBQzdELFFBQVEsWUFBWSxFQUFFO3dCQUNwQixLQUFLLEVBQUU7NEJBQ0wsT0FBTyxpQkFBaUIsQ0FBQzt3QkFDM0IsS0FBSyxFQUFFOzRCQUNMLE9BQU8saUJBQWlCLENBQUM7d0JBQzNCLEtBQUssRUFBRSxDQUFDO3dCQUNSLEtBQUssRUFBRSxDQUFDO3dCQUNSLEtBQUssRUFBRSxDQUFDO3dCQUNSLEtBQUssRUFBRTs0QkFDTCxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FDZCxRQUFRLFdBQVcscUJBQXFCLFlBQVkscURBQXFELEVBQ3pHLDZDQUE2QyxDQUM5QyxDQUFDOzRCQUNGLE9BQU8saUJBQWlCLENBQUM7d0JBQzNCOzRCQUNFLE9BQU8sSUFBSSxZQUFZLEVBQUUsQ0FBQztxQkFDN0I7Z0JBQ0gsQ0FBQyxDQUFDO2dCQUVGLElBQUksQ0FBQyxVQUFVO2dCQUNiLDZDQUE2QztnQkFDN0MsZUFBZSxxQkFBcUIsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLENBQ3pELENBQUM7YUFDSDtTQUNGO0lBQ0gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxrQkFBa0IsQ0FDeEIsUUFBMEIsRUFDMUIsUUFBYyxFQUNkLGVBQTZCO1FBRTdCLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFO1lBQ3ZCLE9BQU87U0FDUjtRQUNELE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxRQUFRLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFFMUQsSUFBSSxDQUFDLGFBQWEsQ0FBQyxlQUFlLENBQUMsV0FBVyxRQUFRLEVBQUUsRUFBRTtZQUN4RCxHQUFHLElBQUEsc0NBQXFCLEVBQ3RCLGVBQWUsQ0FBQyxNQUFNLEVBQ3RCLGVBQWUsQ0FBQyxXQUFXLENBQzVCO1lBQ0QsV0FBVyxFQUFFLEVBQUU7WUFDZixLQUFLLEVBQUU7Z0JBQ0wsSUFBSSxFQUFFLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxXQUFXLElBQUksTUFBTSxFQUFFO2dCQUM3QyxHQUFHLE1BQU0sQ0FBQyxZQUFZO2FBQ3ZCO1lBQ0QsS0FBSyxFQUFFLE1BQU0sQ0FBQyxlQUFlLElBQUksRUFBRTtZQUNuQyxHQUFHLGVBQWU7U0FDbkIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLGdCQUFnQixDQUFDLFFBQTBCO1FBQ2pELE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFdBQVcsUUFBUSxFQUFFLEVBQUU7WUFDNUQsV0FBVyxFQUFFLFVBQVUsUUFBUSxvQkFBb0I7U0FDcEQsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxZQUFZLEdBQUcsQ0FBQyxhQUFhLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFFM0MsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsS0FBSywrQkFBa0IsQ0FBQyxJQUFJLEVBQUU7WUFDM0QsWUFBWSxDQUFDLElBQUksQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1NBQ2pEO1FBRUQsWUFBWSxDQUFDLElBQUksQ0FBQyxZQUFZLFFBQVEsRUFBRSxDQUFDLENBQUM7UUFFMUMsV0FBVyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFFekMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDdkMsT0FBTyxXQUFXLENBQUM7SUFDckIsQ0FBQztJQUVPLGlCQUFpQixDQUN2QixNQUF3QixFQUN4QixRQUFjO1FBRWQsd0VBQXdFO1FBQ3hFLCtGQUErRjtRQUMvRiw4RUFBOEU7UUFDOUUsTUFBTSxlQUFlLEdBQWdCLEVBQUUsQ0FBQztRQUV4QyxlQUFlLENBQUMsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFFckQsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsS0FBSywrQkFBa0IsQ0FBQyxJQUFJLEVBQUU7WUFDM0QsZUFBZSxDQUFDLElBQUksQ0FBQztnQkFDbkIsSUFBSSxFQUFFLFlBQVk7Z0JBQ2xCLElBQUksRUFBRSxzQkFBc0I7Z0JBQzVCLElBQUksRUFBRSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRTthQUM1QyxDQUFDLENBQUM7U0FDSjtRQUVELGVBQWUsQ0FBQyxJQUFJLENBQ2xCO1lBQ0UsSUFBSSxFQUFFLG9CQUFvQjtZQUMxQixHQUFHLEVBQUUsTUFBTSxJQUFJLENBQUMsa0JBQWtCLElBQUksbUJBQW1CLEVBQUU7U0FDNUQsRUFDRDtZQUNFLElBQUksRUFBRSxzQkFBc0I7WUFDNUIsR0FBRyxFQUFFLE1BQU0sbUJBQW1CLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUU7U0FDbkUsRUFDRDtZQUNFLElBQUksRUFBRSxVQUFVLE1BQU0sV0FBVztZQUNqQyxHQUFHLEVBQUUsTUFBTSxtQkFBbUIsa0JBQWtCLFFBQVEsQ0FBQyxJQUFJLEVBQUU7U0FDaEUsRUFDRDtZQUNFLElBQUksRUFBRSxXQUFXLE1BQU0sV0FBVztZQUNsQyxHQUFHLEVBQUUsTUFBTSxtQkFBbUIsSUFBSSxJQUFJLENBQUMsa0JBQWtCLElBQUksSUFBSSxDQUFDLGtCQUFrQixFQUFFO1NBQ3ZGLENBQ0YsQ0FBQztRQUNGLE9BQU87WUFDTCxZQUFZLEVBQUUsdUJBQWMsQ0FBQyxNQUFNLENBQUM7WUFDcEMsZUFBZTtTQUNoQixDQUFDO0lBQ0osQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ssa0JBQWtCLENBQUMsT0FBMkI7UUFDcEQsT0FBTyxPQUFPLENBQUMsbUJBQW1CO1lBQ2hDLENBQUMsQ0FBQyxFQUFFLFdBQVcsRUFBRSxPQUFPLENBQUMsbUJBQW1CLEVBQUU7WUFDOUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxjQUFjO2dCQUN4QixDQUFDLENBQUMsRUFBRSxNQUFNLEVBQUUsT0FBTyxDQUFDLGNBQWMsRUFBRTtnQkFDcEMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztJQUNULENBQUM7O0FBeFlILGtDQXlZQzs7O0FBRUQsU0FBUyxrQkFBa0IsQ0FBQyxPQUEyQjtJQUNyRCxJQUFJLFdBQVcsR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO0lBQ3RDLElBQUksU0FBUyxHQUFHLE9BQU8sQ0FBQyxTQUFTLENBQUM7SUFDbEMsSUFBSSxPQUFPLENBQUMsYUFBYSxFQUFFO1FBQ3pCLElBQUksT0FBTyxDQUFDLFdBQVcsSUFBSSxPQUFPLENBQUMsV0FBVyxLQUFLLE9BQU8sQ0FBQyxhQUFhLEVBQUU7WUFDeEUsTUFBTSxJQUFJLEtBQUssQ0FDYixnRkFBZ0YsQ0FDakYsQ0FBQztTQUNIO1FBRUQsSUFBSSxPQUFPLENBQUMsU0FBUyxJQUFJLE9BQU8sQ0FBQyxTQUFTLEtBQUssT0FBTyxDQUFDLGFBQWEsRUFBRTtZQUNwRSxNQUFNLElBQUksS0FBSyxDQUNiLCtFQUErRSxDQUNoRixDQUFDO1NBQ0g7UUFFRCxJQUFJLFdBQVcsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxFQUFFO1lBQzNDLFdBQVcsR0FBRyxPQUFPLENBQUMsYUFBYSxDQUFDO1NBQ3JDO2FBQU0sSUFBSSxTQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsRUFBRTtZQUNoRCxTQUFTLEdBQUcsT0FBTyxDQUFDLGFBQWEsQ0FBQztTQUNuQzthQUFNO1lBQ0wsTUFBTSxJQUFJLEtBQUssQ0FDYiwyREFBMkQsT0FBTyxDQUFDLGFBQWEsRUFBRSxDQUNuRixDQUFDO1NBQ0g7S0FDRjtJQUNELE9BQU8sRUFBRSxXQUFXLEVBQUUsU0FBUyxFQUFFLENBQUM7QUFDcEMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSBcIm5vZGU6cGF0aFwiO1xuaW1wb3J0IHsgUmFuZ2UsIG1ham9yIH0gZnJvbSBcInNlbXZlclwiO1xuaW1wb3J0IHsgSnNpaVBhY21ha1RhcmdldCwgSlNJSV9UT09MQ0hBSU4gfSBmcm9tIFwiLi9jb25zdHNcIjtcbmltcG9ydCB7IEpzaWlEb2NnZW4gfSBmcm9tIFwiLi9qc2lpLWRvY2dlblwiO1xuaW1wb3J0IHsgVGFzayB9IGZyb20gXCIuLlwiO1xuaW1wb3J0IHsgSm9iLCBTdGVwIH0gZnJvbSBcIi4uL2dpdGh1Yi93b3JrZmxvd3MtbW9kZWxcIjtcbmltcG9ydCB7IEVzbGludCwgTm9kZVBhY2thZ2VNYW5hZ2VyIH0gZnJvbSBcIi4uL2phdmFzY3JpcHRcIjtcbmltcG9ydCB7XG4gIENvbW1vblB1Ymxpc2hPcHRpb25zLFxuICBHb1B1Ymxpc2hPcHRpb25zLFxuICBNYXZlblB1Ymxpc2hPcHRpb25zLFxuICBOdWdldFB1Ymxpc2hPcHRpb25zLFxuICBQeVBpUHVibGlzaE9wdGlvbnMsXG59IGZyb20gXCIuLi9yZWxlYXNlXCI7XG5pbXBvcnQgeyBmaWx0ZXJlZFJ1bnNPbk9wdGlvbnMgfSBmcm9tIFwiLi4vcnVubmVyLW9wdGlvbnNcIjtcbmltcG9ydCB7IFR5cGVTY3JpcHRQcm9qZWN0LCBUeXBlU2NyaXB0UHJvamVjdE9wdGlvbnMgfSBmcm9tIFwiLi4vdHlwZXNjcmlwdFwiO1xuaW1wb3J0IHsgZGVlcE1lcmdlIH0gZnJvbSBcIi4uL3V0aWxcIjtcblxuY29uc3QgRU1BSUxfUkVHRVggPVxuICAvKD86W2EtejAtOSEjJCUmJyorLz0/Xl9ge3x9fi1dKyg/OlxcLlthLXowLTkhIyQlJicqKy89P15fYHt8fX4tXSspKnxcIig/OltcXHgwMS1cXHgwOFxceDBiXFx4MGNcXHgwZS1cXHgxZlxceDIxXFx4MjMtXFx4NWJcXHg1ZC1cXHg3Zl18XFxcXFtcXHgwMS1cXHgwOVxceDBiXFx4MGNcXHgwZS1cXHg3Zl0pKlwiKUAoPzooPzpbYS16MC05XSg/OlthLXowLTktXSpbYS16MC05XSk/XFwuKStbYS16MC05XSg/OlthLXowLTktXSpbYS16MC05XSk/fFxcWyg/Oig/OjI1WzAtNV18MlswLTRdWzAtOV18WzAxXT9bMC05XVswLTldPylcXC4pezN9KD86MjVbMC01XXwyWzAtNF1bMC05XXxbMDFdP1swLTldWzAtOV0/fFthLXowLTktXSpbYS16MC05XTooPzpbXFx4MDEtXFx4MDhcXHgwYlxceDBjXFx4MGUtXFx4MWZcXHgyMS1cXHg1YVxceDUzLVxceDdmXXxcXFxcW1xceDAxLVxceDA5XFx4MGJcXHgwY1xceDBlLVxceDdmXSkrKVxcXSkvO1xuY29uc3QgVVJMX1JFR0VYID1cbiAgLygoKFtBLVphLXpdezMsOX06KD86XFwvXFwvKT8pKD86W1xcLTs6Jj1cXCtcXCQsXFx3XStAKT9bQS1aYS16MC05XFwuXFwtXSt8KD86d3d3XFwufFtcXC07OiY9XFwrXFwkLFxcd10rQClbQS1aYS16MC05XFwuXFwtXSspKCg/OlxcL1tcXCt+JVxcL1xcLlxcd1xcLV9dKik/XFw/Pyg/OltcXC1cXCs9JjslQFxcLlxcd19dKikjPyg/OltcXC5cXCFcXC9cXFxcXFx3XSopKT8pLztcbmNvbnN0IFJFUE9fVEVNUF9ESVJFQ1RPUlkgPSBcIi5yZXBvXCI7XG5cbmV4cG9ydCBpbnRlcmZhY2UgSnNpaVByb2plY3RPcHRpb25zIGV4dGVuZHMgVHlwZVNjcmlwdFByb2plY3RPcHRpb25zIHtcbiAgLyoqXG4gICAqIEBkZWZhdWx0IFwiLlwiXG4gICAqL1xuICByZWFkb25seSByb290ZGlyPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBHaXQgcmVwb3NpdG9yeSBVUkwuXG4gICAqIEBkZWZhdWx0ICRHSVRfUkVNT1RFXG4gICAqL1xuICByZWFkb25seSByZXBvc2l0b3J5VXJsOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBsaWJyYXJ5IGF1dGhvci5cbiAgICogQGRlZmF1bHQgJEdJVF9VU0VSX05BTUVcbiAgICovXG4gIHJlYWRvbmx5IGF1dGhvcjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBFbWFpbCBvciBVUkwgb2YgdGhlIGxpYnJhcnkgYXV0aG9yLlxuICAgKiBAZGVmYXVsdCAkR0lUX1VTRVJfRU1BSUxcbiAgICovXG4gIHJlYWRvbmx5IGF1dGhvckFkZHJlc3M6IHN0cmluZztcblxuICAvKipcbiAgICogUHVibGlzaCB0byBtYXZlblxuICAgKiBAZGVmYXVsdCAtIG5vIHB1Ymxpc2hpbmdcbiAgICovXG4gIHJlYWRvbmx5IHB1Ymxpc2hUb01hdmVuPzogSnNpaUphdmFUYXJnZXQ7XG5cbiAgLyoqXG4gICAqIFB1Ymxpc2ggdG8gcHlwaVxuICAgKiBAZGVmYXVsdCAtIG5vIHB1Ymxpc2hpbmdcbiAgICovXG4gIHJlYWRvbmx5IHB1Ymxpc2hUb1B5cGk/OiBKc2lpUHl0aG9uVGFyZ2V0O1xuXG4gIC8qKlxuICAgKiBQdWJsaXNoIEdvIGJpbmRpbmdzIHRvIGEgZ2l0IHJlcG9zaXRvcnkuXG4gICAqIEBkZWZhdWx0IC0gbm8gcHVibGlzaGluZ1xuICAgKi9cbiAgcmVhZG9ubHkgcHVibGlzaFRvR28/OiBKc2lpR29UYXJnZXQ7XG5cbiAgLyoqXG4gICAqIEBkZXByZWNhdGVkIHVzZSBgcHVibGlzaFRvUHlQaWBcbiAgICovXG4gIHJlYWRvbmx5IHB5dGhvbj86IEpzaWlQeXRob25UYXJnZXQ7XG5cbiAgLyoqXG4gICAqIFB1Ymxpc2ggdG8gTnVHZXRcbiAgICogQGRlZmF1bHQgLSBubyBwdWJsaXNoaW5nXG4gICAqL1xuICByZWFkb25seSBwdWJsaXNoVG9OdWdldD86IEpzaWlEb3ROZXRUYXJnZXQ7XG5cbiAgLyoqXG4gICAqIEBkZXByZWNhdGVkIHVzZSBgcHVibGlzaFRvTnVnZXRgXG4gICAqL1xuICByZWFkb25seSBkb3RuZXQ/OiBKc2lpRG90TmV0VGFyZ2V0O1xuXG4gIC8qKlxuICAgKiBBdXRvbWF0aWNhbGx5IHJ1biBBUEkgY29tcGF0aWJpbGl0eSB0ZXN0IGFnYWluc3QgdGhlIGxhdGVzdCB2ZXJzaW9uIHB1Ymxpc2hlZCB0byBucG0gYWZ0ZXIgY29tcGlsYXRpb24uXG4gICAqXG4gICAqIC0gWW91IGNhbiBtYW51YWxseSBydW4gY29tcGF0aWJpbGl0eSB0ZXN0cyB1c2luZyBgeWFybiBjb21wYXRgIGlmIHRoaXMgZmVhdHVyZSBpcyBkaXNhYmxlZC5cbiAgICogLSBZb3UgY2FuIGlnbm9yZSBjb21wYXRpYmlsaXR5IGZhaWx1cmVzIGJ5IGFkZGluZyBsaW5lcyB0byBhIFwiLmNvbXBhdGlnbm9yZVwiIGZpbGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBjb21wYXQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoZSBpZ25vcmUgZmlsZSBmb3IgQVBJIGNvbXBhdGliaWxpdHkgdGVzdHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IFwiLmNvbXBhdGlnbm9yZVwiXG4gICAqL1xuICByZWFkb25seSBjb21wYXRJZ25vcmU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFjY2VwdHMgYSBsaXN0IG9mIGdsb2IgcGF0dGVybnMuIEZpbGVzIG1hdGNoaW5nIGFueSBvZiB0aG9zZSBwYXR0ZXJucyB3aWxsIGJlIGV4Y2x1ZGVkIGZyb20gdGhlIFR5cGVTY3JpcHQgY29tcGlsZXIgaW5wdXQuXG4gICAqXG4gICAqIEJ5IGRlZmF1bHQsIGpzaWkgd2lsbCBpbmNsdWRlIGFsbCAqLnRzIGZpbGVzIChleGNlcHQgLmQudHMgZmlsZXMpIGluIHRoZSBUeXBlU2NyaXB0IGNvbXBpbGVyIGlucHV0LlxuICAgKiBUaGlzIGNhbiBiZSBwcm9ibGVtYXRpYyBmb3IgZXhhbXBsZSB3aGVuIHRoZSBwYWNrYWdlJ3MgYnVpbGQgb3IgdGVzdCBwcm9jZWR1cmUgZ2VuZXJhdGVzIC50cyBmaWxlc1xuICAgKiB0aGF0IGNhbm5vdCBiZSBjb21waWxlZCB3aXRoIGpzaWkncyBjb21waWxlciBzZXR0aW5ncy5cbiAgICovXG4gIHJlYWRvbmx5IGV4Y2x1ZGVUeXBlc2NyaXB0Pzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIEZpbGUgcGF0aCBmb3IgZ2VuZXJhdGVkIGRvY3MuXG4gICAqIEBkZWZhdWx0IFwiQVBJLm1kXCJcbiAgICovXG4gIHJlYWRvbmx5IGRvY2dlbkZpbGVQYXRoPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBFbWl0IGEgY29tcHJlc3NlZCB2ZXJzaW9uIG9mIHRoZSBhc3NlbWJseVxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgY29tcHJlc3NBc3NlbWJseT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFZlcnNpb24gb2YgdGhlIGpzaWkgY29tcGlsZXIgdG8gdXNlLlxuICAgKlxuICAgKiBTZXQgdG8gXCIqXCIgaWYgeW91IHdhbnQgdG8gbWFudWFsbHkgbWFuYWdlIHRoZSB2ZXJzaW9uIG9mIGpzaWkgaW4geW91clxuICAgKiBwcm9qZWN0IGJ5IG1hbmFnaW5nIHVwZGF0ZXMgdG8gYHBhY2thZ2UuanNvbmAgb24geW91ciBvd24uXG4gICAqXG4gICAqIE5PVEU6IFRoZSBqc2lpIGNvbXBpbGVyIHJlbGVhc2VzIHNpbmNlIDUuMC4wIGFyZSBub3Qgc2VtYW50aWNhbGx5IHZlcnNpb25lZFxuICAgKiBhbmQgc2hvdWxkIHJlbWFpbiBvbiB0aGUgc2FtZSBtaW5vciwgc28gd2UgcmVjb21tZW5kIHVzaW5nIGEgYH5gIGRlcGVuZGVuY3lcbiAgICogKGUuZy4gYH41LjAuMGApLlxuICAgKlxuICAgKiBAZGVmYXVsdCBcIjEueFwiXG4gICAqIEBwam5ldyBcIn41LjAuMFwiXG4gICAqL1xuICByZWFkb25seSBqc2lpVmVyc2lvbj86IHN0cmluZztcbn1cblxuZXhwb3J0IGVudW0gU3RhYmlsaXR5IHtcbiAgRVhQRVJJTUVOVEFMID0gXCJleHBlcmltZW50YWxcIixcbiAgU1RBQkxFID0gXCJzdGFibGVcIixcbiAgREVQUkVDQVRFRCA9IFwiZGVwcmVjYXRlZFwiLFxufVxuXG5leHBvcnQgaW50ZXJmYWNlIEpzaWlKYXZhVGFyZ2V0IGV4dGVuZHMgTWF2ZW5QdWJsaXNoT3B0aW9ucyB7XG4gIHJlYWRvbmx5IGphdmFQYWNrYWdlOiBzdHJpbmc7XG4gIHJlYWRvbmx5IG1hdmVuR3JvdXBJZDogc3RyaW5nO1xuICByZWFkb25seSBtYXZlbkFydGlmYWN0SWQ6IHN0cmluZztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBKc2lpUHl0aG9uVGFyZ2V0IGV4dGVuZHMgUHlQaVB1Ymxpc2hPcHRpb25zIHtcbiAgcmVhZG9ubHkgZGlzdE5hbWU6IHN0cmluZztcbiAgcmVhZG9ubHkgbW9kdWxlOiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgSnNpaURvdE5ldFRhcmdldCBleHRlbmRzIE51Z2V0UHVibGlzaE9wdGlvbnMge1xuICByZWFkb25seSBkb3ROZXROYW1lc3BhY2U6IHN0cmluZztcbiAgcmVhZG9ubHkgcGFja2FnZUlkOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGljb25Vcmw/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogR28gdGFyZ2V0IGNvbmZpZ3VyYXRpb25cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBKc2lpR29UYXJnZXQgZXh0ZW5kcyBHb1B1Ymxpc2hPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSB0YXJnZXQgZ28gbW9kdWxlLlxuICAgKlxuICAgKiBAZXhhbXBsZSBnaXRodWIuY29tL293bmVyL3JlcG9cbiAgICogQGV4YW1wbGUgZ2l0aHViLmNvbS9vd25lci9yZXBvL3N1YmRpclxuICAgKi9cbiAgcmVhZG9ubHkgbW9kdWxlTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgZ28gcGFja2FnZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZXJpdmVkIGZyb20gdGhlIG1vZHVsZSBuYW1lXG4gICAqL1xuICByZWFkb25seSBwYWNrYWdlTmFtZT86IHN0cmluZztcbn1cblxuLyoqXG4gKiBNdWx0aS1sYW5ndWFnZSBqc2lpIGxpYnJhcnkgcHJvamVjdFxuICpcbiAqIEBwamlkIGpzaWlcbiAqL1xuZXhwb3J0IGNsYXNzIEpzaWlQcm9qZWN0IGV4dGVuZHMgVHlwZVNjcmlwdFByb2plY3Qge1xuICBwdWJsaWMgcmVhZG9ubHkgZXNsaW50PzogRXNsaW50O1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgcGFja2FnZUFsbFRhc2s6IFRhc2s7XG5cbiAgLy8gVGhpcyBwcm9qZWN0IGNvcGllcyB0byB3aG9sZSByZXBvIGludG8gdGhlIGFydGlmYWN0c0RpcmVjdG9yeS5cbiAgLy8gV2hpY2ggY2F1c2VzIHRoZSByZWxlYXNlIHRhZyBmaWxlIHRvIGJlIGluIGEgcGF0aCBsaWtlIC4vZGlzdC9kaXN0L3JlbGVhc2V0YWcudHh0XG4gIC8vIFVzZSBhIFRTIGhhY2sgdG8gYWxsb3cgdGhlIHJlbGVhc2UgY29tcG9uZW50IHRvIGdldCB0aGUgY29ycmVjdCBwYXRoIGZyb20gdGhlIHByb2plY3RcbiAgLy8gQHRzLWlnbm9yZVxuICBwcml2YXRlIGdldCByZWxlYXNlVGFnRmlsZVBhdGgoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gcGF0aC5wb3NpeC5ub3JtYWxpemUoXG4gICAgICBwYXRoLnBvc2l4LmpvaW4odGhpcy5hcnRpZmFjdHNEaXJlY3RvcnksIHRoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5KVxuICAgICk7XG4gIH1cblxuICBjb25zdHJ1Y3RvcihvcHRpb25zOiBKc2lpUHJvamVjdE9wdGlvbnMpIHtcbiAgICBjb25zdCB7IGF1dGhvckVtYWlsLCBhdXRob3JVcmwgfSA9IHBhcnNlQXV0aG9yQWRkcmVzcyhvcHRpb25zKTtcblxuICAgIC8vIFRydWUgaWYganNpaSB2ZXJzaW9uIDEueCBpcyBjb21wYXRpYmxlIHdpdGggdGhlIHJlcXVlc3RlZCB2ZXJzaW9uIHJhbmdlLlxuICAgIGNvbnN0IHVzZXNMZWdhY3lKc2lpID1cbiAgICAgIG9wdGlvbnMuanNpaVZlcnNpb24gPT0gbnVsbCB8fFxuICAgICAgKG9wdGlvbnMuanNpaVZlcnNpb24gIT09IFwiKlwiICYmXG4gICAgICAgIG5ldyBSYW5nZShvcHRpb25zLmpzaWlWZXJzaW9uKS5pbnRlcnNlY3RzKG5ldyBSYW5nZShcIjEueFwiKSkpO1xuXG4gICAgY29uc3QgZGVmYXVsdE9wdGlvbnM6IFBhcnRpYWw8VHlwZVNjcmlwdFByb2plY3RPcHRpb25zPiA9IHtcbiAgICAgIHJlcG9zaXRvcnk6IG9wdGlvbnMucmVwb3NpdG9yeVVybCxcbiAgICAgIGF1dGhvck5hbWU6IG9wdGlvbnMuYXV0aG9yLFxuICAgICAgYXV0aG9yRW1haWwsXG4gICAgICBhdXRob3JVcmwsXG4gICAgICBqZXN0T3B0aW9uczogdXNlc0xlZ2FjeUpzaWkgPyB7IGplc3RWZXJzaW9uOiBcIl4yN1wiIH0gOiB1bmRlZmluZWQsXG4gICAgfTtcblxuICAgIGNvbnN0IGZvcmNlZE9wdGlvbnMgPSB7XG4gICAgICByZWxlYXNlVG9OcG06IGZhbHNlLCAvLyB3ZSBoYXZlIGEganNpaSByZWxlYXNlIHdvcmtmbG93XG4gICAgICBkaXNhYmxlVHNjb25maWc6IHRydWUsIC8vIGpzaWkgZ2VuZXJhdGVzIGl0cyBvd24gdHNjb25maWcuanNvblxuICAgICAgZG9jZ2VuOiBmYWxzZSwgLy8gd2UgdXNlIGpzaWktZG9jZ2VuIGhlcmUgc28gZGlzYWJsZSB0eXBlc2NyaXB0IGRvY2dlblxuICAgIH07XG5cbiAgICBjb25zdCBtZXJnZWRPcHRpb25zID0gZGVlcE1lcmdlKFtcbiAgICAgIGRlZmF1bHRPcHRpb25zLFxuICAgICAgb3B0aW9ucyxcbiAgICAgIGZvcmNlZE9wdGlvbnMsXG4gICAgXSkgYXMgVHlwZVNjcmlwdFByb2plY3RPcHRpb25zO1xuXG4gICAgc3VwZXIobWVyZ2VkT3B0aW9ucyk7XG5cbiAgICBjb25zdCBzcmNkaXIgPSB0aGlzLnNyY2RpcjtcbiAgICBjb25zdCBsaWJkaXIgPSB0aGlzLmxpYmRpcjtcblxuICAgIHRoaXMuYWRkRmllbGRzKHsgdHlwZXM6IGAke2xpYmRpcn0vaW5kZXguZC50c2AgfSk7XG5cbiAgICBjb25zdCBjb21wcmVzc0Fzc2VtYmx5ID0gb3B0aW9ucy5jb21wcmVzc0Fzc2VtYmx5ID8/IGZhbHNlO1xuXG4gICAgLy8gdGhpcyBpcyBhbiB1bmhlbHBmdWwgd2FybmluZ1xuICAgIGNvbnN0IGpzaWlGbGFncyA9IFtcIi0tc2lsZW5jZS13YXJuaW5ncz1yZXNlcnZlZC13b3JkXCJdO1xuICAgIGlmIChjb21wcmVzc0Fzc2VtYmx5KSB7XG4gICAgICBqc2lpRmxhZ3MucHVzaChcIi0tY29tcHJlc3MtYXNzZW1ibHlcIik7XG4gICAgfVxuXG4gICAgY29uc3QgY29tcGF0SWdub3JlID0gb3B0aW9ucy5jb21wYXRJZ25vcmUgPz8gXCIuY29tcGF0aWdub3JlXCI7XG5cbiAgICB0aGlzLmFkZEZpZWxkcyh7IHN0YWJpbGl0eTogb3B0aW9ucy5zdGFiaWxpdHkgPz8gU3RhYmlsaXR5LlNUQUJMRSB9KTtcblxuICAgIGlmIChvcHRpb25zLnN0YWJpbGl0eSA9PT0gU3RhYmlsaXR5LkRFUFJFQ0FURUQpIHtcbiAgICAgIHRoaXMuYWRkRmllbGRzKHsgZGVwcmVjYXRlZDogdHJ1ZSB9KTtcbiAgICB9XG5cbiAgICBjb25zdCBjb21wYXRUYXNrID0gdGhpcy5hZGRUYXNrKFwiY29tcGF0XCIsIHtcbiAgICAgIGRlc2NyaXB0aW9uOiBcIlBlcmZvcm0gQVBJIGNvbXBhdGliaWxpdHkgY2hlY2sgYWdhaW5zdCBsYXRlc3QgdmVyc2lvblwiLFxuICAgICAgZXhlYzogYGpzaWktZGlmZiBucG06JChub2RlIC1wIFwicmVxdWlyZShcXCcuL3BhY2thZ2UuanNvblxcJykubmFtZVwiKSAtayAtLWlnbm9yZS1maWxlICR7Y29tcGF0SWdub3JlfSB8fCAoZWNobyBcIlxcblVORVhQRUNURUQgQlJFQUtJTkcgQ0hBTkdFUzogYWRkIGtleXMgc3VjaCBhcyBcXCdyZW1vdmVkOmNvbnN0cnVjdHMuTm9kZS5vZlxcJyB0byAke2NvbXBhdElnbm9yZX0gdG8gc2tpcC5cXG5cIiAmJiBleGl0IDEpYCxcbiAgICB9KTtcblxuICAgIGNvbnN0IGNvbXBhdCA9IG9wdGlvbnMuY29tcGF0ID8/IGZhbHNlO1xuICAgIGlmIChjb21wYXQpIHtcbiAgICAgIHRoaXMuY29tcGlsZVRhc2suc3Bhd24oY29tcGF0VGFzayk7XG4gICAgfVxuXG4gICAgdGhpcy5jb21waWxlVGFzay5yZXNldChbXCJqc2lpXCIsIC4uLmpzaWlGbGFnc10uam9pbihcIiBcIikpO1xuICAgIHRoaXMud2F0Y2hUYXNrLnJlc2V0KFtcImpzaWlcIiwgXCItd1wiLCAuLi5qc2lpRmxhZ3NdLmpvaW4oXCIgXCIpKTtcbiAgICB0aGlzLnBhY2thZ2VBbGxUYXNrID0gdGhpcy5hZGRUYXNrKFwicGFja2FnZS1hbGxcIiwge1xuICAgICAgZGVzY3JpcHRpb246IFwiUGFja2FnZXMgYXJ0aWZhY3RzIGZvciBhbGwgdGFyZ2V0IGxhbmd1YWdlc1wiLFxuICAgIH0pO1xuXG4gICAgLy8gaW4ganNpaSB3ZSBjb25zaWRlciB0aGUgZW50aXJlIHJlcG8gKHBvc3QgYnVpbGQpIGFzIHRoZSBidWlsZCBhcnRpZmFjdFxuICAgIC8vIHdoaWNoIGlzIHRoZW4gdXNlZCB0byBjcmVhdGUgdGhlIGxhbmd1YWdlIGJpbmRpbmdzIGluIHNlcGFyYXRlIGpvYnMuXG4gICAgY29uc3QgcHJlcGFyZVJlcG9Gb3JDSSA9IFtcbiAgICAgIGByc3luYyAtYSAuIC5yZXBvIC0tZXhjbHVkZSAuZ2l0IC0tZXhjbHVkZSBub2RlX21vZHVsZXNgLFxuICAgICAgYHJtIC1yZiAke3RoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5fWAsXG4gICAgICBgbXYgLnJlcG8gJHt0aGlzLmFydGlmYWN0c0RpcmVjdG9yeX1gLFxuICAgIF0uam9pbihcIiAmJiBcIik7XG5cbiAgICAvLyB3aGVuIHJ1bm5pbmcgaW5zaWRlIENJIHdlIGp1c3QgcHJlcGFyZSB0aGUgcmVwbyBmb3IgcGFja2FnaW5nLCB3aGljaFxuICAgIC8vIHRha2VzIHBsYWNlIGluIHNlcGFyYXRlIHRhc2tzLlxuICAgIC8vIG91dHNpZGUgb2YgQ0kgKGkuZSBsb2NhbGx5KSB3ZSBzaW1wbHkgcGFja2FnZSBhbGwgdGFyZ2V0cy5cbiAgICB0aGlzLnBhY2thZ2VUYXNrLnJlc2V0KFxuICAgICAgYGlmIFsgISAteiBcXCR7Q0l9IF07IHRoZW4gJHtwcmVwYXJlUmVwb0ZvckNJfTsgZWxzZSAke3RoaXMucnVuVGFza0NvbW1hbmQoXG4gICAgICAgIHRoaXMucGFja2FnZUFsbFRhc2tcbiAgICAgICl9OyBmaWBcbiAgICApO1xuXG4gICAgY29uc3QgdGFyZ2V0czogUmVjb3JkPHN0cmluZywgYW55PiA9IHt9O1xuXG4gICAgY29uc3QganNpaTogYW55ID0ge1xuICAgICAgb3V0ZGlyOiB0aGlzLmFydGlmYWN0c0RpcmVjdG9yeSxcbiAgICAgIHRhcmdldHMsXG4gICAgICB0c2M6IHtcbiAgICAgICAgb3V0RGlyOiBsaWJkaXIsXG4gICAgICAgIHJvb3REaXI6IHNyY2RpcixcbiAgICAgIH0sXG4gICAgfTtcblxuICAgIGlmIChvcHRpb25zLmV4Y2x1ZGVUeXBlc2NyaXB0KSB7XG4gICAgICBqc2lpLmV4Y2x1ZGVUeXBlc2NyaXB0ID0gb3B0aW9ucy5leGNsdWRlVHlwZXNjcmlwdDtcbiAgICB9XG5cbiAgICB0aGlzLmFkZEZpZWxkcyh7IGpzaWkgfSk7XG5cbiAgICB0aGlzLnJlbGVhc2U/LnB1Ymxpc2hlci5hZGRHaXRIdWJQcmVQdWJsaXNoaW5nU3RlcHMoXG4gICAgICB7XG4gICAgICAgIG5hbWU6IFwiUHJlcGFyZSBSZXBvc2l0b3J5XCIsXG4gICAgICAgIHJ1bjogYG12ICR7dGhpcy5hcnRpZmFjdHNEaXJlY3Rvcnl9ICR7UkVQT19URU1QX0RJUkVDVE9SWX1gLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgbmFtZTogXCJDb2xsZWN0IEdpdEh1YiBNZXRhZGF0YVwiLFxuICAgICAgICBydW46IGBtdiAke1JFUE9fVEVNUF9ESVJFQ1RPUll9LyR7dGhpcy5hcnRpZmFjdHNEaXJlY3Rvcnl9ICR7dGhpcy5hcnRpZmFjdHNEaXJlY3Rvcnl9YCxcbiAgICAgIH1cbiAgICApO1xuXG4gICAgY29uc3QgZXh0cmFKb2JPcHRpb25zOiBQYXJ0aWFsPEpvYj4gPSB7XG4gICAgICAuLi50aGlzLmdldEpvYlJ1bnNPbkNvbmZpZyhvcHRpb25zKSxcbiAgICAgIC4uLihvcHRpb25zLndvcmtmbG93Q29udGFpbmVySW1hZ2VcbiAgICAgICAgPyB7IGNvbnRhaW5lcjogeyBpbWFnZTogb3B0aW9ucy53b3JrZmxvd0NvbnRhaW5lckltYWdlIH0gfVxuICAgICAgICA6IHt9KSxcbiAgICB9O1xuXG4gICAgaWYgKG9wdGlvbnMucmVsZWFzZVRvTnBtICE9IGZhbHNlKSB7XG4gICAgICBjb25zdCB0YXNrID0gdGhpcy5hZGRQYWNrYWdpbmdUYXNrKFwianNcIik7XG4gICAgICB0aGlzLnJlbGVhc2U/LnB1Ymxpc2hlci5wdWJsaXNoVG9OcG0oe1xuICAgICAgICAuLi50aGlzLnBhY21ha0Zvckxhbmd1YWdlKFwianNcIiwgdGFzayksXG4gICAgICAgIHJlZ2lzdHJ5OiB0aGlzLnBhY2thZ2UubnBtUmVnaXN0cnksXG4gICAgICAgIG5wbVRva2VuU2VjcmV0OiB0aGlzLnBhY2thZ2UubnBtVG9rZW5TZWNyZXQsXG4gICAgICAgIG5wbVByb3ZlbmFuY2U6IHRoaXMucGFja2FnZS5ucG1Qcm92ZW5hbmNlLFxuICAgICAgICBjb2RlQXJ0aWZhY3RPcHRpb25zOiBvcHRpb25zLmNvZGVBcnRpZmFjdE9wdGlvbnMsXG4gICAgICB9KTtcbiAgICAgIHRoaXMuYWRkUGFja2FnaW5nVGFyZ2V0KFwianNcIiwgdGFzaywgZXh0cmFKb2JPcHRpb25zKTtcbiAgICB9XG5cbiAgICAvLyB3ZSBjYW5ub3QgY2FsbCBhbiBvcHRpb24gYGphdmFgIGJlY2F1c2UgdGhlIGphdmEgY29kZSBnZW5lcmF0ZWQgYnkganNpaVxuICAgIC8vIGRvZXMgbm90IGNvbXBpbGUgZHVlIHRvIGEgY29uZmxpY3QgYmV0d2VlbiB0aGlzIG9wdGlvbiBuYW1lIGFuZCB0aGUgYGphdmFgXG4gICAgLy8gcGFja2FnZSAoZS5nLiB3aGVuIGBqYXZhLnV0aWwuT2JqZWN0c2AgaXMgcmVmZXJlbmNlZCkuXG4gICAgaWYgKFwiamF2YVwiIGluIG9wdGlvbnMpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcigndGhlIFwiamF2YVwiIG9wdGlvbiBpcyBub3cgY2FsbGVkIFwicHVibGlzaFRvTWF2ZW5cIicpO1xuICAgIH1cblxuICAgIGlmIChvcHRpb25zLnB1Ymxpc2hUb01hdmVuKSB7XG4gICAgICB0YXJnZXRzLmphdmEgPSB7XG4gICAgICAgIHBhY2thZ2U6IG9wdGlvbnMucHVibGlzaFRvTWF2ZW4uamF2YVBhY2thZ2UsXG4gICAgICAgIG1hdmVuOiB7XG4gICAgICAgICAgZ3JvdXBJZDogb3B0aW9ucy5wdWJsaXNoVG9NYXZlbi5tYXZlbkdyb3VwSWQsXG4gICAgICAgICAgYXJ0aWZhY3RJZDogb3B0aW9ucy5wdWJsaXNoVG9NYXZlbi5tYXZlbkFydGlmYWN0SWQsXG4gICAgICAgIH0sXG4gICAgICB9O1xuXG4gICAgICBjb25zdCB0YXNrID0gdGhpcy5hZGRQYWNrYWdpbmdUYXNrKFwiamF2YVwiKTtcblxuICAgICAgdGhpcy5yZWxlYXNlPy5wdWJsaXNoZXIucHVibGlzaFRvTWF2ZW4oe1xuICAgICAgICAuLi50aGlzLnBhY21ha0Zvckxhbmd1YWdlKFwiamF2YVwiLCB0YXNrKSxcbiAgICAgICAgLi4ub3B0aW9ucy5wdWJsaXNoVG9NYXZlbixcbiAgICAgIH0pO1xuXG4gICAgICB0aGlzLmFkZFBhY2thZ2luZ1RhcmdldChcImphdmFcIiwgdGFzaywgZXh0cmFKb2JPcHRpb25zKTtcbiAgICB9XG5cbiAgICBjb25zdCBweXBpID0gb3B0aW9ucy5wdWJsaXNoVG9QeXBpID8/IG9wdGlvbnMucHl0aG9uO1xuICAgIGlmIChweXBpKSB7XG4gICAgICB0YXJnZXRzLnB5dGhvbiA9IHtcbiAgICAgICAgZGlzdE5hbWU6IHB5cGkuZGlzdE5hbWUsXG4gICAgICAgIG1vZHVsZTogcHlwaS5tb2R1bGUsXG4gICAgICB9O1xuXG4gICAgICBjb25zdCB0YXNrID0gdGhpcy5hZGRQYWNrYWdpbmdUYXNrKFwicHl0aG9uXCIpO1xuICAgICAgdGhpcy5yZWxlYXNlPy5wdWJsaXNoZXIucHVibGlzaFRvUHlQaSh7XG4gICAgICAgIC4uLnRoaXMucGFjbWFrRm9yTGFuZ3VhZ2UoXCJweXRob25cIiwgdGFzayksXG4gICAgICAgIC4uLnB5cGksXG4gICAgICB9KTtcblxuICAgICAgdGhpcy5hZGRQYWNrYWdpbmdUYXJnZXQoXCJweXRob25cIiwgdGFzaywgZXh0cmFKb2JPcHRpb25zKTtcbiAgICB9XG5cbiAgICBjb25zdCBudWdldCA9IG9wdGlvbnMucHVibGlzaFRvTnVnZXQgPz8gb3B0aW9ucy5kb3RuZXQ7XG4gICAgaWYgKG51Z2V0KSB7XG4gICAgICB0YXJnZXRzLmRvdG5ldCA9IHtcbiAgICAgICAgbmFtZXNwYWNlOiBudWdldC5kb3ROZXROYW1lc3BhY2UsXG4gICAgICAgIHBhY2thZ2VJZDogbnVnZXQucGFja2FnZUlkLFxuICAgICAgICBpY29uVXJsOiBudWdldC5pY29uVXJsLFxuICAgICAgfTtcblxuICAgICAgY29uc3QgdGFzayA9IHRoaXMuYWRkUGFja2FnaW5nVGFzayhcImRvdG5ldFwiKTtcbiAgICAgIHRoaXMucmVsZWFzZT8ucHVibGlzaGVyLnB1Ymxpc2hUb051Z2V0KHtcbiAgICAgICAgLi4udGhpcy5wYWNtYWtGb3JMYW5ndWFnZShcImRvdG5ldFwiLCB0YXNrKSxcbiAgICAgICAgLi4ubnVnZXQsXG4gICAgICB9KTtcblxuICAgICAgdGhpcy5hZGRQYWNrYWdpbmdUYXJnZXQoXCJkb3RuZXRcIiwgdGFzaywgZXh0cmFKb2JPcHRpb25zKTtcbiAgICB9XG5cbiAgICBjb25zdCBnb2xhbmcgPSBvcHRpb25zLnB1Ymxpc2hUb0dvO1xuICAgIGlmIChnb2xhbmcpIHtcbiAgICAgIHRhcmdldHMuZ28gPSB7XG4gICAgICAgIG1vZHVsZU5hbWU6IGdvbGFuZy5tb2R1bGVOYW1lLFxuICAgICAgICBwYWNrYWdlTmFtZTogZ29sYW5nLnBhY2thZ2VOYW1lLFxuICAgICAgfTtcblxuICAgICAgY29uc3QgdGFzayA9IHRoaXMuYWRkUGFja2FnaW5nVGFzayhcImdvXCIpO1xuICAgICAgdGhpcy5yZWxlYXNlPy5wdWJsaXNoZXIucHVibGlzaFRvR28oe1xuICAgICAgICAuLi50aGlzLnBhY21ha0Zvckxhbmd1YWdlKFwiZ29cIiwgdGFzayksXG4gICAgICAgIC4uLmdvbGFuZyxcbiAgICAgIH0pO1xuXG4gICAgICB0aGlzLmFkZFBhY2thZ2luZ1RhcmdldChcImdvXCIsIHRhc2ssIGV4dHJhSm9iT3B0aW9ucyk7XG4gICAgfVxuXG4gICAgY29uc3QganNpaVN1ZmZpeCA9XG4gICAgICBvcHRpb25zLmpzaWlWZXJzaW9uID09PSBcIipcIlxuICAgICAgICA/IC8vIElmIGpzaWlWZXJzaW9uIGlzIFwiKlwiLCBkb24ndCBzcGVjaWZ5IGFueXRoaW5nIHNvIHRoZSB1c2VyIGNhbiBtYW5hZ2UuXG4gICAgICAgICAgXCJcIlxuICAgICAgICA6IC8vIE90aGVyd2lzZSwgdXNlIGBqc2lpVmVyc2lvbmAgb3IgZmFsbCBiYWNrIHRvIGAxLnhgLlxuICAgICAgICAgIGBAJHtvcHRpb25zLmpzaWlWZXJzaW9uID8/IFwiMS54XCJ9YDtcbiAgICB0aGlzLmFkZERldkRlcHMoXG4gICAgICBganNpaSR7anNpaVN1ZmZpeH1gLFxuICAgICAgYGpzaWktcm9zZXR0YSR7anNpaVN1ZmZpeH1gLFxuICAgICAgXCJqc2lpLWRpZmZcIixcbiAgICAgIFwianNpaS1wYWNtYWtcIlxuICAgICk7XG5cbiAgICB0aGlzLmdpdGlnbm9yZS5leGNsdWRlKFwiLmpzaWlcIiwgXCJ0c2NvbmZpZy5qc29uXCIpO1xuICAgIHRoaXMubnBtaWdub3JlPy5pbmNsdWRlKFwiLmpzaWlcIik7XG5cbiAgICBpZiAob3B0aW9ucy5kb2NnZW4gPz8gdHJ1ZSkge1xuICAgICAgbmV3IEpzaWlEb2NnZW4odGhpcywgeyBmaWxlUGF0aDogb3B0aW9ucy5kb2NnZW5GaWxlUGF0aCB9KTtcbiAgICB9XG5cbiAgICAvLyBqc2lpIHVwZGF0ZXMgLm5wbWlnbm9yZSwgc28gd2UgbWFrZSBpdCB3cml0YWJsZVxuICAgIGlmICh0aGlzLm5wbWlnbm9yZSkge1xuICAgICAgdGhpcy5ucG1pZ25vcmUucmVhZG9ubHkgPSBmYWxzZTtcbiAgICB9XG5cbiAgICAvLyBXaGVuIHVzaW5nIGpzaWlAMS54LCB3ZSBuZWVkIHRvIGFkZCBzb21lIHJlc29sdXRpb25zIHRvIGF2b2lkIGluY2x1ZGluZ1xuICAgIC8vIFR5cGVTY3JpcHQtMy45LWluY29tcGF0aWJsZSBkZXBlbmRlbmNpZXMgdGhhdCBicmVhayB0aGUgY29tcGlsZXIuXG4gICAgaWYgKHVzZXNMZWdhY3lKc2lpKSB7XG4gICAgICAvLyBodHRwczovL2dpdGh1Yi5jb20vcHJvamVuL3Byb2plbi9pc3N1ZXMvMjE2NVxuICAgICAgdGhpcy5wYWNrYWdlLmFkZFBhY2thZ2VSZXNvbHV0aW9ucyhcIkB0eXBlcy9wcmV0dGllckAyLjYuMFwiKTtcblxuICAgICAgLy8gaHR0cHM6Ly9naXRodWIuY29tL3Byb2plbi9wcm9qZW4vaXNzdWVzLzIyNjRcbiAgICAgIHRoaXMucGFja2FnZS5hZGRQYWNrYWdlUmVzb2x1dGlvbnMoXCJAdHlwZXMvYmFiZWxfX3RyYXZlcnNlQDcuMTguMlwiKTtcblxuICAgICAgY29uc3QganNpaVZlcnNpb24gPSBvcHRpb25zLmpzaWlWZXJzaW9uID8/IFwiMS54XCI7XG4gICAgICBpZiAoanNpaVZlcnNpb24uc3RhcnRzV2l0aChcIjEuXCIpKSB7XG4gICAgICAgIGNvbnN0IG1ham9yTm9kZVZlcnNpb24gPSBtYWpvcih0aGlzLnBhY2thZ2UubWluTm9kZVZlcnNpb24gPz8gXCIxNi4wLjBcIik7XG5cbiAgICAgICAgLy8gc2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9wcm9qZW4vcHJvamVuL2lzc3Vlcy8zMzI0XG4gICAgICAgIGNvbnN0IG5vZGVUeXBlc1ZlcnNpb25SYW5nZSA9IChtYWpvclZlcnNpb246IG51bWJlcik6IHN0cmluZyA9PiB7XG4gICAgICAgICAgc3dpdGNoIChtYWpvclZlcnNpb24pIHtcbiAgICAgICAgICAgIGNhc2UgMTY6XG4gICAgICAgICAgICAgIHJldHVybiBgXjE2IDw9IDE2LjE4Ljc4YDtcbiAgICAgICAgICAgIGNhc2UgMTg6XG4gICAgICAgICAgICAgIHJldHVybiBgXjE4IDw9IDE4LjExLjE5YDtcbiAgICAgICAgICAgIGNhc2UgMTk6XG4gICAgICAgICAgICBjYXNlIDIwOlxuICAgICAgICAgICAgY2FzZSAyMTpcbiAgICAgICAgICAgIGNhc2UgMjI6XG4gICAgICAgICAgICAgIHRoaXMubG9nZ2VyLndhcm4oXG4gICAgICAgICAgICAgICAgYGpzaWlAJHtqc2lpVmVyc2lvbn0gYW5kIEB0eXBlcy9ub2RlQF4ke21ham9yVmVyc2lvbn0gYXJlIGluY29tcGF0aWJsZS4gRmFsbGluZyBiYWNrIHRvIEB0eXBlcy9ub2RlQF4xOC5gLFxuICAgICAgICAgICAgICAgIFwiUGxlYXNlIHVwZ3JhZGUgdG8gYSBtb2Rlcm4gdmVyc2lvbiBvZiBqc2lpLlwiXG4gICAgICAgICAgICAgICk7XG4gICAgICAgICAgICAgIHJldHVybiBgXjE4IDw9IDE4LjExLjE5YDtcbiAgICAgICAgICAgIGRlZmF1bHQ6XG4gICAgICAgICAgICAgIHJldHVybiBgXiR7bWFqb3JWZXJzaW9ufWA7XG4gICAgICAgICAgfVxuICAgICAgICB9O1xuXG4gICAgICAgIHRoaXMuYWRkRGV2RGVwcyhcbiAgICAgICAgICAvLyBodHRwczovL2dpdGh1Yi5jb20vcHJvamVuL3Byb2plbi9wdWxsLzMwNzZcbiAgICAgICAgICBgQHR5cGVzL25vZGVAJHtub2RlVHlwZXNWZXJzaW9uUmFuZ2UobWFqb3JOb2RlVmVyc2lvbil9YFxuICAgICAgICApO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgdGFyZ2V0IGxhbmd1YWdlIHRvIHRoZSBidWlsZCB3b3JrZmxvdyBhbmQgY3JlYXRlcyBhIHBhY2thZ2UgdGFzay5cbiAgICogQHBhcmFtIGxhbmd1YWdlXG4gICAqIEByZXR1cm5zXG4gICAqL1xuICBwcml2YXRlIGFkZFBhY2thZ2luZ1RhcmdldChcbiAgICBsYW5ndWFnZTogSnNpaVBhY21ha1RhcmdldCxcbiAgICBwYWNrVGFzazogVGFzayxcbiAgICBleHRyYUpvYk9wdGlvbnM6IFBhcnRpYWw8Sm9iPlxuICApIHtcbiAgICBpZiAoIXRoaXMuYnVpbGRXb3JrZmxvdykge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICBjb25zdCBwYWNtYWsgPSB0aGlzLnBhY21ha0Zvckxhbmd1YWdlKGxhbmd1YWdlLCBwYWNrVGFzayk7XG5cbiAgICB0aGlzLmJ1aWxkV29ya2Zsb3cuYWRkUG9zdEJ1aWxkSm9iKGBwYWNrYWdlLSR7bGFuZ3VhZ2V9YCwge1xuICAgICAgLi4uZmlsdGVyZWRSdW5zT25PcHRpb25zKFxuICAgICAgICBleHRyYUpvYk9wdGlvbnMucnVuc09uLFxuICAgICAgICBleHRyYUpvYk9wdGlvbnMucnVuc09uR3JvdXBcbiAgICAgICksXG4gICAgICBwZXJtaXNzaW9uczoge30sXG4gICAgICB0b29sczoge1xuICAgICAgICBub2RlOiB7IHZlcnNpb246IHRoaXMubm9kZVZlcnNpb24gPz8gXCIxOC54XCIgfSxcbiAgICAgICAgLi4ucGFjbWFrLnB1Ymxpc2hUb29scyxcbiAgICAgIH0sXG4gICAgICBzdGVwczogcGFjbWFrLnByZVB1Ymxpc2hTdGVwcyA/PyBbXSxcbiAgICAgIC4uLmV4dHJhSm9iT3B0aW9ucyxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkUGFja2FnaW5nVGFzayhsYW5ndWFnZTogSnNpaVBhY21ha1RhcmdldCk6IFRhc2sge1xuICAgIGNvbnN0IHBhY2thZ2VUYXNrID0gdGhpcy50YXNrcy5hZGRUYXNrKGBwYWNrYWdlOiR7bGFuZ3VhZ2V9YCwge1xuICAgICAgZGVzY3JpcHRpb246IGBDcmVhdGUgJHtsYW5ndWFnZX0gbGFuZ3VhZ2UgYmluZGluZ3NgLFxuICAgIH0pO1xuICAgIGNvbnN0IGNvbW1hbmRQYXJ0cyA9IFtcImpzaWktcGFjbWFrXCIsIFwiLXZcIl07XG5cbiAgICBpZiAodGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyID09PSBOb2RlUGFja2FnZU1hbmFnZXIuUE5QTSkge1xuICAgICAgY29tbWFuZFBhcnRzLnB1c2goXCItLXBhY2stY29tbWFuZCAncG5wbSBwYWNrJ1wiKTtcbiAgICB9XG5cbiAgICBjb21tYW5kUGFydHMucHVzaChgLS10YXJnZXQgJHtsYW5ndWFnZX1gKTtcblxuICAgIHBhY2thZ2VUYXNrLmV4ZWMoY29tbWFuZFBhcnRzLmpvaW4oXCIgXCIpKTtcblxuICAgIHRoaXMucGFja2FnZUFsbFRhc2suc3Bhd24ocGFja2FnZVRhc2spO1xuICAgIHJldHVybiBwYWNrYWdlVGFzaztcbiAgfVxuXG4gIHByaXZhdGUgcGFjbWFrRm9yTGFuZ3VhZ2UoXG4gICAgdGFyZ2V0OiBKc2lpUGFjbWFrVGFyZ2V0LFxuICAgIHBhY2tUYXNrOiBUYXNrXG4gICk6IENvbW1vblB1Ymxpc2hPcHRpb25zIHtcbiAgICAvLyBhdCB0aGlzIHN0YWdlLCBgYXJ0aWZhY3RzRGlyZWN0b3J5YCBjb250YWlucyB0aGUgcHJlYnVpbHQgcmVwb3NpdG9yeS5cbiAgICAvLyBmb3IgdGhlIHB1Ymxpc2hpbmcgdG8gd29yayBzZWFtbGVzc2VseSwgdGhhdCBkaXJlY3RvcnkgbmVlZHMgdG8gY29udGFpbiB0aGUgYWN0dWFsIGFydGlmYWN0LlxuICAgIC8vIHNvIHdlIG1vdmUgdGhlIHJlcG8sIGNyZWF0ZSB0aGUgYXJ0aWZhY3QsIGFuZCBwdXQgaXQgaW4gdGhlIGV4cGVjdGVkIHBsYWNlLlxuICAgIGNvbnN0IHByZVB1Ymxpc2hTdGVwczogQXJyYXk8U3RlcD4gPSBbXTtcblxuICAgIHByZVB1Ymxpc2hTdGVwcy5wdXNoKC4uLnRoaXMud29ya2Zsb3dCb290c3RyYXBTdGVwcyk7XG5cbiAgICBpZiAodGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyID09PSBOb2RlUGFja2FnZU1hbmFnZXIuUE5QTSkge1xuICAgICAgcHJlUHVibGlzaFN0ZXBzLnB1c2goe1xuICAgICAgICBuYW1lOiBcIlNldHVwIHBucG1cIixcbiAgICAgICAgdXNlczogXCJwbnBtL2FjdGlvbi1zZXR1cEB2M1wiLFxuICAgICAgICB3aXRoOiB7IHZlcnNpb246IHRoaXMucGFja2FnZS5wbnBtVmVyc2lvbiB9LFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgcHJlUHVibGlzaFN0ZXBzLnB1c2goXG4gICAgICB7XG4gICAgICAgIG5hbWU6IFwiUHJlcGFyZSBSZXBvc2l0b3J5XCIsXG4gICAgICAgIHJ1bjogYG12ICR7dGhpcy5hcnRpZmFjdHNEaXJlY3Rvcnl9ICR7UkVQT19URU1QX0RJUkVDVE9SWX1gLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgbmFtZTogXCJJbnN0YWxsIERlcGVuZGVuY2llc1wiLFxuICAgICAgICBydW46IGBjZCAke1JFUE9fVEVNUF9ESVJFQ1RPUll9ICYmICR7dGhpcy5wYWNrYWdlLmluc3RhbGxDb21tYW5kfWAsXG4gICAgICB9LFxuICAgICAge1xuICAgICAgICBuYW1lOiBgQ3JlYXRlICR7dGFyZ2V0fSBhcnRpZmFjdGAsXG4gICAgICAgIHJ1bjogYGNkICR7UkVQT19URU1QX0RJUkVDVE9SWX0gJiYgbnB4IHByb2plbiAke3BhY2tUYXNrLm5hbWV9YCxcbiAgICAgIH0sXG4gICAgICB7XG4gICAgICAgIG5hbWU6IGBDb2xsZWN0ICR7dGFyZ2V0fSBBcnRpZmFjdGAsXG4gICAgICAgIHJ1bjogYG12ICR7UkVQT19URU1QX0RJUkVDVE9SWX0vJHt0aGlzLmFydGlmYWN0c0RpcmVjdG9yeX0gJHt0aGlzLmFydGlmYWN0c0RpcmVjdG9yeX1gLFxuICAgICAgfVxuICAgICk7XG4gICAgcmV0dXJuIHtcbiAgICAgIHB1Ymxpc2hUb29sczogSlNJSV9UT09MQ0hBSU5bdGFyZ2V0XSxcbiAgICAgIHByZVB1Ymxpc2hTdGVwcyxcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIEdlbmVyYXRlcyB0aGUgcnVucy1vbiBjb25maWcgZm9yIEpvYnMuXG4gICAqIFRocm93cyBlcnJvciBpZiAncnVuc09uJyBhbmQgJ3J1bnNPbkdyb3VwJyBhcmUgYm90aCBzZXQuXG4gICAqXG4gICAqIEBwYXJhbSBvcHRpb25zIC0gJ3J1bnNPbicgb3IgJ3J1bnNPbkdyb3VwJy5cbiAgICovXG4gIHByaXZhdGUgZ2V0Sm9iUnVuc09uQ29uZmlnKG9wdGlvbnM6IEpzaWlQcm9qZWN0T3B0aW9ucykge1xuICAgIHJldHVybiBvcHRpb25zLndvcmtmbG93UnVuc09uR3JvdXBcbiAgICAgID8geyBydW5zT25Hcm91cDogb3B0aW9ucy53b3JrZmxvd1J1bnNPbkdyb3VwIH1cbiAgICAgIDogb3B0aW9ucy53b3JrZmxvd1J1bnNPblxuICAgICAgPyB7IHJ1bnNPbjogb3B0aW9ucy53b3JrZmxvd1J1bnNPbiB9XG4gICAgICA6IHt9O1xuICB9XG59XG5cbmZ1bmN0aW9uIHBhcnNlQXV0aG9yQWRkcmVzcyhvcHRpb25zOiBKc2lpUHJvamVjdE9wdGlvbnMpIHtcbiAgbGV0IGF1dGhvckVtYWlsID0gb3B0aW9ucy5hdXRob3JFbWFpbDtcbiAgbGV0IGF1dGhvclVybCA9IG9wdGlvbnMuYXV0aG9yVXJsO1xuICBpZiAob3B0aW9ucy5hdXRob3JBZGRyZXNzKSB7XG4gICAgaWYgKG9wdGlvbnMuYXV0aG9yRW1haWwgJiYgb3B0aW9ucy5hdXRob3JFbWFpbCAhPT0gb3B0aW9ucy5hdXRob3JBZGRyZXNzKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIFwiYXV0aG9yRW1haWwgaXMgZGVwcmVjYXRlZCBhbmQgY2Fubm90IGJlIHVzZWQgaW4gY29uanVuY3Rpb24gd2l0aCBhdXRob3JBZGRyZXNzXCJcbiAgICAgICk7XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMuYXV0aG9yVXJsICYmIG9wdGlvbnMuYXV0aG9yVXJsICE9PSBvcHRpb25zLmF1dGhvckFkZHJlc3MpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgXCJhdXRob3JVcmwgaXMgZGVwcmVjYXRlZCBhbmQgY2Fubm90IGJlIHVzZWQgaW4gY29uanVuY3Rpb24gd2l0aCBhdXRob3JBZGRyZXNzLlwiXG4gICAgICApO1xuICAgIH1cblxuICAgIGlmIChFTUFJTF9SRUdFWC50ZXN0KG9wdGlvbnMuYXV0aG9yQWRkcmVzcykpIHtcbiAgICAgIGF1dGhvckVtYWlsID0gb3B0aW9ucy5hdXRob3JBZGRyZXNzO1xuICAgIH0gZWxzZSBpZiAoVVJMX1JFR0VYLnRlc3Qob3B0aW9ucy5hdXRob3JBZGRyZXNzKSkge1xuICAgICAgYXV0aG9yVXJsID0gb3B0aW9ucy5hdXRob3JBZGRyZXNzO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIGBhdXRob3JBZGRyZXNzIG11c3QgYmUgZWl0aGVyIGFuIGVtYWlsIGFkZHJlc3Mgb3IgYSBVUkw6ICR7b3B0aW9ucy5hdXRob3JBZGRyZXNzfWBcbiAgICAgICk7XG4gICAgfVxuICB9XG4gIHJldHVybiB7IGF1dGhvckVtYWlsLCBhdXRob3JVcmwgfTtcbn1cbiJdfQ==