"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cdk8sDeps = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const semver = require("semver");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
/**
 * Manages dependencies on the CDK8s.
 */
class Cdk8sDeps extends component_1.Component {
    constructor(project, options) {
        super(project);
        this.dependencyType = options.dependencyType;
        this._packageNames = this.packageNames();
        const framework = determineFrameworkVersion(options);
        this.cdk8sVersion = framework.range;
        this.cdk8sMajorVersion = framework.major;
        this.cdk8sMinimumVersion = framework.minimum;
        // constructs library
        this.addConstructsDependency(options.constructsVersion, options.constructsVersionPinning);
        this.addFrameworkDependency(options.cdk8sVersion, options.cdk8sVersionPinning);
        if (options.cdk8sCliDependency) {
            this.addClientDependency(options.cdk8sCliVersion, options.cdk8sCliVersionPinning);
        }
        if (options.cdk8sPlus ?? true) {
            const k8MinorVersion = options.k8sMinorVersion ?? 22;
            this.addCdk8sPlusDependency(k8MinorVersion, options.cdk8sPlusVersion, 
            // (project instanceof Cdk8sPythonApp) ? true : false,
            options.cdk8sPlusVersionPinning);
        }
    }
    getVersionRequirement(requestedVersion, v1Version, v2Version, pinning) {
        const defaultVersion = this.cdk8sMajorVersion === 1 ? v1Version : v2Version;
        const version = `${requestedVersion ?? defaultVersion}`;
        const versionRequirement = pinning ? `${version}` : `^${version}`;
        return versionRequirement;
    }
    addCdk8sPlusDependency(k8MinorVersion, requestedVersion, pinning) {
        const versionRequirement = this.getVersionRequirement(requestedVersion, "1.0.0-beta.222", "2.0.0-rc.26", pinning);
        const cdk8sPlusMajorVersion = semver.minVersion(versionRequirement)?.major;
        if (!cdk8sPlusMajorVersion) {
            throw new Error(`Cannot determine major version of cdk8s-plus version '${versionRequirement}'`);
        }
        const cdk8sPlusLib = this._packageNames.cdk8sPlus.concat("-", k8MinorVersion.toString());
        this.project.deps.addDependency(`${cdk8sPlusLib}@${versionRequirement}`, this.dependencyType);
        return versionRequirement;
    }
    addClientDependency(requestedVersion, pinning) {
        const versionRequirement = this.getVersionRequirement(requestedVersion, "1.1.34", "2.0.28", pinning);
        const cdk8sCliMajorVersion = semver.minVersion(versionRequirement)?.major;
        if (!cdk8sCliMajorVersion) {
            throw new Error(`Cannot determine major version of cdk8s version '${versionRequirement}'`);
        }
        this.project.deps.addDependency(`${this._packageNames.cdk8sClient}@${versionRequirement}`, dependencies_1.DependencyType.BUILD);
        return versionRequirement;
    }
    addFrameworkDependency(requestedVersion, pinning) {
        const versionRequirement = this.getVersionRequirement(requestedVersion, "1.6.33", "2.3.33", pinning);
        const cdk8sMajorVersion = semver.minVersion(versionRequirement)?.major;
        if (!cdk8sMajorVersion) {
            throw new Error(`Cannot determine major version of cdk8s version '${versionRequirement}'`);
        }
        this.project.deps.addDependency(`${this._packageNames.cdk8s}@${versionRequirement}`, this.dependencyType);
        return versionRequirement;
    }
    addConstructsDependency(requestedVersion, pinning) {
        if (requestedVersion && !semver.parse(requestedVersion)) {
            throw new Error(`"constructsVersion" cannot be parsed as a semver version: ${requestedVersion}`);
        }
        const versionRequirement = this.getVersionRequirement(requestedVersion, "3.4.39", "10.1.42", pinning);
        const constructsMajorVersion = semver.minVersion(versionRequirement)?.major;
        if (!constructsMajorVersion) {
            throw new Error(`Cannot determine major version of constructs version '${versionRequirement}'`);
        }
        switch (this.cdk8sMajorVersion) {
            case 1:
                if (constructsMajorVersion !== 3) {
                    throw new Error("AWS CDK 1.x requires constructs 3.x");
                }
                break;
            case 2:
                if (constructsMajorVersion !== 10) {
                    throw new Error("AWS CDK 2.x requires constructs 10.x");
                }
                break;
        }
        this.project.deps.addDependency(`${this._packageNames.constructs}@${versionRequirement}`, this.dependencyType);
        return versionRequirement;
    }
}
exports.Cdk8sDeps = Cdk8sDeps;
_a = JSII_RTTI_SYMBOL_1;
Cdk8sDeps[_a] = { fqn: "projen.cdk8s.Cdk8sDeps", version: "0.79.27" };
function determineFrameworkVersion(options) {
    const ver = semver.parse(options.cdk8sVersion);
    if (!ver) {
        throw new Error(`"cdk8sVersion" cannot be parsed as a semver version: ${options.cdk8sVersion}`);
    }
    return {
        minimum: ver.format(),
        range: options.cdk8sVersionPinning
            ? options.cdk8sVersion
            : `^${options.cdk8sVersion}`,
        major: ver.major,
    };
}
//# sourceMappingURL=data:application/json;base64,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