"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const semver = require("semver");
const yargs = require("yargs");
const inventory = require("../../inventory");
const logging = require("../../logging");
const option_hints_1 = require("../../option-hints");
const projects_1 = require("../../projects");
const util_1 = require("../../util");
const macros_1 = require("../macros");
const util_2 = require("../util");
class Command {
    constructor() {
        this.command = "new [PROJECT-TYPE-NAME] [OPTIONS]";
        this.describe = "Creates a new projen project";
    }
    builder(args) {
        args.positional("PROJECT-TYPE-NAME", {
            describe: "optional only when --from is used and there is a single project type in the external module",
            type: "string",
        });
        args.option("synth", {
            type: "boolean",
            default: true,
            desc: "Synthesize after creating .projenrc.js",
        });
        args.option("comments", {
            type: "boolean",
            default: true,
            desc: "Include commented out options in .projenrc.js (use --no-comments to disable)",
        });
        args.option("from", {
            type: "string",
            alias: "f",
            desc: 'External jsii npm module to create project from. Supports any package spec supported by npm (such as "my-pack@^2.0")',
        });
        args.option("git", {
            type: "boolean",
            default: true,
            desc: "Run `git init` and create an initial commit (use --no-git to disable)",
        });
        args.example("projen new awscdk-app-ts", 'Creates a new project of built-in type "awscdk-app-ts"');
        args.example("projen new --from projen-vue@^2", 'Creates a new project from an external module "projen-vue" with the specified version');
        for (const type of inventory.discover()) {
            args.command(type.pjid, type.docs ?? "", {
                builder: (cargs) => {
                    cargs.showHelpOnFail(false);
                    for (const option of type.options ?? []) {
                        // not all types can be represented in the cli
                        if (!argTypeSupported(option)) {
                            continue;
                        }
                        const defaultValue = argInitialValue(option);
                        cargs.option(option.switch, {
                            group: !option.optional ? "Required:" : "Optional:",
                            type: argType(option),
                            description: argDesc(option),
                            required: !option.optional,
                            // yargs behaves differently for arrays if the defaultValue property is present or not
                            ...(!option.optional && defaultValue
                                ? { default: defaultValue }
                                : {}),
                        });
                    }
                    return cargs;
                },
                handler: (argv) => initProject(process.cwd(), type, argv),
            });
        }
        // Disable strict mode, otherwise the catch-all doesn't work
        args.strictCommands(false);
        args
            .command({
            command: "*",
            describe: false,
            handler,
        })
            .middleware((argv) => {
            // manually set the matched command as the project type
            argv.projectTypeName = argv._[1];
        }, true);
        return args;
    }
    async handler(args) {
        return handler(args);
    }
}
async function handler(args) {
    try {
        // handle --from which means we want to first install a jsii module and then
        // create a project defined within this module.
        if (args.from) {
            return await initProjectFromModule(process.cwd(), args.from, args);
        }
        // project type is defined but was not matched by yargs, so print the list of supported types
        if (args.projectTypeName) {
            const types = inventory.discover();
            throw new util_2.CliError(`Project type "${args.projectTypeName}" not found. Available types:\n`, ...types.map((t) => `    ${t.pjid}`), "", `Please specify a project type.`, `Example: npx projen new ${types[0].pjid}`);
        }
        // Handles the use case that nothing was specified since PROJECT-TYPE is now an optional positional parameter
        yargs.showHelp();
    }
    catch (error) {
        if (error instanceof util_2.CliError) {
            logging.error(error.message);
            logging.empty();
            process.exitCode = 1;
            return;
        }
        // unknown error, likely a node runtime exception in project code
        // rethrow so the full stack trace is displayed
        throw error;
    }
}
/**
 * Returns the yargs option type for a given project option
 */
function argType(option) {
    if (option.kind === "enum") {
        return "string";
    }
    if (isPrimitiveArrayOption(option)) {
        return "array";
    }
    return option.simpleType;
}
/**
 * Returns the description for a given project option
 */
function argDesc(option) {
    let desc = [option.docs?.replace(/\ *\.$/, "") ?? ""];
    const helpDefault = option.initialValue ?? option.default;
    if (option.optional && helpDefault) {
        desc.push(`[default: ${helpDefault.replace(/^\ *-/, "").replace(/\.$/, "").trim()}]`);
    }
    return desc.join(" ");
}
/**
 * Compute the initial value for a given project option
 */
function argInitialValue(option, cwd = process.cwd()) {
    // if we have determined an initial value for the field
    // we can show that value in --help
    if (option.initialValue) {
        return renderDefault(cwd, option.initialValue);
    }
}
/**
 * Currently we only support these field types as command line options:
 * - primitives (string, number, boolean)
 * - lists of primitives
 * - enums
 */
function argTypeSupported(option) {
    return (option.simpleType === "string" ||
        option.simpleType === "number" ||
        option.simpleType === "boolean" ||
        option.kind === "enum" ||
        isPrimitiveArrayOption(option));
}
/**
 * Checks if the given option is a primitive array
 */
function isPrimitiveArrayOption(option) {
    return Boolean(option.jsonLike &&
        option.fullType.collection?.kind === "array" &&
        option.fullType.collection.elementtype.primitive &&
        ["string", "number"].includes(option.fullType.collection.elementtype.primitive));
}
/**
 * Given a value from "@default", processes macros and returns a stringified
 * (quoted) result.
 *
 * @returns a javascript primitive (could be a string, number or boolean)
 */
function renderDefault(cwd, value) {
    return (0, macros_1.tryProcessMacro)(cwd, value) ?? JSON.parse(value);
}
/**
 * Converts yargs command line switches to project type props.
 * @param type Project type
 * @param argv Command line switches
 */
function commandLineToProps(cwd, type, argv) {
    const props = {};
    // initialize props with default values
    for (const prop of type.options) {
        props[prop.name] = argInitialValue(prop, cwd);
    }
    for (const [arg, value] of Object.entries(argv)) {
        for (const prop of type.options) {
            if (prop.switch === arg) {
                let curr = props;
                const queue = [...prop.path];
                while (true) {
                    const p = queue.shift();
                    if (!p) {
                        break;
                    }
                    if (queue.length === 0) {
                        curr[p] = value;
                    }
                    else {
                        curr[p] = curr[p] ?? {};
                        curr = curr[p];
                    }
                }
            }
        }
    }
    return props;
}
/**
 * Generates a new project from an external module.
 *
 * @param spec The name of the external module to load
 * @param args Command line arguments (incl. project type)
 */
async function initProjectFromModule(baseDir, spec, args) {
    const projenVersion = args.projenVersion ?? "latest";
    const installCommand = (0, util_2.renderInstallCommand)(baseDir, `projen@${projenVersion}`);
    if (args.projenVersion) {
        (0, util_1.exec)(installCommand, { cwd: baseDir });
    }
    else {
        // do not overwrite existing installation
        (0, util_1.exec)(`npm ls --prefix="${baseDir}" --depth=0 --pattern projen || ${installCommand}`, { cwd: baseDir });
    }
    const moduleName = (0, util_2.installPackage)(baseDir, spec);
    logging.empty();
    // Find the just installed package and discover the rest recursively from this package folder
    const moduleDir = path.dirname(require.resolve(`${moduleName}/.jsii`, {
        paths: [baseDir],
    }));
    // Only leave projects from the main (requested) package
    const projects = inventory
        .discover(moduleDir)
        .filter((x) => x.moduleName === moduleName); // Only list project types from the requested 'from' module
    if (projects.length < 1) {
        throw new util_2.CliError(`No project types found after installing "${spec}". The module must export at least one class which extends "projen.Project".`);
    }
    const requested = args.projectTypeName;
    const types = projects.map((p) => p.pjid);
    // if user did not specify a project type but the module has more than one, we need them to tell us which one...
    if (!requested && projects.length > 1) {
        throw new util_2.CliError(`Multiple project types found after installing "${spec}":\n`, ...types.map((t) => `    ${t}`), "", `Please specify a project type.`, `Example: npx projen new --from ${spec} ${types[0]}`);
    }
    // if user did not specify a type (and we know we have only one), the select it. otherwise, search by pjid.
    const type = !requested
        ? projects[0]
        : projects.find((p) => p.pjid === requested);
    if (!type) {
        throw new util_2.CliError(`Project type "${requested}" not found in "${spec}". Found:\n`, ...types.map((t) => `    ${t}`), "", `Please specify a valid project type.`, `Example: npx projen new --from ${spec} ${types[0]}`);
    }
    const missingOptions = [];
    for (const option of type.options ?? []) {
        // not all types can be represented in the cli
        if (!argTypeSupported(option)) {
            continue;
        }
        // parse allowed types
        if (args[option.name] !== undefined) {
            args[option.name] = parseArg(args[option.name], argType(option), option);
            args[option.switch] = args[option.name];
            continue;
        }
        // Required option with a default
        if (!option.optional && option.default && option.default !== "undefined") {
            const defaultValue = renderDefault(baseDir, option.default);
            args[option.name] = defaultValue;
            args[option.switch] = defaultValue;
        }
        // Required option, but we could not find a value
        if (!option.optional && !args[option.name]) {
            missingOptions.push(`--${option.switch} [${argType(option)}] ${argDesc(option)}`);
        }
    }
    // We are missing some required options
    if (missingOptions.length) {
        throw new util_2.CliError(`Cannot create "${type.fqn}". Missing required option${missingOptions.length > 1 ? "s" : ""}:`, ...missingOptions.map((m) => `    ${m}`));
    }
    // include a dev dependency for the external module
    args.devDeps = [spec];
    args["dev-deps"] = [spec];
    await initProject(baseDir, type, args);
}
/**
 * Parse command line value as option type
 */
function parseArg(value, type, option) {
    switch (type) {
        case "number":
            return parseInt(value);
        case "boolean":
            return typeof value === "string" ? (0, util_1.isTruthy)(value) : value;
        case "array":
            if (!Array.isArray(value)) {
                value = [value];
            }
            return value.map((v) => parseArg(v, option?.fullType.collection?.elementtype.primitive || "string"));
        // return value unchanged
        case "string":
        default:
            // if we have an unexpected array, use the first element
            if (Array.isArray(value)) {
                return value[0];
            }
            return value;
    }
}
/**
 * Generates a new project.
 * @param type Project type
 * @param args Command line arguments
 * @param additionalProps Additional parameters to include in .projenrc.js
 */
async function initProject(baseDir, type, args) {
    // convert command line arguments to project props using type information
    const props = commandLineToProps(baseDir, type, args);
    projects_1.Projects.createProject({
        dir: props.outdir ?? baseDir,
        projectFqn: type.fqn,
        projectOptions: props,
        optionHints: args.comments
            ? option_hints_1.InitProjectOptionHints.FEATURED
            : option_hints_1.InitProjectOptionHints.NONE,
        synth: args.synth,
        post: args.post,
    });
    if (fs.existsSync(path.join(baseDir, "package.json")) && args.post) {
        (0, util_1.exec)("npm run eslint --if-present", { cwd: baseDir });
    }
    if (args.git) {
        const git = (cmd) => (0, util_1.exec)(`git ${cmd}`, { cwd: baseDir });
        const gitversion = (0, util_1.getGitVersion)((0, util_1.execCapture)("git --version", { cwd: baseDir }).toString());
        logging.debug("system using git version ", gitversion);
        // `git config init.defaultBranch` and `git init -b` are only available since git 2.28.0
        if (gitversion && semver.gte(gitversion, "2.28.0")) {
            const defaultGitInitBranch = (0, util_1.execOrUndefined)("git config init.defaultBranch", {
                cwd: baseDir,
            })?.trim() || "main";
            git(`init -b ${defaultGitInitBranch}`);
            git("add .");
            git('commit --allow-empty -m "chore: project created with projen"');
            logging.debug(`default branch name set to ${defaultGitInitBranch}`);
        }
        else {
            git("init");
            git("add .");
            git('commit --allow-empty -m "chore: project created with projen"');
            logging.debug("older version of git detected, changed default branch name to main");
            git("branch -M main");
        }
    }
}
exports.default = new Command();
//# sourceMappingURL=data:application/json;base64,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