"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const yargs = require("yargs");
const new_1 = require("./cmds/new");
const synth_1 = require("./synth");
const tasks_1 = require("./tasks");
const common_1 = require("../common");
const logging = require("../logging");
const task_runtime_1 = require("../task-runtime");
const util_1 = require("../util");
const DEFAULT_RC = (0, path_1.resolve)(common_1.DEFAULT_PROJEN_RC_JS_FILENAME);
async function main() {
    const ya = yargs;
    ya.command(new_1.default);
    const pathToProjenDir = (0, util_1.findUp)(common_1.PROJEN_DIR, process.cwd());
    const runtime = new task_runtime_1.TaskRuntime(pathToProjenDir ?? ".");
    (0, tasks_1.discoverTaskCommands)(runtime, ya);
    ya.recommendCommands();
    ya.strictCommands();
    ya.showHelpOnFail(false);
    ya.wrap(yargs.terminalWidth());
    ya.option("post", {
        type: "boolean",
        default: true,
        desc: "Run post-synthesis steps such as installing dependencies. Use --no-post to skip",
    });
    ya.option("watch", {
        type: "boolean",
        default: false,
        desc: "Keep running and resynthesize when projenrc changes",
        alias: "w",
    });
    ya.options("debug", { type: "boolean", default: false, desc: "Debug logs" });
    ya.options("rc", {
        deprecated: true,
        desc: "path to .projenrc.js file",
        default: DEFAULT_RC,
        type: "string",
    });
    ya.completion();
    ya.help();
    // do not use the default yargs '--version' implementation since it is
    // global by default (it appears on all subcommands)
    ya.version(false);
    ya.option("version", {
        type: "boolean",
        description: "Show version number",
        global: false,
    });
    const args = await ya.argv;
    if (args.debug) {
        process.env.DEBUG = "true";
    }
    const nodeVersion = (0, util_1.getNodeMajorVersion)();
    if (nodeVersion && nodeVersion < 16) {
        logging.warn(`WARNING: You are using Node v${nodeVersion}, which reaches end of life on April 30, 2023. Support for EOL Node releases may be dropped by projen in the future. Please consider upgrading to Node >= 16 as soon as possible.`);
    }
    // no command means synthesize
    if (args._.length === 0) {
        if (args.version) {
            console.log(common_1.PROJEN_VERSION);
            process.exit(0);
        }
        await (0, synth_1.synth)(runtime, {
            post: args.post,
            watch: args.watch,
            rcfile: args.rc,
        });
    }
}
main().catch((e) => {
    console.error(e.stack);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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