"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.synth = void 0;
const child_process_1 = require("child_process");
const fs = require("fs");
const os = require("os");
const path = require("path");
const common_1 = require("../common");
const logging = require("../logging");
const project_1 = require("../project");
const projenModule = path.dirname(require.resolve("../../package.json"));
async function synth(runtime, options) {
    const workdir = runtime.workdir;
    const rcfile = path.resolve(workdir, options.rcfile ?? common_1.DEFAULT_PROJEN_RC_JS_FILENAME); // TODO: support non javascript projenrc (e.g. java projects)
    // if --rc points to .projenrc.js, then behave as if it wasn't specified.
    if (rcfile === path.resolve(workdir, common_1.DEFAULT_PROJEN_RC_JS_FILENAME)) {
        delete options.rcfile;
    }
    // if there are no tasks, we assume this is not a projen project (modern
    // projects must at least have the "default" task).
    if (runtime.tasks.length === 0 && !fs.existsSync(rcfile)) {
        logging.error('Unable to find projen project. Use "projen new" to create a new project.');
        process.exit(1);
    }
    // run synth once
    const success = await trySynth();
    if (options.watch) {
        // if we are in watch mode, start the watch loop
        watchLoop();
    }
    else if (!success) {
        // make sure exit code is non-zero if we are not in watch mode
        process.exit(1);
    }
    async function trySynth() {
        // determine if post synthesis tasks should be executed (e.g. "yarn install").
        process.env.PROJEN_DISABLE_POST = (!options.post).toString();
        try {
            const defaultTask = runtime.tasks.find((t) => t.name === project_1.Project.DEFAULT_TASK);
            // if "--rc" is specified, ignore the default task
            if (defaultTask) {
                if (!options.rcfile) {
                    runtime.runTask(defaultTask.name);
                    return true;
                }
                else {
                    logging.warn("Default task skipped. Trying legacy synthesis since --rc is specified");
                }
            }
            // for backwards compatibility, if there is a .projenrc.js file, default to "node .projenrc.js"
            if (tryLegacySynth()) {
                return true;
            }
            throw new Error('Unable to find a task named "default"');
        }
        catch (e) {
            logging.error(`Synthesis failed: ${e.message}`);
            return false;
        }
    }
    function watchLoop() {
        logging.info(`Watching for changes in ${workdir}...`);
        const watch = fs.watch(workdir, { recursive: true });
        watch.on("change", (event) => {
            // we only care about "change" events
            if (event !== "change") {
                return;
            }
            process.stdout.write("\x1Bc"); // clear screen
            watch.close();
            trySynth()
                .then(() => watchLoop())
                .catch(() => watchLoop());
        });
    }
    function tryLegacySynth() {
        const rcdir = path.dirname(rcfile);
        if (!fs.existsSync(rcfile)) {
            return false;
        }
        // if node_modules/projen is not a directory or does not exist, create a
        // temporary symlink to the projen that we are currently running in order to
        // allow .projenrc.js to `require()` it.
        const nodeModules = path.resolve(rcdir, "node_modules");
        const projenModulePath = path.resolve(nodeModules, "projen");
        if (!fs.existsSync(path.join(projenModulePath, "package.json")) ||
            !fs.statSync(projenModulePath).isDirectory()) {
            fs.rmSync(projenModulePath, { force: true, recursive: true });
            fs.mkdirSync(nodeModules, { recursive: true });
            fs.symlinkSync(projenModule, projenModulePath, os.platform() === "win32" ? "junction" : null);
        }
        const ret = (0, child_process_1.spawnSync)(process.execPath, [rcfile], {
            stdio: ["inherit", "inherit", "pipe"],
        });
        if (ret.error) {
            throw new Error(`Synthesis failed: ${ret.error}`);
        }
        else if (ret.status !== 0) {
            logging.error(ret.stderr.toString());
            throw new Error(`Synthesis failed: calling "${process.execPath} ${rcfile}" exited with status=${ret.status}`);
        }
        return true;
    }
}
exports.synth = synth;
//# sourceMappingURL=data:application/json;base64,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