"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.discoverTaskCommands = void 0;
const chalk = require("chalk");
const helpers_1 = require("yargs/helpers");
const logging = require("../logging");
/**
 * Reads .projen/tasks.json and adds CLI commands for all tasks.
 * @param ya yargs
 */
function discoverTaskCommands(runtime, ya) {
    const tasks = runtime.manifest.tasks ?? {};
    for (const task of Object.values(tasks)) {
        ya.command(task.name, task.description ?? "", taskCommandHandler(task));
    }
    function taskCommandHandler(task) {
        return (args) => {
            const taskReceivesArgs = task.steps?.reduce((receiveArgs, step) => receiveArgs || (step.receiveArgs ?? false), false);
            let taskArgs = [];
            if (taskReceivesArgs) {
                args.help(false);
                args.strict(false);
                args.strictCommands(false);
                taskArgs = (0, helpers_1.hideBin)(process.argv).slice(1);
            }
            args.option("inspect", {
                alias: "i",
                desc: "show all steps in this task",
            });
            const argv = args.argv;
            if (argv.inspect) {
                return inspectTask(task.name);
            }
            else {
                try {
                    runtime.runTask(task.name, [], taskArgs);
                }
                catch (e) {
                    logging.error(e.message);
                    process.exit(1);
                }
            }
        };
    }
    function inspectTask(name, indent = 0) {
        const writeln = (s) => console.log(" ".repeat(indent) + s);
        const task = runtime.tryFindTask(name);
        if (!task) {
            throw new Error(`${name}: unable to resolve subtask with name "${name}"`);
        }
        if (task.description) {
            writeln(`${chalk.underline("description")}: ${task.description}`);
        }
        for (const [k, v] of Object.entries(task.env ?? {})) {
            writeln(`${chalk.underline("env")}: ${k}=${v}`);
        }
        for (const step of task.steps ?? []) {
            if (step.spawn) {
                writeln(`- ${chalk.bold(step.spawn)}`);
                inspectTask(step.spawn, indent + 2);
            }
            else if (step.exec) {
                writeln(`- exec: ${step.exec}`);
            }
            else if (step.builtin) {
                writeln(`- builtin: ${step.builtin}`);
            }
        }
    }
}
exports.discoverTaskCommands = discoverTaskCommands;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGFza3MuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvY2xpL3Rhc2tzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLCtCQUErQjtBQUUvQiwyQ0FBd0M7QUFDeEMsc0NBQXNDO0FBSXRDOzs7R0FHRztBQUNILFNBQWdCLG9CQUFvQixDQUFDLE9BQW9CLEVBQUUsRUFBYztJQUN2RSxNQUFNLEtBQUssR0FBRyxPQUFPLENBQUMsUUFBUSxDQUFDLEtBQUssSUFBSSxFQUFFLENBQUM7SUFDM0MsS0FBSyxNQUFNLElBQUksSUFBSSxNQUFNLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxFQUFFO1FBQ3ZDLEVBQUUsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsV0FBVyxJQUFJLEVBQUUsRUFBRSxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO0tBQ3pFO0lBRUQsU0FBUyxrQkFBa0IsQ0FBQyxJQUFjO1FBQ3hDLE9BQU8sQ0FBQyxJQUFnQixFQUFFLEVBQUU7WUFDMUIsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FDekMsQ0FBQyxXQUFXLEVBQUUsSUFBSSxFQUFFLEVBQUUsQ0FBQyxXQUFXLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxJQUFJLEtBQUssQ0FBQyxFQUNqRSxLQUFLLENBQ04sQ0FBQztZQUNGLElBQUksUUFBUSxHQUEyQixFQUFFLENBQUM7WUFDMUMsSUFBSSxnQkFBZ0IsRUFBRTtnQkFDcEIsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDakIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDbkIsSUFBSSxDQUFDLGNBQWMsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDM0IsUUFBUSxHQUFHLElBQUEsaUJBQU8sRUFBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQzNDO1lBRUQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxTQUFTLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSxHQUFHO2dCQUNWLElBQUksRUFBRSw2QkFBNkI7YUFDcEMsQ0FBQyxDQUFDO1lBRUgsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQztZQUV2QixJQUFJLElBQUksQ0FBQyxPQUFPLEVBQUU7Z0JBQ2hCLE9BQU8sV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUMvQjtpQkFBTTtnQkFDTCxJQUFJO29CQUNGLE9BQU8sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUUsUUFBUSxDQUFDLENBQUM7aUJBQzFDO2dCQUFDLE9BQU8sQ0FBTSxFQUFFO29CQUNmLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDO29CQUN6QixPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO2lCQUNqQjthQUNGO1FBQ0gsQ0FBQyxDQUFDO0lBQ0osQ0FBQztJQUVELFNBQVMsV0FBVyxDQUFDLElBQVksRUFBRSxNQUFNLEdBQUcsQ0FBQztRQUMzQyxNQUFNLE9BQU8sR0FBRyxDQUFDLENBQVMsRUFBRSxFQUFFLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBRW5FLE1BQU0sSUFBSSxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDdkMsSUFBSSxDQUFDLElBQUksRUFBRTtZQUNULE1BQU0sSUFBSSxLQUFLLENBQUMsR0FBRyxJQUFJLDBDQUEwQyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1NBQzNFO1FBRUQsSUFBSSxJQUFJLENBQUMsV0FBVyxFQUFFO1lBQ3BCLE9BQU8sQ0FBQyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLEtBQUssSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUM7U0FDbkU7UUFFRCxLQUFLLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLEVBQUUsQ0FBQyxFQUFFO1lBQ25ELE9BQU8sQ0FBQyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDakQ7UUFFRCxLQUFLLE1BQU0sSUFBSSxJQUFJLElBQUksQ0FBQyxLQUFLLElBQUksRUFBRSxFQUFFO1lBQ25DLElBQUksSUFBSSxDQUFDLEtBQUssRUFBRTtnQkFDZCxPQUFPLENBQUMsS0FBSyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLENBQUM7Z0JBQ3ZDLFdBQVcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQzthQUNyQztpQkFBTSxJQUFJLElBQUksQ0FBQyxJQUFJLEVBQUU7Z0JBQ3BCLE9BQU8sQ0FBQyxXQUFXLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO2FBQ2pDO2lCQUFNLElBQUksSUFBSSxDQUFDLE9BQU8sRUFBRTtnQkFDdkIsT0FBTyxDQUFDLGNBQWMsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7YUFDdkM7U0FDRjtJQUNILENBQUM7QUFDSCxDQUFDO0FBbkVELG9EQW1FQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNoYWxrIGZyb20gXCJjaGFsa1wiO1xuaW1wb3J0ICogYXMgeWFyZ3MgZnJvbSBcInlhcmdzXCI7XG5pbXBvcnQgeyBoaWRlQmluIH0gZnJvbSBcInlhcmdzL2hlbHBlcnNcIjtcbmltcG9ydCAqIGFzIGxvZ2dpbmcgZnJvbSBcIi4uL2xvZ2dpbmdcIjtcbmltcG9ydCB7IFRhc2tTcGVjIH0gZnJvbSBcIi4uL3Rhc2stbW9kZWxcIjtcbmltcG9ydCB7IFRhc2tSdW50aW1lIH0gZnJvbSBcIi4uL3Rhc2stcnVudGltZVwiO1xuXG4vKipcbiAqIFJlYWRzIC5wcm9qZW4vdGFza3MuanNvbiBhbmQgYWRkcyBDTEkgY29tbWFuZHMgZm9yIGFsbCB0YXNrcy5cbiAqIEBwYXJhbSB5YSB5YXJnc1xuICovXG5leHBvcnQgZnVuY3Rpb24gZGlzY292ZXJUYXNrQ29tbWFuZHMocnVudGltZTogVGFza1J1bnRpbWUsIHlhOiB5YXJncy5Bcmd2KSB7XG4gIGNvbnN0IHRhc2tzID0gcnVudGltZS5tYW5pZmVzdC50YXNrcyA/PyB7fTtcbiAgZm9yIChjb25zdCB0YXNrIG9mIE9iamVjdC52YWx1ZXModGFza3MpKSB7XG4gICAgeWEuY29tbWFuZCh0YXNrLm5hbWUsIHRhc2suZGVzY3JpcHRpb24gPz8gXCJcIiwgdGFza0NvbW1hbmRIYW5kbGVyKHRhc2spKTtcbiAgfVxuXG4gIGZ1bmN0aW9uIHRhc2tDb21tYW5kSGFuZGxlcih0YXNrOiBUYXNrU3BlYykge1xuICAgIHJldHVybiAoYXJnczogeWFyZ3MuQXJndikgPT4ge1xuICAgICAgY29uc3QgdGFza1JlY2VpdmVzQXJncyA9IHRhc2suc3RlcHM/LnJlZHVjZShcbiAgICAgICAgKHJlY2VpdmVBcmdzLCBzdGVwKSA9PiByZWNlaXZlQXJncyB8fCAoc3RlcC5yZWNlaXZlQXJncyA/PyBmYWxzZSksXG4gICAgICAgIGZhbHNlXG4gICAgICApO1xuICAgICAgbGV0IHRhc2tBcmdzOiBBcnJheTxzdHJpbmcgfCBudW1iZXI+ID0gW107XG4gICAgICBpZiAodGFza1JlY2VpdmVzQXJncykge1xuICAgICAgICBhcmdzLmhlbHAoZmFsc2UpO1xuICAgICAgICBhcmdzLnN0cmljdChmYWxzZSk7XG4gICAgICAgIGFyZ3Muc3RyaWN0Q29tbWFuZHMoZmFsc2UpO1xuICAgICAgICB0YXNrQXJncyA9IGhpZGVCaW4ocHJvY2Vzcy5hcmd2KS5zbGljZSgxKTtcbiAgICAgIH1cblxuICAgICAgYXJncy5vcHRpb24oXCJpbnNwZWN0XCIsIHtcbiAgICAgICAgYWxpYXM6IFwiaVwiLFxuICAgICAgICBkZXNjOiBcInNob3cgYWxsIHN0ZXBzIGluIHRoaXMgdGFza1wiLFxuICAgICAgfSk7XG5cbiAgICAgIGNvbnN0IGFyZ3YgPSBhcmdzLmFyZ3Y7XG5cbiAgICAgIGlmIChhcmd2Lmluc3BlY3QpIHtcbiAgICAgICAgcmV0dXJuIGluc3BlY3RUYXNrKHRhc2submFtZSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0cnkge1xuICAgICAgICAgIHJ1bnRpbWUucnVuVGFzayh0YXNrLm5hbWUsIFtdLCB0YXNrQXJncyk7XG4gICAgICAgIH0gY2F0Y2ggKGU6IGFueSkge1xuICAgICAgICAgIGxvZ2dpbmcuZXJyb3IoZS5tZXNzYWdlKTtcbiAgICAgICAgICBwcm9jZXNzLmV4aXQoMSk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9O1xuICB9XG5cbiAgZnVuY3Rpb24gaW5zcGVjdFRhc2sobmFtZTogc3RyaW5nLCBpbmRlbnQgPSAwKSB7XG4gICAgY29uc3Qgd3JpdGVsbiA9IChzOiBzdHJpbmcpID0+IGNvbnNvbGUubG9nKFwiIFwiLnJlcGVhdChpbmRlbnQpICsgcyk7XG5cbiAgICBjb25zdCB0YXNrID0gcnVudGltZS50cnlGaW5kVGFzayhuYW1lKTtcbiAgICBpZiAoIXRhc2spIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgJHtuYW1lfTogdW5hYmxlIHRvIHJlc29sdmUgc3VidGFzayB3aXRoIG5hbWUgXCIke25hbWV9XCJgKTtcbiAgICB9XG5cbiAgICBpZiAodGFzay5kZXNjcmlwdGlvbikge1xuICAgICAgd3JpdGVsbihgJHtjaGFsay51bmRlcmxpbmUoXCJkZXNjcmlwdGlvblwiKX06ICR7dGFzay5kZXNjcmlwdGlvbn1gKTtcbiAgICB9XG5cbiAgICBmb3IgKGNvbnN0IFtrLCB2XSBvZiBPYmplY3QuZW50cmllcyh0YXNrLmVudiA/PyB7fSkpIHtcbiAgICAgIHdyaXRlbG4oYCR7Y2hhbGsudW5kZXJsaW5lKFwiZW52XCIpfTogJHtrfT0ke3Z9YCk7XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBzdGVwIG9mIHRhc2suc3RlcHMgPz8gW10pIHtcbiAgICAgIGlmIChzdGVwLnNwYXduKSB7XG4gICAgICAgIHdyaXRlbG4oYC0gJHtjaGFsay5ib2xkKHN0ZXAuc3Bhd24pfWApO1xuICAgICAgICBpbnNwZWN0VGFzayhzdGVwLnNwYXduLCBpbmRlbnQgKyAyKTtcbiAgICAgIH0gZWxzZSBpZiAoc3RlcC5leGVjKSB7XG4gICAgICAgIHdyaXRlbG4oYC0gZXhlYzogJHtzdGVwLmV4ZWN9YCk7XG4gICAgICB9IGVsc2UgaWYgKHN0ZXAuYnVpbHRpbikge1xuICAgICAgICB3cml0ZWxuKGAtIGJ1aWx0aW46ICR7c3RlcC5idWlsdGlufWApO1xuICAgICAgfVxuICAgIH1cbiAgfVxufVxuIl19