"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CliError = exports.renderInstallCommand = exports.installPackage = void 0;
const fs = require("fs");
const path = require("path");
const logging = require("../logging");
const util_1 = require("../util");
/**
 * Installs the npm module (through `npm install`) to node_modules under `projectDir`.
 * @param spec The npm package spec (e.g. `foo@^1.2` or `foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz`)
 * @returns The installed package name (e.g. `@foo/bar`)
 */
function installPackage(baseDir, spec) {
    const packageJsonPath = path.join(baseDir, "package.json");
    const packageJsonExisted = fs.existsSync(packageJsonPath);
    if (!packageJsonExisted) {
        // Make sure we have a package.json to read from later
        (0, util_1.exec)("npm init --yes", { cwd: baseDir });
    }
    logging.info(`installing external module ${spec}...`);
    (0, util_1.exec)(renderInstallCommand(baseDir, spec), { cwd: baseDir });
    // Get the true installed package name
    const packageJson = JSON.parse(fs.readFileSync(packageJsonPath, "utf-8"));
    const packageName = Object.keys(packageJson.devDependencies).find((name) => name !== "projen");
    if (!packageName) {
        throw new Error(`Unable to resolve package name from spec ${spec}`);
    }
    // if package.json did not exist before calling `npm install`, we should remove it
    // so we can start off clean.
    if (!packageJsonExisted) {
        fs.rmSync(packageJsonPath, { force: true, recursive: true });
    }
    return packageName;
}
exports.installPackage = installPackage;
/**
 * Render a command to install an npm package.
 *
 * Engine checks are ignored at this point so that the module can be installed
 * regardless of the environment. This was needed to unblock the upgrade of the
 * minimum node version of projen, but also okay generally because engine checks
 * will be performed later and for all eternity.
 *
 * @param dir Base directory
 * @param module The module to install (e.g. foo@^1.2)
 * @returns The string that includes the install command ("npm install ...")
 */
function renderInstallCommand(dir, module) {
    return `npm install --save --save-dev -f --no-package-lock --prefix="${dir}" ${module}`;
}
exports.renderInstallCommand = renderInstallCommand;
class CliError extends Error {
    constructor(...lines) {
        super(lines.join("\n"));
        this.name = "CliError";
    }
}
exports.CliError = CliError;
//# sourceMappingURL=data:application/json;base64,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