"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerCompose = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const docker_compose_port_1 = require("./docker-compose-port");
const docker_compose_render_1 = require("./docker-compose-render");
const docker_compose_service_1 = require("./docker-compose-service");
const component_1 = require("../component");
const yaml_1 = require("../yaml");
/**
 * Create a docker-compose YAML file.
 */
class DockerCompose extends component_1.Component {
    /**
     * Depends on a service name.
     */
    static serviceName(serviceName) {
        return {
            serviceName,
        };
    }
    /**
     * Create a port mapping.
     * @param publishedPort Published port number
     * @param targetPort Container's port number
     * @param options Port mapping options
     */
    static portMapping(publishedPort, targetPort, options) {
        const protocol = options?.protocol ?? docker_compose_port_1.DockerComposeProtocol.TCP;
        return {
            target: targetPort,
            published: publishedPort,
            protocol: protocol,
            mode: "host",
        };
    }
    /**
     * Create a bind volume that binds a host path to the target path in the container.
     * @param sourcePath Host path name
     * @param targetPath Target path name
     */
    static bindVolume(sourcePath, targetPath) {
        return {
            bind(_volumeInfo) {
                return {
                    type: "bind",
                    source: sourcePath,
                    target: targetPath,
                };
            },
        };
    }
    /**
     * Create a named volume and mount it to the target path. If you use this
     * named volume in several services, the volume will be shared. In this
     * case, the volume configuration of the first-provided options are used.
     *
     * @param volumeName Name of the volume
     * @param targetPath Target path
     * @param options volume configuration (default: docker compose defaults)
     */
    static namedVolume(volumeName, targetPath, options = {}) {
        return {
            bind(volumeInfo) {
                volumeInfo.addVolumeConfiguration(volumeName, options);
                return {
                    type: "volume",
                    source: volumeName,
                    target: targetPath,
                };
            },
        };
    }
    /**
     * Create a named network and mount it to the target path. If you use this
     * named network in several services, the network will be shared. In this
     * case, the network configuration of the first-provided options are used.
     *
     * @param networkName Name of the network
     * @param options network configuration
     */
    static network(networkName, options = {}) {
        return {
            bind(networkInfo) {
                networkInfo.addNetworkConfiguration(networkName, options);
                return networkName;
            },
        };
    }
    constructor(project, props) {
        super(project);
        const nameSuffix = props?.nameSuffix ? `${props.nameSuffix}.yml` : "yml";
        this.file = new yaml_1.YamlFile(project, `docker-compose.${nameSuffix}`, {
            committed: true,
            readonly: true,
            obj: () => this._synthesizeDockerCompose(),
        });
        if (props?.schemaVersion && !parseFloat(props.schemaVersion)) {
            throw Error("Version tag needs to be a number");
        }
        this.version = props?.schemaVersion ? props.schemaVersion : "3.3";
        this.services = {};
        // Add the services provided via the constructor argument.
        const initialServices = props?.services ?? {};
        for (const [name, serviceDescription] of Object.entries(initialServices)) {
            this.addService(name, serviceDescription);
        }
    }
    /**
     * Add a service to the docker-compose file.
     * @param serviceName name of the service
     * @param description a service description
     */
    addService(serviceName, description) {
        const service = new docker_compose_service_1.DockerComposeService(serviceName, description);
        this.services[serviceName] = service;
        return service;
    }
    /**
     * @internal
     */
    _synthesizeDockerCompose() {
        if (Object.keys(this.services).length === 0) {
            throw new Error("DockerCompose requires at least one service");
        }
        return (0, docker_compose_render_1.renderDockerComposeFile)(this.services, this.version);
    }
}
exports.DockerCompose = DockerCompose;
_a = JSII_RTTI_SYMBOL_1;
DockerCompose[_a] = { fqn: "projen.DockerCompose", version: "0.79.27" };
//# sourceMappingURL=data:application/json;base64,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