"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const _resolve_1 = require("./_resolve");
const common_1 = require("./common");
const component_1 = require("./component");
const projenrc_1 = require("./projenrc");
const util_1 = require("./util");
const constructs_1 = require("./util/constructs");
class FileBase extends component_1.Component {
    /**
     * The projen marker, used to identify files as projen-generated.
     *
     * Value is undefined if the project is being ejected.
     */
    get marker() {
        if (this.project.ejected || !this.shouldAddMarker) {
            return undefined;
        }
        // `marker` is empty if project is being ejected or if explicitly disabled
        const projenrc = projenrc_1.ProjenrcFile.of(this.project)?.filePath ?? common_1.DEFAULT_PROJEN_RC_JS_FILENAME;
        return `${common_1.PROJEN_MARKER}. To modify, edit ${projenrc} and run "${this.project.projenCommand}".`;
    }
    constructor(scope, filePath, options = {}) {
        const project = (0, constructs_1.findClosestProject)(scope);
        const root = project.root;
        const normalizedPath = path.normalize(filePath);
        const projectPath = (0, util_1.normalizePersistedPath)(normalizedPath);
        const absolutePath = path.resolve(project.outdir, projectPath);
        const relativeProjectPath = path.relative(root.outdir, absolutePath);
        const rootProjectPath = (0, util_1.normalizePersistedPath)(relativeProjectPath);
        const autoId = `${new.target.name}@${projectPath}`;
        // Before actually creating the file, ensure the file path is unique within the full project tree
        // This is required because projects can create files inside their subprojects
        if (root.tryFindFile(absolutePath) || scope.node.tryFindChild(autoId)) {
            throw new Error(`There is already a file under ${rootProjectPath}`);
        }
        super(scope, autoId);
        this.node.addMetadata("type", "file");
        this.node.addMetadata("path", rootProjectPath);
        this.readonly = !project.ejected && (options.readonly ?? true);
        this.executable = options.executable ?? false;
        this.path = projectPath;
        this.absolutePath = absolutePath;
        this.shouldAddMarker = options.marker ?? true;
        const globPattern = `/${this.path}`;
        const committed = options.committed ?? project.commitGenerated ?? true;
        if (committed && filePath !== ".gitattributes") {
            project.annotateGenerated(`/${filePath}`);
        }
        const editGitignore = options.editGitignore ?? true;
        if (editGitignore) {
            this.project.addGitIgnore(`${committed ? "!" : ""}${globPattern}`);
        }
        else {
            if (options.committed != null) {
                throw new Error('"gitignore" is disabled, so it does not make sense to specify "committed"');
            }
        }
    }
    /**
     * Writes the file to the project's output directory
     */
    synthesize() {
        const outdir = this.project.outdir;
        const filePath = path.join(outdir, this.path);
        const resolver = {
            resolve: (obj, options) => (0, _resolve_1.resolve)(obj, options),
        };
        const content = this.synthesizeContent(resolver);
        if (content === undefined) {
            // remove file (if exists) and skip rest of synthesis
            (0, fs_1.rmSync)(filePath, { force: true, recursive: true });
            return;
        }
        // check if the file was changed.
        const prev = (0, util_1.tryReadFileSync)(filePath);
        const prevReadonly = !(0, util_1.isWritable)(filePath);
        const successfulExecutableAssertion = (0, util_1.assertExecutablePermissions)(filePath, this.executable);
        if (prev !== undefined &&
            content === prev &&
            prevReadonly === this.readonly &&
            successfulExecutableAssertion) {
            this.project.logger.debug(`no change in ${filePath}`);
            this._changed = false;
            return;
        }
        (0, util_1.writeFile)(filePath, content, {
            readonly: this.readonly,
            executable: this.executable,
        });
        this.checkForProjenMarker();
        this._changed = true;
    }
    /**
     * For debugging, check whether a file was incorrectly generated with
     * or without the projen marker. The projen marker does not *need* to be
     * included on projen-generated files, but it's recommended since it signals
     * that it probably should not be edited directly.
     */
    checkForProjenMarker() {
        const filePath = path.join(this.project.outdir, this.path);
        const contents = (0, util_1.tryReadFileSync)(filePath);
        const containsMarker = contents?.includes(common_1.PROJEN_MARKER);
        if (this.marker && !containsMarker) {
            this.project.logger.debug(`note: expected ${this.path} to contain marker but found none.`);
        }
        else if (!this.marker && containsMarker) {
            this.project.logger.debug(`note: expected ${this.path} to not contain marker but found one anyway.`);
        }
    }
    /**
     * Indicates if the file has been changed during synthesis. This property is
     * only available in `postSynthesize()` hooks. If this is `undefined`, the
     * file has not been synthesized yet.
     */
    get changed() {
        return this._changed;
    }
}
exports.FileBase = FileBase;
_a = JSII_RTTI_SYMBOL_1;
FileBase[_a] = { fqn: "projen.FileBase", version: "0.79.27" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmlsZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9maWxlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsMkJBQTRCO0FBQzVCLDZCQUE2QjtBQUU3Qix5Q0FBcUM7QUFDckMscUNBQXdFO0FBQ3hFLDJDQUF3QztBQUN4Qyx5Q0FBMEM7QUFDMUMsaUNBTWdCO0FBQ2hCLGtEQUF1RDtBQXdDdkQsTUFBc0IsUUFBUyxTQUFRLHFCQUFTO0lBd0I5Qzs7OztPQUlHO0lBQ0gsSUFBVyxNQUFNO1FBQ2YsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxlQUFlLEVBQUU7WUFDakQsT0FBTyxTQUFTLENBQUM7U0FDbEI7UUFFRCwwRUFBMEU7UUFDMUUsTUFBTSxRQUFRLEdBQ1osdUJBQVksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxFQUFFLFFBQVEsSUFBSSxzQ0FBNkIsQ0FBQztRQUMzRSxPQUFPLEdBQUcsc0JBQWEscUJBQXFCLFFBQVEsYUFBYSxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsSUFBSSxDQUFDO0lBQ2xHLENBQUM7SUFFRCxZQUNFLEtBQWlCLEVBQ2pCLFFBQWdCLEVBQ2hCLFVBQTJCLEVBQUU7UUFFN0IsTUFBTSxPQUFPLEdBQUcsSUFBQSwrQkFBa0IsRUFBQyxLQUFLLENBQUMsQ0FBQztRQUMxQyxNQUFNLElBQUksR0FBRyxPQUFPLENBQUMsSUFBSSxDQUFDO1FBQzFCLE1BQU0sY0FBYyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDaEQsTUFBTSxXQUFXLEdBQUcsSUFBQSw2QkFBc0IsRUFBQyxjQUFjLENBQUMsQ0FBQztRQUMzRCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxNQUFNLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDL0QsTUFBTSxtQkFBbUIsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsWUFBWSxDQUFDLENBQUM7UUFDckUsTUFBTSxlQUFlLEdBQUcsSUFBQSw2QkFBc0IsRUFBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQ3BFLE1BQU0sTUFBTSxHQUFHLEdBQUcsR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLElBQUksV0FBVyxFQUFFLENBQUM7UUFFbkQsaUdBQWlHO1FBQ2pHLDhFQUE4RTtRQUM5RSxJQUFJLElBQUksQ0FBQyxXQUFXLENBQUMsWUFBWSxDQUFDLElBQUksS0FBSyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDckUsTUFBTSxJQUFJLEtBQUssQ0FBQyxpQ0FBaUMsZUFBZSxFQUFFLENBQUMsQ0FBQztTQUNyRTtRQUVELEtBQUssQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDckIsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsTUFBTSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sRUFBRSxlQUFlLENBQUMsQ0FBQztRQUUvQyxJQUFJLENBQUMsUUFBUSxHQUFHLENBQUMsT0FBTyxDQUFDLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLElBQUksSUFBSSxDQUFDLENBQUM7UUFDL0QsSUFBSSxDQUFDLFVBQVUsR0FBRyxPQUFPLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQztRQUM5QyxJQUFJLENBQUMsSUFBSSxHQUFHLFdBQVcsQ0FBQztRQUN4QixJQUFJLENBQUMsWUFBWSxHQUFHLFlBQVksQ0FBQztRQUNqQyxJQUFJLENBQUMsZUFBZSxHQUFHLE9BQU8sQ0FBQyxNQUFNLElBQUksSUFBSSxDQUFDO1FBRTlDLE1BQU0sV0FBVyxHQUFHLElBQUksSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ3BDLE1BQU0sU0FBUyxHQUFHLE9BQU8sQ0FBQyxTQUFTLElBQUksT0FBTyxDQUFDLGVBQWUsSUFBSSxJQUFJLENBQUM7UUFDdkUsSUFBSSxTQUFTLElBQUksUUFBUSxLQUFLLGdCQUFnQixFQUFFO1lBQzlDLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLFFBQVEsRUFBRSxDQUFDLENBQUM7U0FDM0M7UUFDRCxNQUFNLGFBQWEsR0FBRyxPQUFPLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQztRQUNwRCxJQUFJLGFBQWEsRUFBRTtZQUNqQixJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxHQUFHLFNBQVMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEdBQUcsV0FBVyxFQUFFLENBQUMsQ0FBQztTQUNwRTthQUFNO1lBQ0wsSUFBSSxPQUFPLENBQUMsU0FBUyxJQUFJLElBQUksRUFBRTtnQkFDN0IsTUFBTSxJQUFJLEtBQUssQ0FDYiwyRUFBMkUsQ0FDNUUsQ0FBQzthQUNIO1NBQ0Y7SUFDSCxDQUFDO0lBV0Q7O09BRUc7SUFDSSxVQUFVO1FBQ2YsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUM7UUFDbkMsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzlDLE1BQU0sUUFBUSxHQUFjO1lBQzFCLE9BQU8sRUFBRSxDQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUUsRUFBRSxDQUFDLElBQUEsa0JBQU8sRUFBQyxHQUFHLEVBQUUsT0FBTyxDQUFDO1NBQ2pELENBQUM7UUFDRixNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDakQsSUFBSSxPQUFPLEtBQUssU0FBUyxFQUFFO1lBQ3pCLHFEQUFxRDtZQUNyRCxJQUFBLFdBQU0sRUFBQyxRQUFRLEVBQUUsRUFBRSxLQUFLLEVBQUUsSUFBSSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1lBQ25ELE9BQU87U0FDUjtRQUVELGlDQUFpQztRQUNqQyxNQUFNLElBQUksR0FBRyxJQUFBLHNCQUFlLEVBQUMsUUFBUSxDQUFDLENBQUM7UUFDdkMsTUFBTSxZQUFZLEdBQUcsQ0FBQyxJQUFBLGlCQUFVLEVBQUMsUUFBUSxDQUFDLENBQUM7UUFDM0MsTUFBTSw2QkFBNkIsR0FBRyxJQUFBLGtDQUEyQixFQUMvRCxRQUFRLEVBQ1IsSUFBSSxDQUFDLFVBQVUsQ0FDaEIsQ0FBQztRQUNGLElBQ0UsSUFBSSxLQUFLLFNBQVM7WUFDbEIsT0FBTyxLQUFLLElBQUk7WUFDaEIsWUFBWSxLQUFLLElBQUksQ0FBQyxRQUFRO1lBQzlCLDZCQUE2QixFQUM3QjtZQUNBLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsUUFBUSxFQUFFLENBQUMsQ0FBQztZQUN0RCxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQztZQUN0QixPQUFPO1NBQ1I7UUFFRCxJQUFBLGdCQUFTLEVBQUMsUUFBUSxFQUFFLE9BQU8sRUFBRTtZQUMzQixRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7WUFDdkIsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVO1NBQzVCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxvQkFBb0IsRUFBRSxDQUFDO1FBRTVCLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDO0lBQ3ZCLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNLLG9CQUFvQjtRQUMxQixNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMzRCxNQUFNLFFBQVEsR0FBRyxJQUFBLHNCQUFlLEVBQUMsUUFBUSxDQUFDLENBQUM7UUFDM0MsTUFBTSxjQUFjLEdBQUcsUUFBUSxFQUFFLFFBQVEsQ0FBQyxzQkFBYSxDQUFDLENBQUM7UUFDekQsSUFBSSxJQUFJLENBQUMsTUFBTSxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQ2xDLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FDdkIsa0JBQWtCLElBQUksQ0FBQyxJQUFJLG9DQUFvQyxDQUNoRSxDQUFDO1NBQ0g7YUFBTSxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sSUFBSSxjQUFjLEVBQUU7WUFDekMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUN2QixrQkFBa0IsSUFBSSxDQUFDLElBQUksOENBQThDLENBQzFFLENBQUM7U0FDSDtJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0gsSUFBVyxPQUFPO1FBQ2hCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQztJQUN2QixDQUFDOztBQXhLSCw0QkF5S0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBybVN5bmMgfSBmcm9tIFwiZnNcIjtcbmltcG9ydCAqIGFzIHBhdGggZnJvbSBcInBhdGhcIjtcbmltcG9ydCB7IElDb25zdHJ1Y3QgfSBmcm9tIFwiY29uc3RydWN0c1wiO1xuaW1wb3J0IHsgcmVzb2x2ZSB9IGZyb20gXCIuL19yZXNvbHZlXCI7XG5pbXBvcnQgeyBQUk9KRU5fTUFSS0VSLCBERUZBVUxUX1BST0pFTl9SQ19KU19GSUxFTkFNRSB9IGZyb20gXCIuL2NvbW1vblwiO1xuaW1wb3J0IHsgQ29tcG9uZW50IH0gZnJvbSBcIi4vY29tcG9uZW50XCI7XG5pbXBvcnQgeyBQcm9qZW5yY0ZpbGUgfSBmcm9tIFwiLi9wcm9qZW5yY1wiO1xuaW1wb3J0IHtcbiAgYXNzZXJ0RXhlY3V0YWJsZVBlcm1pc3Npb25zLFxuICBpc1dyaXRhYmxlLFxuICBub3JtYWxpemVQZXJzaXN0ZWRQYXRoLFxuICB0cnlSZWFkRmlsZVN5bmMsXG4gIHdyaXRlRmlsZSxcbn0gZnJvbSBcIi4vdXRpbFwiO1xuaW1wb3J0IHsgZmluZENsb3Nlc3RQcm9qZWN0IH0gZnJvbSBcIi4vdXRpbC9jb25zdHJ1Y3RzXCI7XG5cbmV4cG9ydCBpbnRlcmZhY2UgRmlsZUJhc2VPcHRpb25zIHtcbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIHRoaXMgZmlsZSBzaG91bGQgYmUgY29tbWl0dGVkIHRvIGdpdCBvciBpZ25vcmVkLiBCeVxuICAgKiBkZWZhdWx0LCBhbGwgZ2VuZXJhdGVkIGZpbGVzIGFyZSBjb21taXR0ZWQgYW5kIGFudGktdGFtcGVyIGlzIHVzZWQgdG9cbiAgICogcHJvdGVjdCBhZ2FpbnN0IG1hbnVhbCBtb2RpZmljYXRpb25zLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBjb21taXR0ZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBVcGRhdGUgdGhlIHByb2plY3QncyAuZ2l0aWdub3JlIGZpbGVcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgZWRpdEdpdGlnbm9yZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdGhlIGdlbmVyYXRlZCBmaWxlIHNob3VsZCBiZSByZWFkb25seS5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgcmVhZG9ubHk/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoZSBnZW5lcmF0ZWQgZmlsZSBzaG91bGQgYmUgbWFya2VkIGFzIGV4ZWN1dGFibGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBleGVjdXRhYmxlPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQWRkcyB0aGUgcHJvamVuIG1hcmtlciB0byB0aGUgZmlsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBtYXJrZXIgd2lsbCBiZSBpbmNsdWRlZCBhcyBsb25nIGFzIHRoZSBwcm9qZWN0IGlzIG5vdCBlamVjdGVkXG4gICAqL1xuICByZWFkb25seSBtYXJrZXI/OiBib29sZWFuO1xufVxuXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgRmlsZUJhc2UgZXh0ZW5kcyBDb21wb25lbnQge1xuICAvKipcbiAgICogVGhlIGZpbGUgcGF0aCwgcmVsYXRpdmUgdG8gdGhlIHByb2plY3QncyBvdXRkaXIuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcGF0aDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgaWYgdGhlIGZpbGUgc2hvdWxkIGJlIHJlYWQtb25seSBvciByZWFkLXdyaXRlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgaWYgdGhlIGZpbGUgc2hvdWxkIGJlIG1hcmtlZCBhcyBleGVjdXRhYmxlXG4gICAqL1xuICBwdWJsaWMgZXhlY3V0YWJsZTogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIGFic29sdXRlIHBhdGggb2YgdGhpcyBmaWxlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGFic29sdXRlUGF0aDogc3RyaW5nO1xuXG4gIHByaXZhdGUgX2NoYW5nZWQ/OiBib29sZWFuO1xuICBwcml2YXRlIHNob3VsZEFkZE1hcmtlcjogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIHByb2plbiBtYXJrZXIsIHVzZWQgdG8gaWRlbnRpZnkgZmlsZXMgYXMgcHJvamVuLWdlbmVyYXRlZC5cbiAgICpcbiAgICogVmFsdWUgaXMgdW5kZWZpbmVkIGlmIHRoZSBwcm9qZWN0IGlzIGJlaW5nIGVqZWN0ZWQuXG4gICAqL1xuICBwdWJsaWMgZ2V0IG1hcmtlcigpOiBzdHJpbmcgfCB1bmRlZmluZWQge1xuICAgIGlmICh0aGlzLnByb2plY3QuZWplY3RlZCB8fCAhdGhpcy5zaG91bGRBZGRNYXJrZXIpIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuXG4gICAgLy8gYG1hcmtlcmAgaXMgZW1wdHkgaWYgcHJvamVjdCBpcyBiZWluZyBlamVjdGVkIG9yIGlmIGV4cGxpY2l0bHkgZGlzYWJsZWRcbiAgICBjb25zdCBwcm9qZW5yYyA9XG4gICAgICBQcm9qZW5yY0ZpbGUub2YodGhpcy5wcm9qZWN0KT8uZmlsZVBhdGggPz8gREVGQVVMVF9QUk9KRU5fUkNfSlNfRklMRU5BTUU7XG4gICAgcmV0dXJuIGAke1BST0pFTl9NQVJLRVJ9LiBUbyBtb2RpZnksIGVkaXQgJHtwcm9qZW5yY30gYW5kIHJ1biBcIiR7dGhpcy5wcm9qZWN0LnByb2plbkNvbW1hbmR9XCIuYDtcbiAgfVxuXG4gIGNvbnN0cnVjdG9yKFxuICAgIHNjb3BlOiBJQ29uc3RydWN0LFxuICAgIGZpbGVQYXRoOiBzdHJpbmcsXG4gICAgb3B0aW9uczogRmlsZUJhc2VPcHRpb25zID0ge31cbiAgKSB7XG4gICAgY29uc3QgcHJvamVjdCA9IGZpbmRDbG9zZXN0UHJvamVjdChzY29wZSk7XG4gICAgY29uc3Qgcm9vdCA9IHByb2plY3Qucm9vdDtcbiAgICBjb25zdCBub3JtYWxpemVkUGF0aCA9IHBhdGgubm9ybWFsaXplKGZpbGVQYXRoKTtcbiAgICBjb25zdCBwcm9qZWN0UGF0aCA9IG5vcm1hbGl6ZVBlcnNpc3RlZFBhdGgobm9ybWFsaXplZFBhdGgpO1xuICAgIGNvbnN0IGFic29sdXRlUGF0aCA9IHBhdGgucmVzb2x2ZShwcm9qZWN0Lm91dGRpciwgcHJvamVjdFBhdGgpO1xuICAgIGNvbnN0IHJlbGF0aXZlUHJvamVjdFBhdGggPSBwYXRoLnJlbGF0aXZlKHJvb3Qub3V0ZGlyLCBhYnNvbHV0ZVBhdGgpO1xuICAgIGNvbnN0IHJvb3RQcm9qZWN0UGF0aCA9IG5vcm1hbGl6ZVBlcnNpc3RlZFBhdGgocmVsYXRpdmVQcm9qZWN0UGF0aCk7XG4gICAgY29uc3QgYXV0b0lkID0gYCR7bmV3LnRhcmdldC5uYW1lfUAke3Byb2plY3RQYXRofWA7XG5cbiAgICAvLyBCZWZvcmUgYWN0dWFsbHkgY3JlYXRpbmcgdGhlIGZpbGUsIGVuc3VyZSB0aGUgZmlsZSBwYXRoIGlzIHVuaXF1ZSB3aXRoaW4gdGhlIGZ1bGwgcHJvamVjdCB0cmVlXG4gICAgLy8gVGhpcyBpcyByZXF1aXJlZCBiZWNhdXNlIHByb2plY3RzIGNhbiBjcmVhdGUgZmlsZXMgaW5zaWRlIHRoZWlyIHN1YnByb2plY3RzXG4gICAgaWYgKHJvb3QudHJ5RmluZEZpbGUoYWJzb2x1dGVQYXRoKSB8fCBzY29wZS5ub2RlLnRyeUZpbmRDaGlsZChhdXRvSWQpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFRoZXJlIGlzIGFscmVhZHkgYSBmaWxlIHVuZGVyICR7cm9vdFByb2plY3RQYXRofWApO1xuICAgIH1cblxuICAgIHN1cGVyKHNjb3BlLCBhdXRvSWQpO1xuICAgIHRoaXMubm9kZS5hZGRNZXRhZGF0YShcInR5cGVcIiwgXCJmaWxlXCIpO1xuICAgIHRoaXMubm9kZS5hZGRNZXRhZGF0YShcInBhdGhcIiwgcm9vdFByb2plY3RQYXRoKTtcblxuICAgIHRoaXMucmVhZG9ubHkgPSAhcHJvamVjdC5lamVjdGVkICYmIChvcHRpb25zLnJlYWRvbmx5ID8/IHRydWUpO1xuICAgIHRoaXMuZXhlY3V0YWJsZSA9IG9wdGlvbnMuZXhlY3V0YWJsZSA/PyBmYWxzZTtcbiAgICB0aGlzLnBhdGggPSBwcm9qZWN0UGF0aDtcbiAgICB0aGlzLmFic29sdXRlUGF0aCA9IGFic29sdXRlUGF0aDtcbiAgICB0aGlzLnNob3VsZEFkZE1hcmtlciA9IG9wdGlvbnMubWFya2VyID8/IHRydWU7XG5cbiAgICBjb25zdCBnbG9iUGF0dGVybiA9IGAvJHt0aGlzLnBhdGh9YDtcbiAgICBjb25zdCBjb21taXR0ZWQgPSBvcHRpb25zLmNvbW1pdHRlZCA/PyBwcm9qZWN0LmNvbW1pdEdlbmVyYXRlZCA/PyB0cnVlO1xuICAgIGlmIChjb21taXR0ZWQgJiYgZmlsZVBhdGggIT09IFwiLmdpdGF0dHJpYnV0ZXNcIikge1xuICAgICAgcHJvamVjdC5hbm5vdGF0ZUdlbmVyYXRlZChgLyR7ZmlsZVBhdGh9YCk7XG4gICAgfVxuICAgIGNvbnN0IGVkaXRHaXRpZ25vcmUgPSBvcHRpb25zLmVkaXRHaXRpZ25vcmUgPz8gdHJ1ZTtcbiAgICBpZiAoZWRpdEdpdGlnbm9yZSkge1xuICAgICAgdGhpcy5wcm9qZWN0LmFkZEdpdElnbm9yZShgJHtjb21taXR0ZWQgPyBcIiFcIiA6IFwiXCJ9JHtnbG9iUGF0dGVybn1gKTtcbiAgICB9IGVsc2Uge1xuICAgICAgaWYgKG9wdGlvbnMuY29tbWl0dGVkICE9IG51bGwpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICdcImdpdGlnbm9yZVwiIGlzIGRpc2FibGVkLCBzbyBpdCBkb2VzIG5vdCBtYWtlIHNlbnNlIHRvIHNwZWNpZnkgXCJjb21taXR0ZWRcIidcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogSW1wbGVtZW50ZWQgYnkgZGVyaXZlZCBjbGFzc2VzIGFuZCByZXR1cm5zIHRoZSBjb250ZW50cyBvZiB0aGUgZmlsZSB0b1xuICAgKiBlbWl0LlxuICAgKiBAcGFyYW0gcmVzb2x2ZXIgQ2FsbCBgcmVzb2x2ZXIucmVzb2x2ZShvYmopYCBvbiBhbnkgb2JqZWN0cyBpbiBvcmRlciB0b1xuICAgKiByZXNvbHZlIHRva2VuIGZ1bmN0aW9ucy5cbiAgICogQHJldHVybnMgdGhlIGNvbnRlbnQgdG8gc3ludGhlc2l6ZSBvciB1bmRlZmluZWQgdG8gc2tpcCB0aGUgZmlsZVxuICAgKi9cbiAgcHJvdGVjdGVkIGFic3RyYWN0IHN5bnRoZXNpemVDb250ZW50KHJlc29sdmVyOiBJUmVzb2x2ZXIpOiBzdHJpbmcgfCB1bmRlZmluZWQ7XG5cbiAgLyoqXG4gICAqIFdyaXRlcyB0aGUgZmlsZSB0byB0aGUgcHJvamVjdCdzIG91dHB1dCBkaXJlY3RvcnlcbiAgICovXG4gIHB1YmxpYyBzeW50aGVzaXplKCkge1xuICAgIGNvbnN0IG91dGRpciA9IHRoaXMucHJvamVjdC5vdXRkaXI7XG4gICAgY29uc3QgZmlsZVBhdGggPSBwYXRoLmpvaW4ob3V0ZGlyLCB0aGlzLnBhdGgpO1xuICAgIGNvbnN0IHJlc29sdmVyOiBJUmVzb2x2ZXIgPSB7XG4gICAgICByZXNvbHZlOiAob2JqLCBvcHRpb25zKSA9PiByZXNvbHZlKG9iaiwgb3B0aW9ucyksXG4gICAgfTtcbiAgICBjb25zdCBjb250ZW50ID0gdGhpcy5zeW50aGVzaXplQ29udGVudChyZXNvbHZlcik7XG4gICAgaWYgKGNvbnRlbnQgPT09IHVuZGVmaW5lZCkge1xuICAgICAgLy8gcmVtb3ZlIGZpbGUgKGlmIGV4aXN0cykgYW5kIHNraXAgcmVzdCBvZiBzeW50aGVzaXNcbiAgICAgIHJtU3luYyhmaWxlUGF0aCwgeyBmb3JjZTogdHJ1ZSwgcmVjdXJzaXZlOiB0cnVlIH0pO1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIC8vIGNoZWNrIGlmIHRoZSBmaWxlIHdhcyBjaGFuZ2VkLlxuICAgIGNvbnN0IHByZXYgPSB0cnlSZWFkRmlsZVN5bmMoZmlsZVBhdGgpO1xuICAgIGNvbnN0IHByZXZSZWFkb25seSA9ICFpc1dyaXRhYmxlKGZpbGVQYXRoKTtcbiAgICBjb25zdCBzdWNjZXNzZnVsRXhlY3V0YWJsZUFzc2VydGlvbiA9IGFzc2VydEV4ZWN1dGFibGVQZXJtaXNzaW9ucyhcbiAgICAgIGZpbGVQYXRoLFxuICAgICAgdGhpcy5leGVjdXRhYmxlXG4gICAgKTtcbiAgICBpZiAoXG4gICAgICBwcmV2ICE9PSB1bmRlZmluZWQgJiZcbiAgICAgIGNvbnRlbnQgPT09IHByZXYgJiZcbiAgICAgIHByZXZSZWFkb25seSA9PT0gdGhpcy5yZWFkb25seSAmJlxuICAgICAgc3VjY2Vzc2Z1bEV4ZWN1dGFibGVBc3NlcnRpb25cbiAgICApIHtcbiAgICAgIHRoaXMucHJvamVjdC5sb2dnZXIuZGVidWcoYG5vIGNoYW5nZSBpbiAke2ZpbGVQYXRofWApO1xuICAgICAgdGhpcy5fY2hhbmdlZCA9IGZhbHNlO1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIHdyaXRlRmlsZShmaWxlUGF0aCwgY29udGVudCwge1xuICAgICAgcmVhZG9ubHk6IHRoaXMucmVhZG9ubHksXG4gICAgICBleGVjdXRhYmxlOiB0aGlzLmV4ZWN1dGFibGUsXG4gICAgfSk7XG5cbiAgICB0aGlzLmNoZWNrRm9yUHJvamVuTWFya2VyKCk7XG5cbiAgICB0aGlzLl9jaGFuZ2VkID0gdHJ1ZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGb3IgZGVidWdnaW5nLCBjaGVjayB3aGV0aGVyIGEgZmlsZSB3YXMgaW5jb3JyZWN0bHkgZ2VuZXJhdGVkIHdpdGhcbiAgICogb3Igd2l0aG91dCB0aGUgcHJvamVuIG1hcmtlci4gVGhlIHByb2plbiBtYXJrZXIgZG9lcyBub3QgKm5lZWQqIHRvIGJlXG4gICAqIGluY2x1ZGVkIG9uIHByb2plbi1nZW5lcmF0ZWQgZmlsZXMsIGJ1dCBpdCdzIHJlY29tbWVuZGVkIHNpbmNlIGl0IHNpZ25hbHNcbiAgICogdGhhdCBpdCBwcm9iYWJseSBzaG91bGQgbm90IGJlIGVkaXRlZCBkaXJlY3RseS5cbiAgICovXG4gIHByaXZhdGUgY2hlY2tGb3JQcm9qZW5NYXJrZXIoKSB7XG4gICAgY29uc3QgZmlsZVBhdGggPSBwYXRoLmpvaW4odGhpcy5wcm9qZWN0Lm91dGRpciwgdGhpcy5wYXRoKTtcbiAgICBjb25zdCBjb250ZW50cyA9IHRyeVJlYWRGaWxlU3luYyhmaWxlUGF0aCk7XG4gICAgY29uc3QgY29udGFpbnNNYXJrZXIgPSBjb250ZW50cz8uaW5jbHVkZXMoUFJPSkVOX01BUktFUik7XG4gICAgaWYgKHRoaXMubWFya2VyICYmICFjb250YWluc01hcmtlcikge1xuICAgICAgdGhpcy5wcm9qZWN0LmxvZ2dlci5kZWJ1ZyhcbiAgICAgICAgYG5vdGU6IGV4cGVjdGVkICR7dGhpcy5wYXRofSB0byBjb250YWluIG1hcmtlciBidXQgZm91bmQgbm9uZS5gXG4gICAgICApO1xuICAgIH0gZWxzZSBpZiAoIXRoaXMubWFya2VyICYmIGNvbnRhaW5zTWFya2VyKSB7XG4gICAgICB0aGlzLnByb2plY3QubG9nZ2VyLmRlYnVnKFxuICAgICAgICBgbm90ZTogZXhwZWN0ZWQgJHt0aGlzLnBhdGh9IHRvIG5vdCBjb250YWluIG1hcmtlciBidXQgZm91bmQgb25lIGFueXdheS5gXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgaWYgdGhlIGZpbGUgaGFzIGJlZW4gY2hhbmdlZCBkdXJpbmcgc3ludGhlc2lzLiBUaGlzIHByb3BlcnR5IGlzXG4gICAqIG9ubHkgYXZhaWxhYmxlIGluIGBwb3N0U3ludGhlc2l6ZSgpYCBob29rcy4gSWYgdGhpcyBpcyBgdW5kZWZpbmVkYCwgdGhlXG4gICAqIGZpbGUgaGFzIG5vdCBiZWVuIHN5bnRoZXNpemVkIHlldC5cbiAgICovXG4gIHB1YmxpYyBnZXQgY2hhbmdlZCgpOiBib29sZWFuIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy5fY2hhbmdlZDtcbiAgfVxufVxuXG4vKipcbiAqIEFQSSBmb3IgcmVzb2x2aW5nIHRva2VucyB3aGVuIHN5bnRoZXNpemluZyBmaWxlIGNvbnRlbnQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSVJlc29sdmVyIHtcbiAgLyoqXG4gICAqIEdpdmVuIGEgdmFsdWUgKG9iamVjdC9zdHJpbmcvYXJyYXkvd2hhdGV2ZXIsIGxvb2tzIHVwIGFueSBmdW5jdGlvbnMgaW5zaWRlXG4gICAqIHRoZSBvYmplY3QgYW5kIHJldHVybnMgYW4gb2JqZWN0IHdoZXJlIGFsbCBmdW5jdGlvbnMgYXJlIGNhbGxlZC5cbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZSB0byByZXNvbHZlXG4gICAqIEBwYWNrYWdlIG9wdGlvbnMgUmVzb2x2ZSBvcHRpb25zXG4gICAqL1xuICByZXNvbHZlKHZhbHVlOiBhbnksIG9wdGlvbnM/OiBSZXNvbHZlT3B0aW9ucyk6IGFueTtcbn1cblxuLyoqXG4gKiBSZXNvbHZlIG9wdGlvbnMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVzb2x2ZU9wdGlvbnMge1xuICAvKipcbiAgICogT21pdHMgZW1wdHkgYXJyYXlzIGFuZCBvYmplY3RzLlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgb21pdEVtcHR5PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQ29udGV4dCBhcmd1bWVudHMuXG4gICAqIEBkZWZhdWx0IFtdXG4gICAqL1xuICByZWFkb25seSBhcmdzPzogYW55W107XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgSVJlc29sdmFibGUge1xuICAvKipcbiAgICogUmVzb2x2ZXMgYW5kIHJldHVybnMgY29udGVudC5cbiAgICovXG4gIHRvSlNPTigpOiBhbnk7XG59XG4iXX0=