"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutoMerge = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
/**
 * Sets up mergify to merging approved pull requests.
 *
 * If `buildJob` is specified, the specified GitHub workflow job ID is required
 * to succeed in order for the PR to be merged.
 *
 * `approvedReviews` specified the number of code review approvals required for
 * the PR to be merged.
 */
class AutoMerge extends component_1.Component {
    constructor(github, options = {}) {
        super(github.project);
        this.lazyConditions = new Array();
        const mergify = github.mergify;
        if (!mergify) {
            throw new Error("auto merging requires mergify to be enabled");
        }
        const blockingLabels = options.blockingLabels ?? ["do-not-merge"];
        const blockingCondition = blockingLabels?.length
            ? [`-label~=(${blockingLabels.join("|")})`]
            : [];
        const mergeAction = {
            delete_head_branch: {},
            queue: {
                // squash all commits into a single commit when merging
                // method: "squash",
                method: "squash",
                name: "default",
                // use PR title+body as the commit message
                commit_message_template: [
                    "{{ title }} (#{{ number }})",
                    "",
                    "{{ body }}",
                ].join("\n"),
            },
        };
        const approvedReviews = options.approvedReviews ?? 1;
        // add initial conditions (additional conditions can be added later)
        this.addConditions(`#approved-reviews-by>=${approvedReviews}`);
        this.addConditions(...blockingCondition);
        const ruleName = options.ruleName ?? "Automatic merge on approval and successful build";
        const queueName = options.queueName ?? "default";
        mergify.addRule({
            name: ruleName,
            actions: mergeAction,
            conditions: (() => this.renderConditions()),
        });
        mergify.addQueue({
            name: queueName,
            updateMethod: "merge",
            conditions: (() => this.renderConditions()),
        });
        this.project.addPackageIgnore("/.mergify.yml");
    }
    /**
     * Adds conditions to the auto merge rule.
     * @param conditions The conditions to add (mergify syntax)
     */
    addConditions(...conditions) {
        this.addConditionsLater({ render: () => conditions });
    }
    /**
     * Adds conditions that will be rendered only during synthesis.
     * @param later The later
     */
    addConditionsLater(later) {
        this.lazyConditions.push(later);
    }
    renderConditions() {
        const output = new Array();
        for (const later of this.lazyConditions) {
            output.push(...later.render());
        }
        return output;
    }
}
exports.AutoMerge = AutoMerge;
_a = JSII_RTTI_SYMBOL_1;
AutoMerge[_a] = { fqn: "projen.github.AutoMerge", version: "0.79.27" };
//# sourceMappingURL=data:application/json;base64,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