"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHub = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const actions_provider_1 = require("./actions-provider");
const dependabot_1 = require("./dependabot");
const github_credentials_1 = require("./github-credentials");
const mergify_1 = require("./mergify");
const pr_template_1 = require("./pr-template");
const pull_request_lint_1 = require("./pull-request-lint");
const workflows_1 = require("./workflows");
const component_1 = require("../component");
class GitHub extends component_1.Component {
    /**
     * Returns the `GitHub` component of a project or `undefined` if the project
     * does not have a GitHub component.
     */
    static of(project) {
        const isGitHub = (c) => c instanceof GitHub;
        return project.components.find(isGitHub);
    }
    constructor(project, options = {}) {
        super(project);
        this.actions = new actions_provider_1.GitHubActionsProvider();
        this.workflowsEnabled = options.workflows ?? true;
        this._downloadLfs = options.downloadLfs;
        if (options.projenCredentials && options.projenTokenSecret) {
            throw new Error("projenTokenSecret is deprecated, please use projenCredentials instead");
        }
        if (options.projenTokenSecret) {
            this.projenCredentials = github_credentials_1.GithubCredentials.fromPersonalAccessToken({
                secret: options.projenTokenSecret,
            });
        }
        else if (options.projenCredentials) {
            this.projenCredentials = options.projenCredentials;
        }
        else {
            this.projenCredentials = github_credentials_1.GithubCredentials.fromPersonalAccessToken({
                secret: "PROJEN_GITHUB_TOKEN",
            });
        }
        if (options.mergify ?? true) {
            this.mergify = new mergify_1.Mergify(this, options.mergifyOptions);
        }
        if (options.pullRequestLint ?? true) {
            new pull_request_lint_1.PullRequestLint(this, options.pullRequestLintOptions);
        }
    }
    /**
     * All workflows.
     */
    get workflows() {
        const isWorkflow = (c) => c instanceof workflows_1.GithubWorkflow;
        return this.project.components
            .filter(isWorkflow)
            .sort((w1, w2) => w1.name.localeCompare(w2.name));
    }
    /**
     * Adds a workflow to the project.
     * @param name Name of the workflow
     * @returns a GithubWorkflow instance
     */
    addWorkflow(name) {
        const workflow = new workflows_1.GithubWorkflow(this, name);
        return workflow;
    }
    addPullRequestTemplate(...content) {
        return new pr_template_1.PullRequestTemplate(this, { lines: content });
    }
    addDependabot(options) {
        return new dependabot_1.Dependabot(this, options);
    }
    /**
     * Finds a GitHub workflow by name. Returns `undefined` if the workflow cannot be found.
     * @param name The name of the GitHub workflow
     */
    tryFindWorkflow(name) {
        return this.workflows.find((w) => w.name === name);
    }
    /**
     * Whether downloading from LFS is enabled for this GitHub project
     */
    get downloadLfs() {
        return this._downloadLfs ?? this.project.gitattributes.hasLfsPatterns;
    }
}
exports.GitHub = GitHub;
_a = JSII_RTTI_SYMBOL_1;
GitHub[_a] = { fqn: "projen.github.GitHub", version: "0.79.27" };
//# sourceMappingURL=data:application/json;base64,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