"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PullRequestLint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const _1 = require(".");
const workflows_model_1 = require("./workflows-model");
const component_1 = require("../component");
const runner_options_1 = require("../runner-options");
/**
 * Configure validations to run on GitHub pull requests.
 * Only generates a file if at least one linter is configured.
 */
class PullRequestLint extends component_1.Component {
    constructor(github, options = {}) {
        super(github.project);
        this.github = github;
        this.options = options;
        const checkSemanticTitle = options.semanticTitle ?? true;
        const checkContributorStatement = Boolean(options.contributorStatement);
        // should only create a workflow if one or more linters are enabled
        if (!checkSemanticTitle && !checkContributorStatement) {
            return;
        }
        const workflow = github.addWorkflow("pull-request-lint");
        workflow.on({
            pullRequestTarget: {
                types: [
                    "labeled",
                    "opened",
                    "synchronize",
                    "reopened",
                    "ready_for_review",
                    "edited",
                ],
            },
        });
        if (checkSemanticTitle) {
            const opts = options.semanticTitleOptions ?? {};
            const types = opts.types ?? ["feat", "fix", "chore"];
            const validateJob = {
                name: "Validate PR title",
                ...(0, runner_options_1.filteredRunsOnOptions)(options.runsOn, options.runsOnGroup),
                permissions: {
                    pullRequests: workflows_model_1.JobPermission.WRITE,
                },
                steps: [
                    {
                        uses: "amannn/action-semantic-pull-request@v5.4.0",
                        env: {
                            GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                        },
                        with: {
                            types: types.join("\n"),
                            requireScope: opts.requireScope ?? false,
                        },
                    },
                ],
            };
            workflow.addJobs({ validate: validateJob });
        }
        if (options.contributorStatement) {
            const opts = options.contributorStatementOptions ?? {};
            const users = opts.exemptUsers ?? [];
            const labels = opts.exemptLabels ?? [];
            const conditions = [
                ...labels.map((l) => `contains(github.event.pull_request.labels.*.name, '${l}')`),
                ...users.map((u) => `github.event.pull_request.user.login == '${u}'`),
            ];
            const script = (core) => {
                const actual = process.env.PR_BODY.replace(/\r?\n/g, "\n");
                const expected = process.env.EXPECTED.replace(/\r?\n/g, "\n");
                if (!actual.includes(expected)) {
                    console.log("%j", actual);
                    console.log("%j", expected);
                    core.setFailed(`${process.env.HELP}: ${expected}`);
                }
            };
            const helpMessage = "Contributor statement missing from PR description. Please include the following text in the PR description";
            const contributorStatement = {
                name: "Require Contributor Statement",
                runsOn: options.runsOn ?? ["ubuntu-latest"],
                permissions: {
                    pullRequests: workflows_model_1.JobPermission.READ,
                },
                if: conditions.length ? `!(${conditions.join(" || ")})` : undefined,
                env: {
                    PR_BODY: "${{ github.event.pull_request.body }}",
                    EXPECTED: options.contributorStatement,
                    HELP: helpMessage,
                },
                steps: [
                    {
                        uses: "actions/github-script@v6",
                        with: {
                            script: fnBody(script),
                        },
                    },
                ],
            };
            workflow.addJobs({ contributorStatement });
        }
    }
    preSynthesize() {
        if (this.options.contributorStatement) {
            // Append to PR template in preSynthesize so it's always at the end of the file
            const prTemplate = _1.PullRequestTemplate.of(this.project) ??
                this.github.addPullRequestTemplate();
            prTemplate?.addLine("");
            prTemplate?.addLine("---");
            prTemplate?.addLine(this.options.contributorStatement);
            prTemplate?.addLine("");
        }
    }
}
exports.PullRequestLint = PullRequestLint;
_a = JSII_RTTI_SYMBOL_1;
PullRequestLint[_a] = { fqn: "projen.github.PullRequestLint", version: "0.79.27" };
/**
 * Helper to generate a JS script as string from a function object
 * @returns A prettified string of the function's body
 */
function fnBody(fn) {
    const def = fn.toString().replace(/\r?\n/g, "\n");
    const body = def
        .substring(def.indexOf("{") + 1, def.lastIndexOf("}"))
        .split("\n");
    const minIndentation = Math.min(...body
        .filter((l) => l.trim()) // ignore empty lines
        .map((l) => l.search(/\S|$/)));
    return body
        .map((l) => l.replace(" ".repeat(minIndentation), ""))
        .join("\n")
        .trim();
}
//# sourceMappingURL=data:application/json;base64,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