"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkflowActions = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constants_1 = require("./constants");
const workflow_steps_1 = require("./workflow-steps");
function context(value) {
    return `\${{ ${value} }}`;
}
const REPO = context("github.repository");
const RUN_ID = context("github.run_id");
const SERVER_URL = context("github.server_url");
const RUN_URL = `${SERVER_URL}/${REPO}/actions/runs/${RUN_ID}`;
const GIT_PATCH_FILE_DEFAULT = ".repo.patch";
const RUNNER_TEMP = "${{ runner.temp }}";
/**
 * A set of utility functions for creating GitHub actions in workflows.
 */
class WorkflowActions {
    /**
     * Creates a .patch file from the current git diff and uploads it as an
     * artifact. Use `checkoutWithPatch` to download and apply in another job.
     *
     * If a patch was uploaded, the action can optionally fail the job.
     *
     * @param options Options
     * @returns Job steps
     */
    static uploadGitPatch(options) {
        const MUTATIONS_FOUND = `steps.${options.stepId}.outputs.${options.outputName}`;
        const GIT_PATCH_FILE = options.patchFile ?? GIT_PATCH_FILE_DEFAULT;
        const steps = [
            {
                id: options.stepId,
                name: options.stepName ?? "Find mutations",
                run: [
                    "git add .",
                    `git diff --staged --patch --exit-code > ${GIT_PATCH_FILE} || echo "${options.outputName}=true" >> $GITHUB_OUTPUT`,
                ].join("\n"),
            },
            workflow_steps_1.WorkflowSteps.uploadArtifact({
                if: MUTATIONS_FOUND,
                name: "Upload patch",
                with: { name: GIT_PATCH_FILE, path: GIT_PATCH_FILE },
            }),
        ];
        if (options.mutationError) {
            steps.push({
                name: "Fail build on mutation",
                if: MUTATIONS_FOUND,
                run: [
                    `echo "::error::${options.mutationError}"`,
                    `cat ${GIT_PATCH_FILE}`,
                    "exit 1",
                ].join("\n"),
            });
        }
        return steps;
    }
    /**
     * Checks out a repository and applies a git patch that was created using
     * `uploadGitPatch`.
     *
     * @param options Options
     * @returns Job steps
     */
    static checkoutWithPatch(options = {}) {
        const { patchFile, ...restOfOptions } = options;
        const GIT_PATCH_FILE = options.patchFile ?? GIT_PATCH_FILE_DEFAULT;
        return [
            workflow_steps_1.WorkflowSteps.checkout({ with: restOfOptions }),
            {
                name: "Download patch",
                uses: "actions/download-artifact@v4",
                with: { name: GIT_PATCH_FILE, path: RUNNER_TEMP },
            },
            {
                name: "Apply patch",
                run: `[ -s ${RUNNER_TEMP}/${GIT_PATCH_FILE} ] && git apply ${RUNNER_TEMP}/${GIT_PATCH_FILE} || echo "Empty patch. Skipping."`,
            },
        ];
    }
    /**
     * A step that creates a pull request based on the current repo state.
     *
     * @param options Options
     * @returns Job steps
     */
    static createPullRequest(options) {
        const workflowName = options.workflowName;
        const branchName = options.branchName ?? `github-actions/${workflowName}`;
        const stepId = options.stepId ?? "create-pr";
        const stepName = options.stepName ?? "Create Pull Request";
        const gitIdentity = options.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        const committer = `${gitIdentity.name} <${gitIdentity.email}>`;
        const pullRequestDescription = options.pullRequestDescription
            .trimEnd()
            .endsWith(".")
            ? options.pullRequestDescription.trimEnd()
            : `${options.pullRequestDescription.trimEnd()}.`;
        const title = options.pullRequestTitle;
        const description = [
            `${pullRequestDescription} See details in [workflow run].`,
            "",
            `[Workflow Run]: ${RUN_URL}`,
            "",
            "------",
            "",
            `*Automatically created by projen via the "${workflowName}" workflow*`,
        ].join("\n");
        return [
            {
                name: stepName,
                id: stepId,
                uses: "peter-evans/create-pull-request@v6",
                with: {
                    token: options.credentials?.tokenRef,
                    "commit-message": `${title}\n\n${description}`,
                    branch: branchName,
                    base: options.baseBranch,
                    title: title,
                    labels: options.labels?.join(",") || undefined,
                    assignees: options.assignees?.join(",") || undefined,
                    body: description,
                    author: committer,
                    committer: committer,
                    signoff: options.signoff ?? true,
                },
            },
        ];
    }
    /**
     * Configures the git identity (user name and email).
     * @param id The identity to use
     * @returns Job steps
     *
     * @deprecated use `WorkflowSteps.setupGitIdentity` instead
     */
    static setupGitIdentity(id) {
        return [workflow_steps_1.WorkflowSteps.setupGitIdentity({ gitIdentity: id })];
    }
}
exports.WorkflowActions = WorkflowActions;
_a = JSII_RTTI_SYMBOL_1;
WorkflowActions[_a] = { fqn: "projen.github.WorkflowActions", version: "0.79.27" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid29ya2Zsb3ctYWN0aW9ucy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9naXRodWIvd29ya2Zsb3ctYWN0aW9ucy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUNBLDJDQUEwRDtBQUMxRCxxREFBK0Q7QUFHL0QsU0FBUyxPQUFPLENBQUMsS0FBYTtJQUM1QixPQUFPLFFBQVEsS0FBSyxLQUFLLENBQUM7QUFDNUIsQ0FBQztBQUVELE1BQU0sSUFBSSxHQUFHLE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0FBQzFDLE1BQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxlQUFlLENBQUMsQ0FBQztBQUN4QyxNQUFNLFVBQVUsR0FBRyxPQUFPLENBQUMsbUJBQW1CLENBQUMsQ0FBQztBQUNoRCxNQUFNLE9BQU8sR0FBRyxHQUFHLFVBQVUsSUFBSSxJQUFJLGlCQUFpQixNQUFNLEVBQUUsQ0FBQztBQUMvRCxNQUFNLHNCQUFzQixHQUFHLGFBQWEsQ0FBQztBQUM3QyxNQUFNLFdBQVcsR0FBRyxvQkFBb0IsQ0FBQztBQUV6Qzs7R0FFRztBQUNILE1BQWEsZUFBZTtJQUMxQjs7Ozs7Ozs7T0FRRztJQUNJLE1BQU0sQ0FBQyxjQUFjLENBQUMsT0FBOEI7UUFDekQsTUFBTSxlQUFlLEdBQUcsU0FBUyxPQUFPLENBQUMsTUFBTSxZQUFZLE9BQU8sQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUNoRixNQUFNLGNBQWMsR0FBRyxPQUFPLENBQUMsU0FBUyxJQUFJLHNCQUFzQixDQUFDO1FBRW5FLE1BQU0sS0FBSyxHQUFjO1lBQ3ZCO2dCQUNFLEVBQUUsRUFBRSxPQUFPLENBQUMsTUFBTTtnQkFDbEIsSUFBSSxFQUFFLE9BQU8sQ0FBQyxRQUFRLElBQUksZ0JBQWdCO2dCQUMxQyxHQUFHLEVBQUU7b0JBQ0gsV0FBVztvQkFDWCwyQ0FBMkMsY0FBYyxhQUFhLE9BQU8sQ0FBQyxVQUFVLDBCQUEwQjtpQkFDbkgsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO2FBQ2I7WUFDRCw4QkFBYSxDQUFDLGNBQWMsQ0FBQztnQkFDM0IsRUFBRSxFQUFFLGVBQWU7Z0JBQ25CLElBQUksRUFBRSxjQUFjO2dCQUNwQixJQUFJLEVBQUUsRUFBRSxJQUFJLEVBQUUsY0FBYyxFQUFFLElBQUksRUFBRSxjQUFjLEVBQUU7YUFDckQsQ0FBQztTQUNILENBQUM7UUFFRixJQUFJLE9BQU8sQ0FBQyxhQUFhLEVBQUU7WUFDekIsS0FBSyxDQUFDLElBQUksQ0FBQztnQkFDVCxJQUFJLEVBQUUsd0JBQXdCO2dCQUM5QixFQUFFLEVBQUUsZUFBZTtnQkFDbkIsR0FBRyxFQUFFO29CQUNILGtCQUFrQixPQUFPLENBQUMsYUFBYSxHQUFHO29CQUMxQyxPQUFPLGNBQWMsRUFBRTtvQkFDdkIsUUFBUTtpQkFDVCxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7YUFDYixDQUFDLENBQUM7U0FDSjtRQUVELE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUNEOzs7Ozs7T0FNRztJQUNJLE1BQU0sQ0FBQyxpQkFBaUIsQ0FDN0IsVUFBb0MsRUFBRTtRQUV0QyxNQUFNLEVBQUUsU0FBUyxFQUFFLEdBQUcsYUFBYSxFQUFFLEdBQUcsT0FBTyxDQUFDO1FBQ2hELE1BQU0sY0FBYyxHQUFHLE9BQU8sQ0FBQyxTQUFTLElBQUksc0JBQXNCLENBQUM7UUFFbkUsT0FBTztZQUNMLDhCQUFhLENBQUMsUUFBUSxDQUFDLEVBQUUsSUFBSSxFQUFFLGFBQWEsRUFBRSxDQUFDO1lBQy9DO2dCQUNFLElBQUksRUFBRSxnQkFBZ0I7Z0JBQ3RCLElBQUksRUFBRSw4QkFBOEI7Z0JBQ3BDLElBQUksRUFBRSxFQUFFLElBQUksRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLFdBQVcsRUFBRTthQUNsRDtZQUNEO2dCQUNFLElBQUksRUFBRSxhQUFhO2dCQUNuQixHQUFHLEVBQUUsUUFBUSxXQUFXLElBQUksY0FBYyxtQkFBbUIsV0FBVyxJQUFJLGNBQWMsbUNBQW1DO2FBQzlIO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxpQkFBaUIsQ0FDN0IsT0FBaUM7UUFFakMsTUFBTSxZQUFZLEdBQUcsT0FBTyxDQUFDLFlBQVksQ0FBQztRQUMxQyxNQUFNLFVBQVUsR0FBRyxPQUFPLENBQUMsVUFBVSxJQUFJLGtCQUFrQixZQUFZLEVBQUUsQ0FBQztRQUMxRSxNQUFNLE1BQU0sR0FBRyxPQUFPLENBQUMsTUFBTSxJQUFJLFdBQVcsQ0FBQztRQUM3QyxNQUFNLFFBQVEsR0FBRyxPQUFPLENBQUMsUUFBUSxJQUFJLHFCQUFxQixDQUFDO1FBQzNELE1BQU0sV0FBVyxHQUFHLE9BQU8sQ0FBQyxXQUFXLElBQUksdUNBQTJCLENBQUM7UUFDdkUsTUFBTSxTQUFTLEdBQUcsR0FBRyxXQUFXLENBQUMsSUFBSSxLQUFLLFdBQVcsQ0FBQyxLQUFLLEdBQUcsQ0FBQztRQUMvRCxNQUFNLHNCQUFzQixHQUFHLE9BQU8sQ0FBQyxzQkFBc0I7YUFDMUQsT0FBTyxFQUFFO2FBQ1QsUUFBUSxDQUFDLEdBQUcsQ0FBQztZQUNkLENBQUMsQ0FBQyxPQUFPLENBQUMsc0JBQXNCLENBQUMsT0FBTyxFQUFFO1lBQzFDLENBQUMsQ0FBQyxHQUFHLE9BQU8sQ0FBQyxzQkFBc0IsQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDO1FBRW5ELE1BQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQztRQUN2QyxNQUFNLFdBQVcsR0FBRztZQUNsQixHQUFHLHNCQUFzQixpQ0FBaUM7WUFDMUQsRUFBRTtZQUNGLG1CQUFtQixPQUFPLEVBQUU7WUFDNUIsRUFBRTtZQUNGLFFBQVE7WUFDUixFQUFFO1lBQ0YsNkNBQTZDLFlBQVksYUFBYTtTQUN2RSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUViLE9BQU87WUFDTDtnQkFDRSxJQUFJLEVBQUUsUUFBUTtnQkFDZCxFQUFFLEVBQUUsTUFBTTtnQkFDVixJQUFJLEVBQUUsb0NBQW9DO2dCQUMxQyxJQUFJLEVBQUU7b0JBQ0osS0FBSyxFQUFFLE9BQU8sQ0FBQyxXQUFXLEVBQUUsUUFBUTtvQkFDcEMsZ0JBQWdCLEVBQUUsR0FBRyxLQUFLLE9BQU8sV0FBVyxFQUFFO29CQUM5QyxNQUFNLEVBQUUsVUFBVTtvQkFDbEIsSUFBSSxFQUFFLE9BQU8sQ0FBQyxVQUFVO29CQUN4QixLQUFLLEVBQUUsS0FBSztvQkFDWixNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksU0FBUztvQkFDOUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLFNBQVM7b0JBQ3BELElBQUksRUFBRSxXQUFXO29CQUNqQixNQUFNLEVBQUUsU0FBUztvQkFDakIsU0FBUyxFQUFFLFNBQVM7b0JBQ3BCLE9BQU8sRUFBRSxPQUFPLENBQUMsT0FBTyxJQUFJLElBQUk7aUJBQ2pDO2FBQ0Y7U0FDRixDQUFDO0lBQ0osQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFlO1FBQzVDLE9BQU8sQ0FBQyw4QkFBYSxDQUFDLGdCQUFnQixDQUFDLEVBQUUsV0FBVyxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUMvRCxDQUFDOztBQXRJSCwwQ0F1SUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBHaXRJZGVudGl0eSwgR2l0aHViQ3JlZGVudGlhbHMgfSBmcm9tIFwiLlwiO1xuaW1wb3J0IHsgREVGQVVMVF9HSVRIVUJfQUNUSU9OU19VU0VSIH0gZnJvbSBcIi4vY29uc3RhbnRzXCI7XG5pbXBvcnQgeyBDaGVja291dFdpdGgsIFdvcmtmbG93U3RlcHMgfSBmcm9tIFwiLi93b3JrZmxvdy1zdGVwc1wiO1xuaW1wb3J0IHsgSm9iU3RlcCB9IGZyb20gXCIuL3dvcmtmbG93cy1tb2RlbFwiO1xuXG5mdW5jdGlvbiBjb250ZXh0KHZhbHVlOiBzdHJpbmcpIHtcbiAgcmV0dXJuIGBcXCR7eyAke3ZhbHVlfSB9fWA7XG59XG5cbmNvbnN0IFJFUE8gPSBjb250ZXh0KFwiZ2l0aHViLnJlcG9zaXRvcnlcIik7XG5jb25zdCBSVU5fSUQgPSBjb250ZXh0KFwiZ2l0aHViLnJ1bl9pZFwiKTtcbmNvbnN0IFNFUlZFUl9VUkwgPSBjb250ZXh0KFwiZ2l0aHViLnNlcnZlcl91cmxcIik7XG5jb25zdCBSVU5fVVJMID0gYCR7U0VSVkVSX1VSTH0vJHtSRVBPfS9hY3Rpb25zL3J1bnMvJHtSVU5fSUR9YDtcbmNvbnN0IEdJVF9QQVRDSF9GSUxFX0RFRkFVTFQgPSBcIi5yZXBvLnBhdGNoXCI7XG5jb25zdCBSVU5ORVJfVEVNUCA9IFwiJHt7IHJ1bm5lci50ZW1wIH19XCI7XG5cbi8qKlxuICogQSBzZXQgb2YgdXRpbGl0eSBmdW5jdGlvbnMgZm9yIGNyZWF0aW5nIEdpdEh1YiBhY3Rpb25zIGluIHdvcmtmbG93cy5cbiAqL1xuZXhwb3J0IGNsYXNzIFdvcmtmbG93QWN0aW9ucyB7XG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgLnBhdGNoIGZpbGUgZnJvbSB0aGUgY3VycmVudCBnaXQgZGlmZiBhbmQgdXBsb2FkcyBpdCBhcyBhblxuICAgKiBhcnRpZmFjdC4gVXNlIGBjaGVja291dFdpdGhQYXRjaGAgdG8gZG93bmxvYWQgYW5kIGFwcGx5IGluIGFub3RoZXIgam9iLlxuICAgKlxuICAgKiBJZiBhIHBhdGNoIHdhcyB1cGxvYWRlZCwgdGhlIGFjdGlvbiBjYW4gb3B0aW9uYWxseSBmYWlsIHRoZSBqb2IuXG4gICAqXG4gICAqIEBwYXJhbSBvcHRpb25zIE9wdGlvbnNcbiAgICogQHJldHVybnMgSm9iIHN0ZXBzXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHVwbG9hZEdpdFBhdGNoKG9wdGlvbnM6IFVwbG9hZEdpdFBhdGNoT3B0aW9ucyk6IEpvYlN0ZXBbXSB7XG4gICAgY29uc3QgTVVUQVRJT05TX0ZPVU5EID0gYHN0ZXBzLiR7b3B0aW9ucy5zdGVwSWR9Lm91dHB1dHMuJHtvcHRpb25zLm91dHB1dE5hbWV9YDtcbiAgICBjb25zdCBHSVRfUEFUQ0hfRklMRSA9IG9wdGlvbnMucGF0Y2hGaWxlID8/IEdJVF9QQVRDSF9GSUxFX0RFRkFVTFQ7XG5cbiAgICBjb25zdCBzdGVwczogSm9iU3RlcFtdID0gW1xuICAgICAge1xuICAgICAgICBpZDogb3B0aW9ucy5zdGVwSWQsXG4gICAgICAgIG5hbWU6IG9wdGlvbnMuc3RlcE5hbWUgPz8gXCJGaW5kIG11dGF0aW9uc1wiLFxuICAgICAgICBydW46IFtcbiAgICAgICAgICBcImdpdCBhZGQgLlwiLFxuICAgICAgICAgIGBnaXQgZGlmZiAtLXN0YWdlZCAtLXBhdGNoIC0tZXhpdC1jb2RlID4gJHtHSVRfUEFUQ0hfRklMRX0gfHwgZWNobyBcIiR7b3B0aW9ucy5vdXRwdXROYW1lfT10cnVlXCIgPj4gJEdJVEhVQl9PVVRQVVRgLFxuICAgICAgICBdLmpvaW4oXCJcXG5cIiksXG4gICAgICB9LFxuICAgICAgV29ya2Zsb3dTdGVwcy51cGxvYWRBcnRpZmFjdCh7XG4gICAgICAgIGlmOiBNVVRBVElPTlNfRk9VTkQsXG4gICAgICAgIG5hbWU6IFwiVXBsb2FkIHBhdGNoXCIsXG4gICAgICAgIHdpdGg6IHsgbmFtZTogR0lUX1BBVENIX0ZJTEUsIHBhdGg6IEdJVF9QQVRDSF9GSUxFIH0sXG4gICAgICB9KSxcbiAgICBdO1xuXG4gICAgaWYgKG9wdGlvbnMubXV0YXRpb25FcnJvcikge1xuICAgICAgc3RlcHMucHVzaCh7XG4gICAgICAgIG5hbWU6IFwiRmFpbCBidWlsZCBvbiBtdXRhdGlvblwiLFxuICAgICAgICBpZjogTVVUQVRJT05TX0ZPVU5ELFxuICAgICAgICBydW46IFtcbiAgICAgICAgICBgZWNobyBcIjo6ZXJyb3I6OiR7b3B0aW9ucy5tdXRhdGlvbkVycm9yfVwiYCxcbiAgICAgICAgICBgY2F0ICR7R0lUX1BBVENIX0ZJTEV9YCxcbiAgICAgICAgICBcImV4aXQgMVwiLFxuICAgICAgICBdLmpvaW4oXCJcXG5cIiksXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICByZXR1cm4gc3RlcHM7XG4gIH1cbiAgLyoqXG4gICAqIENoZWNrcyBvdXQgYSByZXBvc2l0b3J5IGFuZCBhcHBsaWVzIGEgZ2l0IHBhdGNoIHRoYXQgd2FzIGNyZWF0ZWQgdXNpbmdcbiAgICogYHVwbG9hZEdpdFBhdGNoYC5cbiAgICpcbiAgICogQHBhcmFtIG9wdGlvbnMgT3B0aW9uc1xuICAgKiBAcmV0dXJucyBKb2Igc3RlcHNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgY2hlY2tvdXRXaXRoUGF0Y2goXG4gICAgb3B0aW9uczogQ2hlY2tvdXRXaXRoUGF0Y2hPcHRpb25zID0ge31cbiAgKTogSm9iU3RlcFtdIHtcbiAgICBjb25zdCB7IHBhdGNoRmlsZSwgLi4ucmVzdE9mT3B0aW9ucyB9ID0gb3B0aW9ucztcbiAgICBjb25zdCBHSVRfUEFUQ0hfRklMRSA9IG9wdGlvbnMucGF0Y2hGaWxlID8/IEdJVF9QQVRDSF9GSUxFX0RFRkFVTFQ7XG5cbiAgICByZXR1cm4gW1xuICAgICAgV29ya2Zsb3dTdGVwcy5jaGVja291dCh7IHdpdGg6IHJlc3RPZk9wdGlvbnMgfSksXG4gICAgICB7XG4gICAgICAgIG5hbWU6IFwiRG93bmxvYWQgcGF0Y2hcIixcbiAgICAgICAgdXNlczogXCJhY3Rpb25zL2Rvd25sb2FkLWFydGlmYWN0QHY0XCIsXG4gICAgICAgIHdpdGg6IHsgbmFtZTogR0lUX1BBVENIX0ZJTEUsIHBhdGg6IFJVTk5FUl9URU1QIH0sXG4gICAgICB9LFxuICAgICAge1xuICAgICAgICBuYW1lOiBcIkFwcGx5IHBhdGNoXCIsXG4gICAgICAgIHJ1bjogYFsgLXMgJHtSVU5ORVJfVEVNUH0vJHtHSVRfUEFUQ0hfRklMRX0gXSAmJiBnaXQgYXBwbHkgJHtSVU5ORVJfVEVNUH0vJHtHSVRfUEFUQ0hfRklMRX0gfHwgZWNobyBcIkVtcHR5IHBhdGNoLiBTa2lwcGluZy5cImAsXG4gICAgICB9LFxuICAgIF07XG4gIH1cblxuICAvKipcbiAgICogQSBzdGVwIHRoYXQgY3JlYXRlcyBhIHB1bGwgcmVxdWVzdCBiYXNlZCBvbiB0aGUgY3VycmVudCByZXBvIHN0YXRlLlxuICAgKlxuICAgKiBAcGFyYW0gb3B0aW9ucyBPcHRpb25zXG4gICAqIEByZXR1cm5zIEpvYiBzdGVwc1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBjcmVhdGVQdWxsUmVxdWVzdChcbiAgICBvcHRpb25zOiBDcmVhdGVQdWxsUmVxdWVzdE9wdGlvbnNcbiAgKTogSm9iU3RlcFtdIHtcbiAgICBjb25zdCB3b3JrZmxvd05hbWUgPSBvcHRpb25zLndvcmtmbG93TmFtZTtcbiAgICBjb25zdCBicmFuY2hOYW1lID0gb3B0aW9ucy5icmFuY2hOYW1lID8/IGBnaXRodWItYWN0aW9ucy8ke3dvcmtmbG93TmFtZX1gO1xuICAgIGNvbnN0IHN0ZXBJZCA9IG9wdGlvbnMuc3RlcElkID8/IFwiY3JlYXRlLXByXCI7XG4gICAgY29uc3Qgc3RlcE5hbWUgPSBvcHRpb25zLnN0ZXBOYW1lID8/IFwiQ3JlYXRlIFB1bGwgUmVxdWVzdFwiO1xuICAgIGNvbnN0IGdpdElkZW50aXR5ID0gb3B0aW9ucy5naXRJZGVudGl0eSA/PyBERUZBVUxUX0dJVEhVQl9BQ1RJT05TX1VTRVI7XG4gICAgY29uc3QgY29tbWl0dGVyID0gYCR7Z2l0SWRlbnRpdHkubmFtZX0gPCR7Z2l0SWRlbnRpdHkuZW1haWx9PmA7XG4gICAgY29uc3QgcHVsbFJlcXVlc3REZXNjcmlwdGlvbiA9IG9wdGlvbnMucHVsbFJlcXVlc3REZXNjcmlwdGlvblxuICAgICAgLnRyaW1FbmQoKVxuICAgICAgLmVuZHNXaXRoKFwiLlwiKVxuICAgICAgPyBvcHRpb25zLnB1bGxSZXF1ZXN0RGVzY3JpcHRpb24udHJpbUVuZCgpXG4gICAgICA6IGAke29wdGlvbnMucHVsbFJlcXVlc3REZXNjcmlwdGlvbi50cmltRW5kKCl9LmA7XG5cbiAgICBjb25zdCB0aXRsZSA9IG9wdGlvbnMucHVsbFJlcXVlc3RUaXRsZTtcbiAgICBjb25zdCBkZXNjcmlwdGlvbiA9IFtcbiAgICAgIGAke3B1bGxSZXF1ZXN0RGVzY3JpcHRpb259IFNlZSBkZXRhaWxzIGluIFt3b3JrZmxvdyBydW5dLmAsXG4gICAgICBcIlwiLFxuICAgICAgYFtXb3JrZmxvdyBSdW5dOiAke1JVTl9VUkx9YCxcbiAgICAgIFwiXCIsXG4gICAgICBcIi0tLS0tLVwiLFxuICAgICAgXCJcIixcbiAgICAgIGAqQXV0b21hdGljYWxseSBjcmVhdGVkIGJ5IHByb2plbiB2aWEgdGhlIFwiJHt3b3JrZmxvd05hbWV9XCIgd29ya2Zsb3cqYCxcbiAgICBdLmpvaW4oXCJcXG5cIik7XG5cbiAgICByZXR1cm4gW1xuICAgICAge1xuICAgICAgICBuYW1lOiBzdGVwTmFtZSxcbiAgICAgICAgaWQ6IHN0ZXBJZCxcbiAgICAgICAgdXNlczogXCJwZXRlci1ldmFucy9jcmVhdGUtcHVsbC1yZXF1ZXN0QHY2XCIsXG4gICAgICAgIHdpdGg6IHtcbiAgICAgICAgICB0b2tlbjogb3B0aW9ucy5jcmVkZW50aWFscz8udG9rZW5SZWYsXG4gICAgICAgICAgXCJjb21taXQtbWVzc2FnZVwiOiBgJHt0aXRsZX1cXG5cXG4ke2Rlc2NyaXB0aW9ufWAsXG4gICAgICAgICAgYnJhbmNoOiBicmFuY2hOYW1lLFxuICAgICAgICAgIGJhc2U6IG9wdGlvbnMuYmFzZUJyYW5jaCxcbiAgICAgICAgICB0aXRsZTogdGl0bGUsXG4gICAgICAgICAgbGFiZWxzOiBvcHRpb25zLmxhYmVscz8uam9pbihcIixcIikgfHwgdW5kZWZpbmVkLFxuICAgICAgICAgIGFzc2lnbmVlczogb3B0aW9ucy5hc3NpZ25lZXM/LmpvaW4oXCIsXCIpIHx8IHVuZGVmaW5lZCxcbiAgICAgICAgICBib2R5OiBkZXNjcmlwdGlvbixcbiAgICAgICAgICBhdXRob3I6IGNvbW1pdHRlcixcbiAgICAgICAgICBjb21taXR0ZXI6IGNvbW1pdHRlcixcbiAgICAgICAgICBzaWdub2ZmOiBvcHRpb25zLnNpZ25vZmYgPz8gdHJ1ZSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb25maWd1cmVzIHRoZSBnaXQgaWRlbnRpdHkgKHVzZXIgbmFtZSBhbmQgZW1haWwpLlxuICAgKiBAcGFyYW0gaWQgVGhlIGlkZW50aXR5IHRvIHVzZVxuICAgKiBAcmV0dXJucyBKb2Igc3RlcHNcbiAgICpcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBXb3JrZmxvd1N0ZXBzLnNldHVwR2l0SWRlbnRpdHlgIGluc3RlYWRcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgc2V0dXBHaXRJZGVudGl0eShpZDogR2l0SWRlbnRpdHkpOiBKb2JTdGVwW10ge1xuICAgIHJldHVybiBbV29ya2Zsb3dTdGVwcy5zZXR1cEdpdElkZW50aXR5KHsgZ2l0SWRlbnRpdHk6IGlkIH0pXTtcbiAgfVxufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGBjaGVja291dFdpdGhQYXRjaGAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ2hlY2tvdXRXaXRoUGF0Y2hPcHRpb25zIGV4dGVuZHMgQ2hlY2tvdXRXaXRoIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBhcnRpZmFjdCB0aGUgcGF0Y2ggaXMgc3RvcmVkIGFzLlxuICAgKiBAZGVmYXVsdCBcIi5yZXBvLnBhdGNoXCJcbiAgICovXG4gIHJlYWRvbmx5IHBhdGNoRmlsZT86IHN0cmluZztcbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBgdXBsb2FkR2l0UGF0Y2hgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFVwbG9hZEdpdFBhdGNoT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgc3RlcCBJRCB3aGljaCBwcm9kdWNlcyB0aGUgb3V0cHV0IHdoaWNoIGluZGljYXRlcyBpZiBhIHBhdGNoIHdhcyBjcmVhdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgc3RlcElkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBzdGVwLlxuICAgKiBAZGVmYXVsdCBcIkZpbmQgbXV0YXRpb25zXCJcbiAgICovXG4gIHJlYWRvbmx5IHN0ZXBOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgYXJ0aWZhY3QgdGhlIHBhdGNoIGlzIHN0b3JlZCBhcy5cbiAgICogQGRlZmF1bHQgXCIucmVwby5wYXRjaFwiXG4gICAqL1xuICByZWFkb25seSBwYXRjaEZpbGU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBvdXRwdXQgdG8gZW1pdC4gSXQgd2lsbCBiZSBzZXQgdG8gYHRydWVgIGlmIHRoZXJlIHdhcyBhIGRpZmYuXG4gICAqL1xuICByZWFkb25seSBvdXRwdXROYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEZhaWwgaWYgYSBtdXRhdGlvbiB3YXMgZm91bmQgYW5kIHByaW50IHRoaXMgZXJyb3IgbWVzc2FnZS5cbiAgICogQGRlZmF1bHQgLSBkbyBub3QgZmFpbCB1cG9uIG11dGF0aW9uXG4gICAqL1xuICByZWFkb25seSBtdXRhdGlvbkVycm9yPzogc3RyaW5nO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIENyZWF0ZVB1bGxSZXF1ZXN0T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgc3RlcCBJRCB3aGljaCBwcm9kdWNlcyB0aGUgb3V0cHV0IHdoaWNoIGluZGljYXRlcyBpZiBhIHBhdGNoIHdhcyBjcmVhdGVkLlxuICAgKiBAZGVmYXVsdCBcImNyZWF0ZV9wclwiXG4gICAqL1xuICByZWFkb25seSBzdGVwSWQ/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBzdGVwIGRpc3BsYXllZCBvbiBHaXRIdWIuXG4gICAqIEBkZWZhdWx0IFwiQ3JlYXRlIFB1bGwgUmVxdWVzdFwiXG4gICAqL1xuICByZWFkb25seSBzdGVwTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGpvYiBjcmVkZW50aWFscyB1c2VkIHRvIGNyZWF0ZSB0aGUgcHVsbCByZXF1ZXN0LlxuICAgKlxuICAgKiBQcm92aWRlZCBjcmVkZW50aWFscyBtdXN0IGhhdmUgcGVybWlzc2lvbnMgdG8gY3JlYXRlIGEgcHVsbCByZXF1ZXN0IG9uIHRoZSByZXBvc2l0b3J5LlxuICAgKi9cbiAgcmVhZG9ubHkgY3JlZGVudGlhbHM/OiBHaXRodWJDcmVkZW50aWFscztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIHdvcmtmbG93IHRoYXQgd2lsbCBjcmVhdGUgdGhlIFBSXG4gICAqL1xuICByZWFkb25seSB3b3JrZmxvd05hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGZ1bGwgdGl0bGUgdXNlZCB0byBjcmVhdGUgdGhlIHB1bGwgcmVxdWVzdC5cbiAgICpcbiAgICogSWYgUFIgdGl0bGVzIGFyZSB2YWxpZGF0ZWQgaW4gdGhpcyByZXBvLCB0aGUgdGl0bGUgc2hvdWxkIGNvbXBseSB3aXRoIHRoZSByZXNwZWN0aXZlIHJ1bGVzLlxuICAgKi9cbiAgcmVhZG9ubHkgcHVsbFJlcXVlc3RUaXRsZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBEZXNjcmlwdGlvbiBhZGRlZCB0byB0aGUgcHVsbCByZXF1ZXN0LlxuICAgKlxuICAgKiBQcm92aWRlbmNlIGluZm9ybWF0aW9uIGFyZSBhdXRvbWF0aWNhbGx5IGFkZGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgcHVsbFJlcXVlc3REZXNjcmlwdGlvbjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBTZXRzIHRoZSBwdWxsIHJlcXVlc3QgYmFzZSBicmFuY2guXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gVGhlIGJyYW5jaCBjaGVja2VkIG91dCBpbiB0aGUgd29ya2Zsb3cuXG4gICAqL1xuICByZWFkb25seSBiYXNlQnJhbmNoPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcHVsbCByZXF1ZXN0IGJyYW5jaCBuYW1lLlxuICAgKlxuICAgKiBAZGVmYXVsdCBgZ2l0aHViLWFjdGlvbnMvJHtvcHRpb25zLndvcmtmbG93TmFtZX1gXG4gICAqL1xuICByZWFkb25seSBicmFuY2hOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgZ2l0IGlkZW50aXR5IHVzZWQgdG8gY3JlYXRlIHRoZSBjb21taXQuXG4gICAqIEBkZWZhdWx0IC0gdGhlIGRlZmF1bHQgZ2l0aHViLWFjdGlvbnMgdXNlclxuICAgKi9cbiAgcmVhZG9ubHkgZ2l0SWRlbnRpdHk/OiBHaXRJZGVudGl0eTtcblxuICAvKipcbiAgICogQWRkIFNpZ25lZC1vZmYtYnkgbGluZSBieSB0aGUgY29tbWl0dGVyIGF0IHRoZSBlbmQgb2YgdGhlIGNvbW1pdCBsb2cgbWVzc2FnZS5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgc2lnbm9mZj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIExhYmVscyB0byBhcHBseSBvbiB0aGUgUFIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gbGFiZWxzLlxuICAgKi9cbiAgcmVhZG9ubHkgbGFiZWxzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIEFzc2lnbmVlcyB0byBhZGQgb24gdGhlIFBSLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGFzc2lnbmVlc1xuICAgKi9cbiAgcmVhZG9ubHkgYXNzaWduZWVzPzogc3RyaW5nW107XG59XG4iXX0=