"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkflowSteps = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const object_1 = require("../util/object");
/**
 * A collection of very commonly used, individual, GitHub Workflow Job steps.
 */
class WorkflowSteps {
    /**
     * Checks out a repository.
     *
     * @param options Options to configure the `checkout` JobStep
     * @returns A JobStep that checks out a repository
     */
    static checkout(options = {}) {
        const checkoutWith = (0, object_1.removeNullOrUndefinedProperties)({
            "fetch-depth": options?.with?.fetchDepth,
            token: options?.with?.token,
            ref: options?.with?.ref,
            repository: options?.with?.repository,
            ...(options?.with?.lfs ? { lfs: true } : {}),
        });
        return {
            ...this.buildJobStepConfig({
                ...options,
                name: options.name ?? "Checkout",
            }),
            uses: "actions/checkout@v4",
            with: Object.keys(checkoutWith).length > 0 ? checkoutWith : undefined,
        };
    }
    /**
     * Configures the git identity (user name and email).
     *
     * @param options Options to configure the git identity JobStep
     * @returns Job step that configures the provided git identity
     */
    static setupGitIdentity(options) {
        return {
            ...this.buildJobStepConfig({
                ...options,
                name: options.name ?? "Set git identity",
            }),
            run: [
                `git config user.name "${options.gitIdentity.name}"`,
                `git config user.email "${options.gitIdentity.email}"`,
            ].join("\n"),
        };
    }
    /**
     * Checks if a tag exists.
     *
     * Requires a checkout step to have been run before this step with "fetch-depth" set to "0".
     *
     * Outputs:
     * - `exists`: A string value of 'true' or 'false' indicating if the tag exists.
     *
     * @param tag The tag to check. You may use valid bash code instead of a literal string in this field.
     * @param options Options to configure the `tag-exists` JobStep
     * @returns Job step that checks if the provided tag exists
     */
    static tagExists(tag, options) {
        const checkTag = (remoteTag) => `git ls-remote -q --exit-code --tags origin ${remoteTag}`;
        const varIsSet = (variable) => `[ ! -z "$${variable}" ]`;
        const setOutput = (value) => `(echo "exists=${value ? "true" : "false"}" >> $GITHUB_OUTPUT)`;
        return {
            ...this.buildJobStepConfig({
                ...options,
                name: options.name ?? "Check if tag exists",
                id: options.id ?? "check-tag",
            }),
            run: [
                `TAG=${tag}`,
                `(${varIsSet("TAG")} && ${checkTag("$TAG")} && ${setOutput(true)}) || ${setOutput(false)}`,
                "cat $GITHUB_OUTPUT",
            ].join("\n"),
        };
    }
    /**
     * Uploads an artifact.
     *
     * @param options Options to configure the `upload-artifact` JobStep
     * @returns A JobStep that uploads an artifact
     */
    static uploadArtifact(options) {
        const uploadArtifactWith = (0, object_1.removeNullOrUndefinedProperties)({
            name: options?.with?.name,
            path: options?.with?.path,
            overwrite: options?.with?.overwrite ?? true,
            "if-no-files-found": options?.with?.ifNoFilesFound,
            "retention-days": options?.with?.retentionDays,
            "compression-level": options?.with?.compressionLevel,
        });
        return {
            ...this.buildJobStepConfig({
                ...options,
                name: options.name ?? "Upload artifact",
            }),
            uses: "actions/upload-artifact@v4",
            with: uploadArtifactWith,
        };
    }
    /**
     * Downloads an artifact.
     *
     * @param options Options to configure the `download-artifact` JobStep
     * @returns A JobStep that downloads an artifact
     */
    static downloadArtifact(options) {
        const downloadArtifactWith = options?.with
            ? (0, object_1.removeNullOrUndefinedProperties)({
                name: options?.with?.name,
                path: options?.with?.path,
                pattern: options?.with?.pattern,
                repository: options?.with?.repository,
                "merge-multiple": options?.with?.mergeMultiple,
                "github-token": options?.with?.token,
                "run-id": options?.with?.runId,
            })
            : undefined;
        return {
            ...this.buildJobStepConfig({
                ...options,
                name: options?.name ?? "Download artifact",
            }),
            uses: "actions/download-artifact@v4",
            with: downloadArtifactWith,
        };
    }
    /**
     * Simple adapter to ensure we only include the necessary fields for a JobStepConfiguration.
     */
    static buildJobStepConfig(options) {
        return {
            continueOnError: options?.continueOnError,
            env: options?.env,
            id: options?.id,
            if: options?.if,
            name: options?.name,
            timeoutMinutes: options?.timeoutMinutes,
            workingDirectory: options?.workingDirectory,
        };
    }
}
exports.WorkflowSteps = WorkflowSteps;
_a = JSII_RTTI_SYMBOL_1;
WorkflowSteps[_a] = { fqn: "projen.github.WorkflowSteps", version: "0.79.27" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid29ya2Zsb3ctc3RlcHMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvZ2l0aHViL3dvcmtmbG93LXN0ZXBzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBRUEsMkNBQWlFO0FBRWpFOztHQUVHO0FBQ0gsTUFBYSxhQUFhO0lBQ3hCOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLFFBQVEsQ0FBQyxVQUEyQixFQUFFO1FBQ2xELE1BQU0sWUFBWSxHQUFHLElBQUEsd0NBQStCLEVBQUM7WUFDbkQsYUFBYSxFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUUsVUFBVTtZQUN4QyxLQUFLLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxLQUFLO1lBQzNCLEdBQUcsRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLEdBQUc7WUFDdkIsVUFBVSxFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUUsVUFBVTtZQUNyQyxHQUFHLENBQUMsT0FBTyxFQUFFLElBQUksRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsR0FBRyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7U0FDN0MsQ0FBQyxDQUFDO1FBRUgsT0FBTztZQUNMLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUFDO2dCQUN6QixHQUFHLE9BQU87Z0JBQ1YsSUFBSSxFQUFFLE9BQU8sQ0FBQyxJQUFJLElBQUksVUFBVTthQUNqQyxDQUFDO1lBQ0YsSUFBSSxFQUFFLHFCQUFxQjtZQUMzQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDdEUsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFnQztRQUM3RCxPQUFPO1lBQ0wsR0FBRyxJQUFJLENBQUMsa0JBQWtCLENBQUM7Z0JBQ3pCLEdBQUcsT0FBTztnQkFDVixJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUksSUFBSSxrQkFBa0I7YUFDekMsQ0FBQztZQUNGLEdBQUcsRUFBRTtnQkFDSCx5QkFBeUIsT0FBTyxDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUc7Z0JBQ3BELDBCQUEwQixPQUFPLENBQUMsV0FBVyxDQUFDLEtBQUssR0FBRzthQUN2RCxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7U0FDYixDQUFDO0lBQ0osQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksTUFBTSxDQUFDLFNBQVMsQ0FBQyxHQUFXLEVBQUUsT0FBNkI7UUFDaEUsTUFBTSxRQUFRLEdBQUcsQ0FBQyxTQUFpQixFQUFFLEVBQUUsQ0FDckMsOENBQThDLFNBQVMsRUFBRSxDQUFDO1FBQzVELE1BQU0sUUFBUSxHQUFHLENBQUMsUUFBZ0IsRUFBRSxFQUFFLENBQUMsWUFBWSxRQUFRLEtBQUssQ0FBQztRQUNqRSxNQUFNLFNBQVMsR0FBRyxDQUFDLEtBQWMsRUFBRSxFQUFFLENBQ25DLGlCQUFpQixLQUFLLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsT0FBTyxzQkFBc0IsQ0FBQztRQUVsRSxPQUFPO1lBQ0wsR0FBRyxJQUFJLENBQUMsa0JBQWtCLENBQUM7Z0JBQ3pCLEdBQUcsT0FBTztnQkFDVixJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUksSUFBSSxxQkFBcUI7Z0JBQzNDLEVBQUUsRUFBRSxPQUFPLENBQUMsRUFBRSxJQUFJLFdBQVc7YUFDOUIsQ0FBQztZQUNGLEdBQUcsRUFBRTtnQkFDSCxPQUFPLEdBQUcsRUFBRTtnQkFDWixJQUFJLFFBQVEsQ0FBQyxLQUFLLENBQUMsT0FBTyxRQUFRLENBQUMsTUFBTSxDQUFDLE9BQU8sU0FBUyxDQUN4RCxJQUFJLENBQ0wsUUFBUSxTQUFTLENBQUMsS0FBSyxDQUFDLEVBQUU7Z0JBQzNCLG9CQUFvQjthQUNyQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7U0FDYixDQUFDO0lBQ0osQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLGNBQWMsQ0FBQyxPQUE4QjtRQUN6RCxNQUFNLGtCQUFrQixHQUN0QixJQUFBLHdDQUErQixFQUFDO1lBQzlCLElBQUksRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLElBQUk7WUFDekIsSUFBSSxFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUUsSUFBSTtZQUN6QixTQUFTLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxTQUFTLElBQUksSUFBSTtZQUMzQyxtQkFBbUIsRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLGNBQWM7WUFDbEQsZ0JBQWdCLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxhQUFhO1lBQzlDLG1CQUFtQixFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUUsZ0JBQWdCO1NBQ3JELENBQUMsQ0FBQztRQUVMLE9BQU87WUFDTCxHQUFHLElBQUksQ0FBQyxrQkFBa0IsQ0FBQztnQkFDekIsR0FBRyxPQUFPO2dCQUNWLElBQUksRUFBRSxPQUFPLENBQUMsSUFBSSxJQUFJLGlCQUFpQjthQUN4QyxDQUFDO1lBQ0YsSUFBSSxFQUFFLDRCQUE0QjtZQUNsQyxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsZ0JBQWdCLENBQUMsT0FBaUM7UUFDOUQsTUFBTSxvQkFBb0IsR0FBcUMsT0FBTyxFQUFFLElBQUk7WUFDMUUsQ0FBQyxDQUFDLElBQUEsd0NBQStCLEVBQUM7Z0JBQzlCLElBQUksRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLElBQUk7Z0JBQ3pCLElBQUksRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLElBQUk7Z0JBQ3pCLE9BQU8sRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLE9BQU87Z0JBQy9CLFVBQVUsRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLFVBQVU7Z0JBQ3JDLGdCQUFnQixFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUUsYUFBYTtnQkFDOUMsY0FBYyxFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUUsS0FBSztnQkFDcEMsUUFBUSxFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUUsS0FBSzthQUMvQixDQUFDO1lBQ0osQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUVkLE9BQU87WUFDTCxHQUFHLElBQUksQ0FBQyxrQkFBa0IsQ0FBQztnQkFDekIsR0FBRyxPQUFPO2dCQUNWLElBQUksRUFBRSxPQUFPLEVBQUUsSUFBSSxJQUFJLG1CQUFtQjthQUMzQyxDQUFDO1lBQ0YsSUFBSSxFQUFFLDhCQUE4QjtZQUNwQyxJQUFJLEVBQUUsb0JBQW9CO1NBQzNCLENBQUM7SUFDSixDQUFDO0lBRUQ7O09BRUc7SUFDSyxNQUFNLENBQUMsa0JBQWtCLENBQy9CLE9BQTZCO1FBRTdCLE9BQU87WUFDTCxlQUFlLEVBQUUsT0FBTyxFQUFFLGVBQWU7WUFDekMsR0FBRyxFQUFFLE9BQU8sRUFBRSxHQUFHO1lBQ2pCLEVBQUUsRUFBRSxPQUFPLEVBQUUsRUFBRTtZQUNmLEVBQUUsRUFBRSxPQUFPLEVBQUUsRUFBRTtZQUNmLElBQUksRUFBRSxPQUFPLEVBQUUsSUFBSTtZQUNuQixjQUFjLEVBQUUsT0FBTyxFQUFFLGNBQWM7WUFDdkMsZ0JBQWdCLEVBQUUsT0FBTyxFQUFFLGdCQUFnQjtTQUM1QyxDQUFDO0lBQ0osQ0FBQzs7QUF2Skgsc0NBd0pDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgR2l0SWRlbnRpdHkgfSBmcm9tIFwiLi90YXNrLXdvcmtmbG93XCI7XG5pbXBvcnQgeyBKb2JTdGVwQ29uZmlndXJhdGlvbiwgSm9iU3RlcCB9IGZyb20gXCIuL3dvcmtmbG93cy1tb2RlbFwiO1xuaW1wb3J0IHsgcmVtb3ZlTnVsbE9yVW5kZWZpbmVkUHJvcGVydGllcyB9IGZyb20gXCIuLi91dGlsL29iamVjdFwiO1xuXG4vKipcbiAqIEEgY29sbGVjdGlvbiBvZiB2ZXJ5IGNvbW1vbmx5IHVzZWQsIGluZGl2aWR1YWwsIEdpdEh1YiBXb3JrZmxvdyBKb2Igc3RlcHMuXG4gKi9cbmV4cG9ydCBjbGFzcyBXb3JrZmxvd1N0ZXBzIHtcbiAgLyoqXG4gICAqIENoZWNrcyBvdXQgYSByZXBvc2l0b3J5LlxuICAgKlxuICAgKiBAcGFyYW0gb3B0aW9ucyBPcHRpb25zIHRvIGNvbmZpZ3VyZSB0aGUgYGNoZWNrb3V0YCBKb2JTdGVwXG4gICAqIEByZXR1cm5zIEEgSm9iU3RlcCB0aGF0IGNoZWNrcyBvdXQgYSByZXBvc2l0b3J5XG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGNoZWNrb3V0KG9wdGlvbnM6IENoZWNrb3V0T3B0aW9ucyA9IHt9KTogSm9iU3RlcCB7XG4gICAgY29uc3QgY2hlY2tvdXRXaXRoID0gcmVtb3ZlTnVsbE9yVW5kZWZpbmVkUHJvcGVydGllcyh7XG4gICAgICBcImZldGNoLWRlcHRoXCI6IG9wdGlvbnM/LndpdGg/LmZldGNoRGVwdGgsXG4gICAgICB0b2tlbjogb3B0aW9ucz8ud2l0aD8udG9rZW4sXG4gICAgICByZWY6IG9wdGlvbnM/LndpdGg/LnJlZixcbiAgICAgIHJlcG9zaXRvcnk6IG9wdGlvbnM/LndpdGg/LnJlcG9zaXRvcnksXG4gICAgICAuLi4ob3B0aW9ucz8ud2l0aD8ubGZzID8geyBsZnM6IHRydWUgfSA6IHt9KSxcbiAgICB9KTtcblxuICAgIHJldHVybiB7XG4gICAgICAuLi50aGlzLmJ1aWxkSm9iU3RlcENvbmZpZyh7XG4gICAgICAgIC4uLm9wdGlvbnMsXG4gICAgICAgIG5hbWU6IG9wdGlvbnMubmFtZSA/PyBcIkNoZWNrb3V0XCIsXG4gICAgICB9KSxcbiAgICAgIHVzZXM6IFwiYWN0aW9ucy9jaGVja291dEB2NFwiLFxuICAgICAgd2l0aDogT2JqZWN0LmtleXMoY2hlY2tvdXRXaXRoKS5sZW5ndGggPiAwID8gY2hlY2tvdXRXaXRoIDogdW5kZWZpbmVkLFxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogQ29uZmlndXJlcyB0aGUgZ2l0IGlkZW50aXR5ICh1c2VyIG5hbWUgYW5kIGVtYWlsKS5cbiAgICpcbiAgICogQHBhcmFtIG9wdGlvbnMgT3B0aW9ucyB0byBjb25maWd1cmUgdGhlIGdpdCBpZGVudGl0eSBKb2JTdGVwXG4gICAqIEByZXR1cm5zIEpvYiBzdGVwIHRoYXQgY29uZmlndXJlcyB0aGUgcHJvdmlkZWQgZ2l0IGlkZW50aXR5XG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHNldHVwR2l0SWRlbnRpdHkob3B0aW9uczogU2V0dXBHaXRJZGVudGl0eU9wdGlvbnMpOiBKb2JTdGVwIHtcbiAgICByZXR1cm4ge1xuICAgICAgLi4udGhpcy5idWlsZEpvYlN0ZXBDb25maWcoe1xuICAgICAgICAuLi5vcHRpb25zLFxuICAgICAgICBuYW1lOiBvcHRpb25zLm5hbWUgPz8gXCJTZXQgZ2l0IGlkZW50aXR5XCIsXG4gICAgICB9KSxcbiAgICAgIHJ1bjogW1xuICAgICAgICBgZ2l0IGNvbmZpZyB1c2VyLm5hbWUgXCIke29wdGlvbnMuZ2l0SWRlbnRpdHkubmFtZX1cImAsXG4gICAgICAgIGBnaXQgY29uZmlnIHVzZXIuZW1haWwgXCIke29wdGlvbnMuZ2l0SWRlbnRpdHkuZW1haWx9XCJgLFxuICAgICAgXS5qb2luKFwiXFxuXCIpLFxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2tzIGlmIGEgdGFnIGV4aXN0cy5cbiAgICpcbiAgICogUmVxdWlyZXMgYSBjaGVja291dCBzdGVwIHRvIGhhdmUgYmVlbiBydW4gYmVmb3JlIHRoaXMgc3RlcCB3aXRoIFwiZmV0Y2gtZGVwdGhcIiBzZXQgdG8gXCIwXCIuXG4gICAqXG4gICAqIE91dHB1dHM6XG4gICAqIC0gYGV4aXN0c2A6IEEgc3RyaW5nIHZhbHVlIG9mICd0cnVlJyBvciAnZmFsc2UnIGluZGljYXRpbmcgaWYgdGhlIHRhZyBleGlzdHMuXG4gICAqXG4gICAqIEBwYXJhbSB0YWcgVGhlIHRhZyB0byBjaGVjay4gWW91IG1heSB1c2UgdmFsaWQgYmFzaCBjb2RlIGluc3RlYWQgb2YgYSBsaXRlcmFsIHN0cmluZyBpbiB0aGlzIGZpZWxkLlxuICAgKiBAcGFyYW0gb3B0aW9ucyBPcHRpb25zIHRvIGNvbmZpZ3VyZSB0aGUgYHRhZy1leGlzdHNgIEpvYlN0ZXBcbiAgICogQHJldHVybnMgSm9iIHN0ZXAgdGhhdCBjaGVja3MgaWYgdGhlIHByb3ZpZGVkIHRhZyBleGlzdHNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgdGFnRXhpc3RzKHRhZzogc3RyaW5nLCBvcHRpb25zOiBKb2JTdGVwQ29uZmlndXJhdGlvbik6IEpvYlN0ZXAge1xuICAgIGNvbnN0IGNoZWNrVGFnID0gKHJlbW90ZVRhZzogc3RyaW5nKSA9PlxuICAgICAgYGdpdCBscy1yZW1vdGUgLXEgLS1leGl0LWNvZGUgLS10YWdzIG9yaWdpbiAke3JlbW90ZVRhZ31gO1xuICAgIGNvbnN0IHZhcklzU2V0ID0gKHZhcmlhYmxlOiBzdHJpbmcpID0+IGBbICEgLXogXCIkJHt2YXJpYWJsZX1cIiBdYDtcbiAgICBjb25zdCBzZXRPdXRwdXQgPSAodmFsdWU6IGJvb2xlYW4pID0+XG4gICAgICBgKGVjaG8gXCJleGlzdHM9JHt2YWx1ZSA/IFwidHJ1ZVwiIDogXCJmYWxzZVwifVwiID4+ICRHSVRIVUJfT1VUUFVUKWA7XG5cbiAgICByZXR1cm4ge1xuICAgICAgLi4udGhpcy5idWlsZEpvYlN0ZXBDb25maWcoe1xuICAgICAgICAuLi5vcHRpb25zLFxuICAgICAgICBuYW1lOiBvcHRpb25zLm5hbWUgPz8gXCJDaGVjayBpZiB0YWcgZXhpc3RzXCIsXG4gICAgICAgIGlkOiBvcHRpb25zLmlkID8/IFwiY2hlY2stdGFnXCIsXG4gICAgICB9KSxcbiAgICAgIHJ1bjogW1xuICAgICAgICBgVEFHPSR7dGFnfWAsXG4gICAgICAgIGAoJHt2YXJJc1NldChcIlRBR1wiKX0gJiYgJHtjaGVja1RhZyhcIiRUQUdcIil9ICYmICR7c2V0T3V0cHV0KFxuICAgICAgICAgIHRydWVcbiAgICAgICAgKX0pIHx8ICR7c2V0T3V0cHV0KGZhbHNlKX1gLFxuICAgICAgICBcImNhdCAkR0lUSFVCX09VVFBVVFwiLFxuICAgICAgXS5qb2luKFwiXFxuXCIpLFxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogVXBsb2FkcyBhbiBhcnRpZmFjdC5cbiAgICpcbiAgICogQHBhcmFtIG9wdGlvbnMgT3B0aW9ucyB0byBjb25maWd1cmUgdGhlIGB1cGxvYWQtYXJ0aWZhY3RgIEpvYlN0ZXBcbiAgICogQHJldHVybnMgQSBKb2JTdGVwIHRoYXQgdXBsb2FkcyBhbiBhcnRpZmFjdFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyB1cGxvYWRBcnRpZmFjdChvcHRpb25zOiBVcGxvYWRBcnRpZmFjdE9wdGlvbnMpOiBKb2JTdGVwIHtcbiAgICBjb25zdCB1cGxvYWRBcnRpZmFjdFdpdGg6IFVwbG9hZEFydGlmYWN0V2l0aCA9XG4gICAgICByZW1vdmVOdWxsT3JVbmRlZmluZWRQcm9wZXJ0aWVzKHtcbiAgICAgICAgbmFtZTogb3B0aW9ucz8ud2l0aD8ubmFtZSxcbiAgICAgICAgcGF0aDogb3B0aW9ucz8ud2l0aD8ucGF0aCxcbiAgICAgICAgb3ZlcndyaXRlOiBvcHRpb25zPy53aXRoPy5vdmVyd3JpdGUgPz8gdHJ1ZSxcbiAgICAgICAgXCJpZi1uby1maWxlcy1mb3VuZFwiOiBvcHRpb25zPy53aXRoPy5pZk5vRmlsZXNGb3VuZCxcbiAgICAgICAgXCJyZXRlbnRpb24tZGF5c1wiOiBvcHRpb25zPy53aXRoPy5yZXRlbnRpb25EYXlzLFxuICAgICAgICBcImNvbXByZXNzaW9uLWxldmVsXCI6IG9wdGlvbnM/LndpdGg/LmNvbXByZXNzaW9uTGV2ZWwsXG4gICAgICB9KTtcblxuICAgIHJldHVybiB7XG4gICAgICAuLi50aGlzLmJ1aWxkSm9iU3RlcENvbmZpZyh7XG4gICAgICAgIC4uLm9wdGlvbnMsXG4gICAgICAgIG5hbWU6IG9wdGlvbnMubmFtZSA/PyBcIlVwbG9hZCBhcnRpZmFjdFwiLFxuICAgICAgfSksXG4gICAgICB1c2VzOiBcImFjdGlvbnMvdXBsb2FkLWFydGlmYWN0QHY0XCIsXG4gICAgICB3aXRoOiB1cGxvYWRBcnRpZmFjdFdpdGgsXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEb3dubG9hZHMgYW4gYXJ0aWZhY3QuXG4gICAqXG4gICAqIEBwYXJhbSBvcHRpb25zIE9wdGlvbnMgdG8gY29uZmlndXJlIHRoZSBgZG93bmxvYWQtYXJ0aWZhY3RgIEpvYlN0ZXBcbiAgICogQHJldHVybnMgQSBKb2JTdGVwIHRoYXQgZG93bmxvYWRzIGFuIGFydGlmYWN0XG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGRvd25sb2FkQXJ0aWZhY3Qob3B0aW9ucz86IERvd25sb2FkQXJ0aWZhY3RPcHRpb25zKTogSm9iU3RlcCB7XG4gICAgY29uc3QgZG93bmxvYWRBcnRpZmFjdFdpdGg6IERvd25sb2FkQXJ0aWZhY3RXaXRoIHwgdW5kZWZpbmVkID0gb3B0aW9ucz8ud2l0aFxuICAgICAgPyByZW1vdmVOdWxsT3JVbmRlZmluZWRQcm9wZXJ0aWVzKHtcbiAgICAgICAgICBuYW1lOiBvcHRpb25zPy53aXRoPy5uYW1lLFxuICAgICAgICAgIHBhdGg6IG9wdGlvbnM/LndpdGg/LnBhdGgsXG4gICAgICAgICAgcGF0dGVybjogb3B0aW9ucz8ud2l0aD8ucGF0dGVybixcbiAgICAgICAgICByZXBvc2l0b3J5OiBvcHRpb25zPy53aXRoPy5yZXBvc2l0b3J5LFxuICAgICAgICAgIFwibWVyZ2UtbXVsdGlwbGVcIjogb3B0aW9ucz8ud2l0aD8ubWVyZ2VNdWx0aXBsZSxcbiAgICAgICAgICBcImdpdGh1Yi10b2tlblwiOiBvcHRpb25zPy53aXRoPy50b2tlbixcbiAgICAgICAgICBcInJ1bi1pZFwiOiBvcHRpb25zPy53aXRoPy5ydW5JZCxcbiAgICAgICAgfSlcbiAgICAgIDogdW5kZWZpbmVkO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIC4uLnRoaXMuYnVpbGRKb2JTdGVwQ29uZmlnKHtcbiAgICAgICAgLi4ub3B0aW9ucyxcbiAgICAgICAgbmFtZTogb3B0aW9ucz8ubmFtZSA/PyBcIkRvd25sb2FkIGFydGlmYWN0XCIsXG4gICAgICB9KSxcbiAgICAgIHVzZXM6IFwiYWN0aW9ucy9kb3dubG9hZC1hcnRpZmFjdEB2NFwiLFxuICAgICAgd2l0aDogZG93bmxvYWRBcnRpZmFjdFdpdGgsXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTaW1wbGUgYWRhcHRlciB0byBlbnN1cmUgd2Ugb25seSBpbmNsdWRlIHRoZSBuZWNlc3NhcnkgZmllbGRzIGZvciBhIEpvYlN0ZXBDb25maWd1cmF0aW9uLlxuICAgKi9cbiAgcHJpdmF0ZSBzdGF0aWMgYnVpbGRKb2JTdGVwQ29uZmlnKFxuICAgIG9wdGlvbnM6IEpvYlN0ZXBDb25maWd1cmF0aW9uXG4gICk6IEpvYlN0ZXBDb25maWd1cmF0aW9uIHtcbiAgICByZXR1cm4ge1xuICAgICAgY29udGludWVPbkVycm9yOiBvcHRpb25zPy5jb250aW51ZU9uRXJyb3IsXG4gICAgICBlbnY6IG9wdGlvbnM/LmVudixcbiAgICAgIGlkOiBvcHRpb25zPy5pZCxcbiAgICAgIGlmOiBvcHRpb25zPy5pZixcbiAgICAgIG5hbWU6IG9wdGlvbnM/Lm5hbWUsXG4gICAgICB0aW1lb3V0TWludXRlczogb3B0aW9ucz8udGltZW91dE1pbnV0ZXMsXG4gICAgICB3b3JraW5nRGlyZWN0b3J5OiBvcHRpb25zPy53b3JraW5nRGlyZWN0b3J5LFxuICAgIH07XG4gIH1cbn1cbmV4cG9ydCBpbnRlcmZhY2UgQ2hlY2tvdXRPcHRpb25zIGV4dGVuZHMgSm9iU3RlcENvbmZpZ3VyYXRpb24ge1xuICAvKipcbiAgICogT3B0aW9ucyBmb3IgYGNoZWNrb3V0YC5cbiAgICovXG4gIHJlYWRvbmx5IHdpdGg/OiBDaGVja291dFdpdGg7XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgYGNoZWNrb3V0YC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDaGVja291dFdpdGgge1xuICAvKipcbiAgICogTnVtYmVyIG9mIGNvbW1pdHMgdG8gZmV0Y2guIDAgaW5kaWNhdGVzIGFsbCBoaXN0b3J5IGZvciBhbGwgYnJhbmNoZXMgYW5kIHRhZ3MuXG4gICAqXG4gICAqIEBkZWZhdWx0IDFcbiAgICovXG4gIHJlYWRvbmx5IGZldGNoRGVwdGg/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBXaGV0aGVyIExGUyBpcyBlbmFibGVkIGZvciB0aGUgR2l0SHViIHJlcG9zaXRvcnlcbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGxmcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEJyYW5jaCBvciB0YWcgbmFtZS5cbiAgICogQGRlZmF1bHQgLSB0aGUgZGVmYXVsdCBicmFuY2ggaXMgaW1wbGljaXRseSB1c2VkXG4gICAqL1xuICByZWFkb25seSByZWY/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSByZXBvc2l0b3J5IChvd25lci9yZXBvKSB0byB1c2UuXG4gICAqIEBkZWZhdWx0IC0gdGhlIGRlZmF1bHQgcmVwb3NpdG9yeSBpcyBpbXBsaWNpdGx5IHVzZWRcbiAgICovXG4gIHJlYWRvbmx5IHJlcG9zaXRvcnk/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEEgR2l0SHViIHRva2VuIHRvIHVzZSB3aGVuIGNoZWNraW5nIG91dCB0aGUgcmVwb3NpdG9yeS5cbiAgICpcbiAgICogSWYgdGhlIGludGVudCBpcyB0byBwdXNoIGNoYW5nZXMgYmFjayB0byB0aGUgYnJhbmNoLCB0aGVuIHlvdSBtdXN0IHVzZSBhXG4gICAqIFBBVCB3aXRoIGByZXBvYCAoYW5kIHBvc3NpYmx5IGB3b3JrZmxvd3NgKSBwZXJtaXNzaW9ucy5cbiAgICogQGRlZmF1bHQgLSB0aGUgZGVmYXVsdCBHSVRIVUJfVE9LRU4gaXMgaW1wbGljaXRseSB1c2VkXG4gICAqL1xuICByZWFkb25seSB0b2tlbj86IHN0cmluZztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBTZXR1cEdpdElkZW50aXR5T3B0aW9ucyBleHRlbmRzIEpvYlN0ZXBDb25maWd1cmF0aW9uIHtcbiAgLyoqXG4gICAqIFRoZSBpZGVudGl0eSB0byB1c2UuXG4gICAqL1xuICByZWFkb25seSBnaXRJZGVudGl0eTogR2l0SWRlbnRpdHk7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgVXBsb2FkQXJ0aWZhY3RXaXRoIHtcbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIGFydGlmYWN0IHRvIHVwbG9hZC5cbiAgICpcbiAgICogQGRlZmF1bHQgXCJhcnRpZmFjdFwiXG4gICAqL1xuICByZWFkb25seSBuYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBIGZpbGUsIGRpcmVjdG9yeSBvciB3aWxkY2FyZCBwYXR0ZXJuIHRoYXQgZGVzY3JpYmVzIHdoYXQgdG8gdXBsb2FkXG4gICAqL1xuICByZWFkb25seSBwYXRoOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBkZXNpcmVkIGJlaGF2aW9yIGlmIG5vIGZpbGVzIGFyZSBmb3VuZCB1c2luZyB0aGUgcHJvdmlkZWQgcGF0aC5cbiAgICogQXZhaWxhYmxlIE9wdGlvbnM6XG4gICAqICAgd2FybjogT3V0cHV0IGEgd2FybmluZyBidXQgZG8gbm90IGZhaWwgdGhlIGFjdGlvblxuICAgKiAgIGVycm9yOiBGYWlsIHRoZSBhY3Rpb24gd2l0aCBhbiBlcnJvciBtZXNzYWdlXG4gICAqICAgaWdub3JlOiBEbyBub3Qgb3V0cHV0IGFueSB3YXJuaW5ncyBvciBlcnJvcnMsIHRoZSBhY3Rpb24gZG9lcyBub3QgZmFpbFxuICAgKlxuICAgKiBAZGVmYXVsdCBcIndhcm5cIlxuICAgKi9cbiAgcmVhZG9ubHkgaWZOb0ZpbGVzRm91bmQ/OiBcImVycm9yXCIgfCBcIndhcm5cIiB8IFwiaWdub3JlXCI7XG5cbiAgLyoqXG4gICAqIER1cmF0aW9uIGFmdGVyIHdoaWNoIGFydGlmYWN0IHdpbGwgZXhwaXJlIGluIGRheXMuIDAgbWVhbnMgdXNpbmcgZGVmYXVsdCByZXBvc2l0b3J5IHJldGVudGlvbi5cbiAgICpcbiAgICogTWluaW11bSAxIGRheS5cbiAgICogTWF4aW11bSA5MCBkYXlzIHVubGVzcyBjaGFuZ2VkIGZyb20gdGhlIHJlcG9zaXRvcnkgc2V0dGluZ3MgcGFnZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBUaGUgZGVmYXVsdCByZXBvc2l0b3J5IHJldGVudGlvblxuICAgKi9cbiAgcmVhZG9ubHkgcmV0ZW50aW9uRGF5cz86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIGxldmVsIG9mIGNvbXByZXNzaW9uIGZvciBabGliIHRvIGJlIGFwcGxpZWQgdG8gdGhlIGFydGlmYWN0IGFyY2hpdmUuXG4gICAqXG4gICAqIFRoZSB2YWx1ZSBjYW4gcmFuZ2UgZnJvbSAwIHRvIDkuXG4gICAqIEZvciBsYXJnZSBmaWxlcyB0aGF0IGFyZSBub3QgZWFzaWx5IGNvbXByZXNzZWQsIGEgdmFsdWUgb2YgMCBpcyByZWNvbW1lbmRlZCBmb3Igc2lnbmlmaWNhbnRseSBmYXN0ZXIgdXBsb2Fkcy5cbiAgICpcbiAgICogQGRlZmF1bHQgNlxuICAgKi9cbiAgcmVhZG9ubHkgY29tcHJlc3Npb25MZXZlbD86IG51bWJlcjtcblxuICAvKipcbiAgICogV2hldGhlciBhY3Rpb24gc2hvdWxkIG92ZXJ3cml0ZSBhbiBleGlzdGluZyBhcnRpZmFjdCB3aXRoIHRoZSBzYW1lIG5hbWUgKHNob3VsZCBvbmUgZXhpc3QpXG4gICAqXG4gICAqIEludHJvZHVjZWQgaW4gdjQgYW5kIHJlcHJlc2VudHMgYSBicmVha2luZyBjaGFuZ2UgZnJvbSB0aGUgYmVoYXZpb3Igb2YgdGhlIHYzIGFjdGlvbi5cbiAgICogVG8gbWFpbnRhaW4gYmFja3dhcmRzIGNvbXBhdGliaWxpdHkgd2l0aCBleGlzdGluZywgdGhpcyBzaG91bGQgYmUgc2V0IHRoZSBgdHJ1ZWAgKHRoZSBkZWZhdWx0KS5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgb3ZlcndyaXRlPzogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBVcGxvYWRBcnRpZmFjdE9wdGlvbnMgZXh0ZW5kcyBKb2JTdGVwQ29uZmlndXJhdGlvbiB7XG4gIC8qKlxuICAgKiBPcHRpb25zIGZvciBgdXBsb2FkLWFydGlmYWN0YC5cbiAgICovXG4gIHJlYWRvbmx5IHdpdGg6IFVwbG9hZEFydGlmYWN0V2l0aDtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBEb3dubG9hZEFydGlmYWN0V2l0aCB7XG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoZSBhcnRpZmFjdCB0byBkb3dubG9hZFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIElmIHVuc3BlY2lmaWVkLCBhbGwgYXJ0aWZhY3RzIGZvciB0aGUgcnVuIGFyZSBkb3dubG9hZGVkXG4gICAqL1xuICByZWFkb25seSBuYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBIGZpbGUsIGRpcmVjdG9yeSBvciB3aWxkY2FyZCBwYXR0ZXJuIHRoYXQgZGVzY3JpYmVzIHdoYXQgdG8gZG93bmxvYWQuIFN1cHBvcnRzIGJhc2ljIHRpbGRlIGV4cGFuc2lvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSAkR0lUSFVCX1dPUktTUEFDRVxuICAgKi9cbiAgcmVhZG9ubHkgcGF0aD86IHN0cmluZztcblxuICAvKipcbiAgICogQSBnbG9iIHBhdHRlcm4gdG8gdGhlIGFydGlmYWN0cyB0aGF0IHNob3VsZCBiZSBkb3dubG9hZGVkXG4gICAqIFRoaXMgaXMgaWdub3JlZCBpZiBuYW1lIGlzIHNwZWNpZmllZFxuICAgKi9cbiAgcmVhZG9ubHkgcGF0dGVybj86IHN0cmluZztcblxuICAvKipcbiAgICogV2hlbiBtdWx0aXBsZSBhcnRpZmFjdHMgYXJlIG1hdGNoZWQsIHRoaXMgY2hhbmdlcyB0aGUgYmVoYXZpb3Igb2YgdGhlIGRlc3RpbmF0aW9uIGRpcmVjdG9yaWVzXG4gICAqIElmIHRydWUsIHRoZSBkb3dubG9hZGVkIGFydGlmYWN0cyB3aWxsIGJlIGluIHRoZSBzYW1lIGRpcmVjdG9yeSBzcGVjaWZpZWQgYnkgcGF0aFxuICAgKiBJZiBmYWxzZSwgdGhlIGRvd25sb2FkZWQgYXJ0aWZhY3RzIHdpbGwgYmUgZXh0cmFjdGVkIGludG8gaW5kaXZpZHVhbCBuYW1lZCBkaXJlY3RvcmllcyB3aXRoaW4gdGhlIHNwZWNpZmllZCBwYXRoXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBtZXJnZU11bHRpcGxlPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIEdpdEh1YiB0b2tlbiB1c2VkIHRvIGF1dGhlbnRpY2F0ZSB3aXRoIHRoZSBHaXRIdWIgQVBJIHRvIGRvd25sb2FkIGFydGlmYWN0cyBmcm9tIGEgZGlmZmVyZW50IHJlcG9zaXRvcnkgb3IgZnJvbSBhIGRpZmZlcmVudCB3b3JrZmxvdyBydW5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBJZiB1bnNwZWNpZmllZCwgdGhlIGFjdGlvbiB3aWxsIGRvd25sb2FkIGFydGlmYWN0cyBmcm9tIHRoZSBjdXJyZW50IHJlcG8gYW5kIHRoZSBjdXJyZW50IHdvcmtmbG93IHJ1blxuICAgKi9cbiAgcmVhZG9ubHkgdG9rZW4/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSByZXBvc2l0b3J5IG93bmVyIGFuZCB0aGUgcmVwb3NpdG9yeSBuYW1lIGpvaW5lZCB0b2dldGhlciBieSBcIi9cIlxuICAgKiBJZiBnaXRodWItdG9rZW4gaXMgc3BlY2lmaWVkLCB0aGlzIGlzIHRoZSByZXBvc2l0b3J5IHRoYXQgYXJ0aWZhY3RzIHdpbGwgYmUgZG93bmxvYWRlZCBmcm9tXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gJHt7IGdpdGh1Yi5yZXBvc2l0b3J5IH19XG4gICAqL1xuICByZWFkb25seSByZXBvc2l0b3J5Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgaWQgb2YgdGhlIHdvcmtmbG93IHJ1biB3aGVyZSB0aGUgZGVzaXJlZCBkb3dubG9hZCBhcnRpZmFjdCB3YXMgdXBsb2FkZWQgZnJvbVxuICAgKiBJZiBnaXRodWItdG9rZW4gaXMgc3BlY2lmaWVkLCB0aGlzIGlzIHRoZSBydW4gdGhhdCBhcnRpZmFjdHMgd2lsbCBiZSBkb3dubG9hZGVkIGZyb21cbiAgICpcbiAgICogQGRlZmF1bHQgLSAke3sgZ2l0aHViLnJ1bl9pZCB9fVxuICAgKi9cbiAgcmVhZG9ubHkgcnVuSWQ/OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgRG93bmxvYWRBcnRpZmFjdE9wdGlvbnMgZXh0ZW5kcyBKb2JTdGVwQ29uZmlndXJhdGlvbiB7XG4gIC8qKlxuICAgKiBPcHRpb25zIGZvciBgZG93bmxvYWQtYXJ0aWZhY3RgLlxuICAgKi9cbiAgcmVhZG9ubHkgd2l0aDogRG93bmxvYWRBcnRpZmFjdFdpdGg7XG59XG4iXX0=