"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const case_1 = require("case");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const util_1 = require("../util");
const yaml_1 = require("../yaml");
/**
 * Workflow for GitHub.
 *
 * A workflow is a configurable automated process made up of one or more jobs.
 *
 * @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions
 */
class GithubWorkflow extends component_1.Component {
    constructor(github, name, options = {}) {
        super(github.project, `${new.target.name}#${name}`);
        this.events = {};
        this.jobs = {};
        this.name = name;
        this.concurrency = options.concurrency;
        this.projenCredentials = github.projenCredentials;
        this.actions = github.actions;
        const workflowsEnabled = github.workflowsEnabled || options.force;
        if (workflowsEnabled) {
            this.file = new yaml_1.YamlFile(this.project, `.github/workflows/${name.toLocaleLowerCase()}.yml`, {
                obj: () => this.renderWorkflow(),
                // GitHub needs to read the file from the repository in order to work.
                committed: true,
            });
        }
    }
    /**
     * Add events to triggers the workflow.
     *
     * @param events The event(s) to trigger the workflow.
     */
    on(events) {
        this.events = {
            ...this.events,
            ...events,
        };
    }
    /**
     * Adds a single job to the workflow.
     * @param id The job name (unique within the workflow)
     * @param job The job specification
     */
    addJob(id, job) {
        this.addJobs({ [id]: job });
    }
    /**
     * Add jobs to the workflow.
     *
     * @param jobs Jobs to add.
     */
    addJobs(jobs) {
        verifyJobConstraints(jobs);
        this.jobs = {
            ...this.jobs,
            ...jobs,
        };
    }
    /**
     * Get a single job from the workflow.
     * @param id The job name (unique within the workflow)
     */
    getJob(id) {
        return this.jobs[id];
    }
    /**
     * Updates a single job to the workflow.
     * @param id The job name (unique within the workflow)
     */
    updateJob(id, job) {
        this.updateJobs({ [id]: job });
    }
    /**
     * Updates jobs for this worklow
     * Does a complete replace, it does not try to merge the jobs
     *
     * @param jobs Jobs to update.
     */
    updateJobs(jobs) {
        verifyJobConstraints(jobs);
        const newJobIds = Object.keys(jobs);
        const updatedJobs = Object.entries(this.jobs).map(([jobId, job]) => {
            if (newJobIds.includes(jobId)) {
                return [jobId, jobs[jobId]];
            }
            return [jobId, job];
        });
        this.jobs = {
            ...Object.fromEntries(updatedJobs),
        };
    }
    /**
     * Removes a single job to the workflow.
     * @param id The job name (unique within the workflow)
     */
    removeJob(id) {
        const updatedJobs = Object.entries(this.jobs).filter(([jobId]) => jobId !== id);
        this.jobs = {
            ...Object.fromEntries(updatedJobs),
        };
    }
    renderWorkflow() {
        return {
            name: this.name,
            "run-name": this.runName,
            on: snakeCaseKeys(this.events),
            concurrency: this.concurrency,
            jobs: renderJobs(this.jobs, this.actions),
        };
    }
}
exports.GithubWorkflow = GithubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GithubWorkflow[_a] = { fqn: "projen.github.GithubWorkflow", version: "0.79.27" };
function snakeCaseKeys(obj) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(snakeCaseKeys);
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === "object" && v != null) {
            v = snakeCaseKeys(v);
        }
        result[(0, case_1.snake)(k)] = v;
    }
    return result;
}
function renderJobs(jobs, actions) {
    const result = {};
    for (const [name, job] of Object.entries(jobs)) {
        result[name] = renderJob(job);
    }
    return result;
    /** @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions */
    function renderJob(job) {
        const steps = new Array();
        // https://docs.github.com/en/actions/using-workflows/reusing-workflows#supported-keywords-for-jobs-that-call-a-reusable-workflow
        if ("uses" in job) {
            return {
                name: job.name,
                needs: arrayOrScalar(job.needs),
                if: job.if,
                permissions: (0, util_1.kebabCaseKeys)(job.permissions),
                concurrency: job.concurrency,
                uses: job.uses,
                with: job.with,
                secrets: job.secrets,
                strategy: renderJobStrategy(job.strategy),
            };
        }
        if (job.tools) {
            steps.push(...setupTools(job.tools));
        }
        const userDefinedSteps = (0, util_1.kebabCaseKeys)((0, _resolve_1.resolve)(job.steps), false);
        steps.push(...userDefinedSteps);
        return {
            name: job.name,
            needs: arrayOrScalar(job.needs),
            "runs-on": arrayOrScalar(job.runsOnGroup) ?? arrayOrScalar(job.runsOn),
            permissions: (0, util_1.kebabCaseKeys)(job.permissions),
            environment: job.environment,
            concurrency: job.concurrency,
            outputs: renderJobOutputs(job.outputs),
            env: job.env,
            defaults: (0, util_1.kebabCaseKeys)(job.defaults),
            if: job.if,
            steps: steps.map(renderStep),
            "timeout-minutes": job.timeoutMinutes,
            strategy: renderJobStrategy(job.strategy),
            "continue-on-error": job.continueOnError,
            container: job.container,
            services: job.services,
        };
    }
    function renderJobOutputs(output) {
        if (output == null) {
            return undefined;
        }
        const rendered = {};
        for (const [name, { stepId, outputName }] of Object.entries(output)) {
            rendered[name] = `\${{ steps.${stepId}.outputs.${outputName} }}`;
        }
        return rendered;
    }
    function renderJobStrategy(strategy) {
        if (strategy == null) {
            return undefined;
        }
        const rendered = {
            "max-parallel": strategy.maxParallel,
            "fail-fast": strategy.failFast,
        };
        if (strategy.matrix) {
            const matrix = {
                include: strategy.matrix.include,
                exclude: strategy.matrix.exclude,
            };
            for (const [key, values] of Object.entries(strategy.matrix.domain ?? {})) {
                if (key in matrix) {
                    // A domain key was set to `include`, or `exclude`:
                    throw new Error(`Illegal job strategy matrix key: ${key}`);
                }
                matrix[key] = values;
            }
            rendered.matrix = matrix;
        }
        return rendered;
    }
    function renderStep(step) {
        return {
            name: step.name,
            id: step.id,
            if: step.if,
            uses: step.uses && actions.get(step.uses),
            env: step.env,
            run: step.run,
            with: step.with,
            "continue-on-error": step.continueOnError,
            "timeout-minutes": step.timeoutMinutes,
            "working-directory": step.workingDirectory,
        };
    }
}
function arrayOrScalar(arr) {
    if (!Array.isArray(arr)) {
        return arr;
    }
    if (arr == null || arr.length === 0) {
        return arr;
    }
    if (arr.length === 1) {
        return arr[0];
    }
    return arr;
}
function setupTools(tools) {
    const steps = [];
    if (tools.java) {
        steps.push({
            uses: "actions/setup-java@v3",
            with: { distribution: "temurin", "java-version": tools.java.version },
        });
    }
    if (tools.node) {
        steps.push({
            uses: "actions/setup-node@v4",
            with: { "node-version": tools.node.version },
        });
    }
    if (tools.python) {
        steps.push({
            uses: "actions/setup-python@v5",
            with: { "python-version": tools.python.version },
        });
    }
    if (tools.go) {
        steps.push({
            uses: "actions/setup-go@v3",
            with: { "go-version": tools.go.version },
        });
    }
    if (tools.dotnet) {
        steps.push({
            uses: "actions/setup-dotnet@v3",
            with: { "dotnet-version": tools.dotnet.version },
        });
    }
    return steps;
}
function verifyJobConstraints(jobs) {
    // verify that job has a "permissions" statement to ensure workflow can
    // operate in repos with default tokens set to readonly
    for (const [id, job] of Object.entries(jobs)) {
        if (!job.permissions) {
            throw new Error(`${id}: all workflow jobs must have a "permissions" clause to ensure workflow can operate in restricted repositories`);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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