"use strict";
// @see https://docs.gitlab.com/ee/ci/yaml/
// Schema from: https://gitlab.com/gitlab-org/gitlab/-/raw/master/app/assets/javascripts/editor/schema/ci.json
// Initial TypeScript Interfaces from: https://app.quicktype.io/#
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkflowWhen = exports.Strategy = exports.LinkType = exports.DefaultElement = exports.KubernetesEnum = exports.DeploymentTier = exports.Action = exports.JobWhen = exports.PullPolicy = exports.CacheWhen = exports.CachePolicy = void 0;
/**
 * Configure the upload and download behaviour of a cache.
 * @see https://docs.gitlab.com/ee/ci/yaml/#cachepolicy
 */
var CachePolicy;
(function (CachePolicy) {
    /** Only download the cache when the job starts, but never upload changes when the job finishes. */
    CachePolicy["PULL"] = "pull";
    /** Only upload a cache when the job finishes, but never download the cache when the job starts. */
    CachePolicy["PUSH"] = "push";
    /** The job downloads the cache when the job starts, and uploads changes to the cache when the job ends. */
    CachePolicy["PULL_PUSH"] = "pull-push";
})(CachePolicy || (exports.CachePolicy = CachePolicy = {}));
/**
 * Configure when artifacts are uploaded depended on job status.
 * @see https://docs.gitlab.com/ee/ci/yaml/#cachewhen
 */
var CacheWhen;
(function (CacheWhen) {
    /** Upload artifacts regardless of job status. */
    CacheWhen["ALWAYS"] = "always";
    /** Upload artifacts only when the job fails. */
    CacheWhen["ON_FAILURE"] = "on_failure";
    /** Upload artifacts only when the job succeeds (this is the default). */
    CacheWhen["ON_SUCCESS"] = "on_success";
})(CacheWhen || (exports.CacheWhen = CacheWhen = {}));
/**
 * Describes the conditions for when to pull an image.
 * @see https://docs.gitlab.com/ee/ci/yaml/#servicepull_policy
 */
var PullPolicy;
(function (PullPolicy) {
    PullPolicy["ALWAYS"] = "always";
    PullPolicy["NEVER"] = "never";
    PullPolicy["IF_NOT_PRESENT"] = "if-not-present";
})(PullPolicy || (exports.PullPolicy = PullPolicy = {}));
/**
 * Describes the conditions for when to run the job. Defaults to 'on_success'.
 * @see https://docs.gitlab.com/ee/ci/yaml/#when
 */
var JobWhen;
(function (JobWhen) {
    JobWhen["ALWAYS"] = "always";
    JobWhen["DELAYED"] = "delayed";
    JobWhen["MANUAL"] = "manual";
    JobWhen["NEVER"] = "never";
    JobWhen["ON_FAILURE"] = "on_failure";
    JobWhen["ON_SUCCESS"] = "on_success";
})(JobWhen || (exports.JobWhen = JobWhen = {}));
/**
 * Specifies what this job will do. 'start' (default) indicates the job will start the
 * deployment. 'prepare' indicates this will not affect the deployment. 'stop' indicates
 * this will stop the deployment.
 */
var Action;
(function (Action) {
    Action["PREPARE"] = "prepare";
    Action["START"] = "start";
    Action["STOP"] = "stop";
})(Action || (exports.Action = Action = {}));
/**
 * Explicitly specifies the tier of the deployment environment if non-standard environment
 * name is used.
 */
var DeploymentTier;
(function (DeploymentTier) {
    DeploymentTier["DEVELOPMENT"] = "development";
    DeploymentTier["OTHER"] = "other";
    DeploymentTier["PRODUCTION"] = "production";
    DeploymentTier["STAGING"] = "staging";
    DeploymentTier["TESTING"] = "testing";
})(DeploymentTier || (exports.DeploymentTier = DeploymentTier = {}));
/**
 * Filter job based on if Kubernetes integration is active.
 */
var KubernetesEnum;
(function (KubernetesEnum) {
    KubernetesEnum["ACTIVE"] = "active";
})(KubernetesEnum || (exports.KubernetesEnum = KubernetesEnum = {}));
var DefaultElement;
(function (DefaultElement) {
    DefaultElement["AFTER_SCRIPT"] = "after_script";
    DefaultElement["ARTIFACTS"] = "artifacts";
    DefaultElement["BEFORE_SCRIPT"] = "before_script";
    DefaultElement["CACHE"] = "cache";
    DefaultElement["IMAGE"] = "image";
    DefaultElement["INTERRUPTIBLE"] = "interruptible";
    DefaultElement["RETRY"] = "retry";
    DefaultElement["SERVICES"] = "services";
    DefaultElement["TAGS"] = "tags";
    DefaultElement["TIMEOUT"] = "timeout";
})(DefaultElement || (exports.DefaultElement = DefaultElement = {}));
/**
 * The content kind of what users can download via url.
 */
var LinkType;
(function (LinkType) {
    LinkType["IMAGE"] = "image";
    LinkType["OTHER"] = "other";
    LinkType["PACKAGE"] = "package";
    LinkType["RUNBOOK"] = "runbook";
})(LinkType || (exports.LinkType = LinkType = {}));
/**
 * You can mirror the pipeline status from the triggered pipeline to the source bridge job
 * by using strategy: depend
 * @see https://docs.gitlab.com/ee/ci/yaml/#triggerstrategy
 */
var Strategy;
(function (Strategy) {
    Strategy["DEPEND"] = "depend";
})(Strategy || (exports.Strategy = Strategy = {}));
/**
 * Describes the conditions for when to run the job. Defaults to 'on_success'.
 * The value can only be 'always' or 'never' when used with workflow.
 * @see https://docs.gitlab.com/ee/ci/yaml/#workflowrules
 */
var WorkflowWhen;
(function (WorkflowWhen) {
    WorkflowWhen["ALWAYS"] = "always";
    WorkflowWhen["NEVER"] = "never";
})(WorkflowWhen || (exports.WorkflowWhen = WorkflowWhen = {}));
//# sourceMappingURL=data:application/json;base64,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