"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CiConfiguration = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const case_1 = require("case");
const component_1 = require("../component");
const yaml_1 = require("../yaml");
/**
 * CI for GitLab.
 * A CI is a configurable automated process made up of one or more stages/jobs.
 * @see https://docs.gitlab.com/ee/ci/yaml/
 */
class CiConfiguration extends component_1.Component {
    get defaultCache() {
        return this._defaultCache;
    }
    constructor(project, name, options) {
        super(project);
        /**
         * Defines default scripts that should run *after* all jobs. Can be overriden by the job level `afterScript`.
         */
        this.defaultAfterScript = [];
        /**
         * Defines default scripts that should run *before* all jobs. Can be overriden by the job level `afterScript`.
         */
        this.defaultBeforeScript = [];
        /**
         * A default list of additional Docker images to run scripts in. The service image is linked to the image specified in the  image parameter.
         */
        this.defaultServices = [];
        /**
         * Used to select a specific runner from the list of all runners that are available for the project.
         */
        this.defaultTags = [];
        /**
         * Can be `Include` or `Include[]`. Each `Include` will be a string, or an
         * object with properties for the method if including external YAML file. The external
         * content will be fetched, included and evaluated along the `.gitlab-ci.yml`.
         */
        this.include = [];
        /**
         * Groups jobs into stages. All jobs in one stage must complete before next stage is
         * executed. Defaults to ['build', 'test', 'deploy'].
         */
        this.stages = [];
        /**
         * Global variables that are passed to jobs.
         * If the job already has that variable defined, the job-level variable takes precedence.
         */
        this.variables = {};
        /**
         * The jobs in the CI configuration.
         */
        this.jobs = {};
        this.name = path.parse(name).name;
        this.path =
            this.name === "gitlab-ci"
                ? ".gitlab-ci.yml"
                : `.gitlab/ci-templates/${name.toLocaleLowerCase()}.yml`;
        this.file = new yaml_1.YamlFile(this.project, this.path, {
            obj: () => this.renderCI(),
            // GitLab needs to read the file from the repository in order to work.
            committed: true,
        });
        const defaults = options?.default;
        if (defaults) {
            this.defaultAfterScript.push(...(defaults.afterScript ?? []));
            this.defaultArtifacts = defaults.artifacts;
            defaults.beforeScript &&
                this.defaultBeforeScript.push(...defaults.beforeScript);
            defaults.cache && this.addDefaultCaches(defaults.cache);
            this.defaultImage = defaults.image;
            this.defaultInterruptible = defaults.interruptible;
            this.defaultRetry = defaults.retry;
            defaults.services && this.addServices(...defaults.services);
            defaults.tags && this.defaultTags.push(...defaults.tags);
            this.defaultTimeout = defaults.timeout;
        }
        this.pages = options?.pages;
        this.workflow = options?.workflow;
        if (options?.stages) {
            this.addStages(...options.stages);
        }
        if (options?.variables) {
            this.addGlobalVariables(options.variables);
        }
        if (options?.jobs) {
            this.addJobs(options.jobs);
        }
    }
    /**
     * Add additional yml/yaml files to the CI includes
     * @param includes The includes to add.
     */
    addIncludes(...includes) {
        for (const additional of includes) {
            this.assertIsValidInclude(additional);
            for (const existing of this.include) {
                if (this.areEqualIncludes(existing, additional)) {
                    throw new Error(`${this.name}: GitLab CI ${existing} already contains one or more templates specified in ${additional}.`);
                }
            }
            this.include.push(additional);
        }
    }
    /**
     * Throw an error if the provided Include is invalid.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param include the Include to validate.
     */
    assertIsValidInclude(include) {
        const combos = [
            include.local,
            include.file && include.project,
            include.remote,
            include.template,
        ];
        const len = combos.filter((x) => Boolean(x)).length;
        if (len !== 1) {
            throw new Error(`${this.name}: GitLab CI include ${include} contains ${len} property combination(s).
        A valid include configuration specifies *one* of the following property combinations.
        * local
        * file, project
        * remote
        * template  
        `);
        }
    }
    /**
     * Check if the equality of Includes.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param x First include to compare.
     * @param y Second include to compare.
     * @returns Whether the includes are equal.
     */
    areEqualIncludes(x, y) {
        if (x.local === y.local && x.local !== undefined) {
            return true;
        }
        else if (x.template === y.template && x.template !== undefined) {
            return true;
        }
        else if (x.remote === y.remote && x.remote !== undefined) {
            return true;
        }
        else if (x.project === y.project && x.ref === y.ref) {
            const xFiles = x.file ? x.file : [];
            const yFiles = y.file ? y.file : [];
            const allFiles = xFiles.concat(yFiles);
            return new Set(allFiles).size !== allFiles.length;
        }
        return false;
    }
    /**
     * Add additional services.
     * @param services The services to add.
     */
    addServices(...services) {
        for (const additional of services) {
            for (const existing of this.defaultServices) {
                if (additional.name === existing.name &&
                    additional.alias === existing.alias) {
                    throw new Error(`${this.name}: GitLab CI already contains service ${additional}.`);
                }
            }
            this.defaultServices.push(additional);
        }
    }
    /**
     * Add a globally defined variable to the CI configuration.
     * @param variables The variables to add.
     */
    addGlobalVariables(variables) {
        for (const [key, value] of Object.entries(variables)) {
            if (this.variables[key] !== undefined) {
                throw new Error(`${this.name}: GitLab CI already contains variable ${key}.`);
            }
            this.variables[key] = value;
        }
    }
    /**
     * Add stages to the CI configuration if not already present.
     * @param stages stages to add.
     */
    addStages(...stages) {
        for (const stage of stages) {
            if (!this.stages.includes(stage)) {
                this.stages.push(stage);
            }
        }
    }
    /**
     * Add jobs and their stages to the CI configuration.
     * @param jobs Jobs to add.
     */
    addJobs(jobs) {
        for (const [key, value] of Object.entries(jobs)) {
            if (this.jobs[key] !== undefined) {
                throw new Error(`${this.name}: GitLab CI already contains job ${key}.`);
            }
            this.jobs[key] = value;
            if (value.stage) {
                this.addStages(value.stage);
            }
            if (value.cache) {
                this.assertIsValidCacheSetup(value.cache);
            }
        }
    }
    isValidCacheSetup(caches) {
        const MAX_CONFIGURABLE_CACHES = 4;
        return caches.length <= MAX_CONFIGURABLE_CACHES;
    }
    assertIsValidCacheSetup(caches) {
        if (!this.isValidCacheSetup(caches)) {
            throw new Error(`${this.name}: GitLab CI can only define up to 4 caches, got: ${caches.length}`);
        }
    }
    /**
     * Adds up to 4 default caches configuration to the CI configuration.
     * @param caches Caches to add.
     */
    addDefaultCaches(caches) {
        this.assertIsValidCacheSetup(caches);
        this._defaultCache = caches;
    }
    renderCI() {
        return {
            default: this.renderDefault(),
            include: this.include.length > 0 ? snakeCaseKeys(this.include) : undefined,
            pages: snakeCaseKeys(this.pages),
            services: this.defaultServices.length > 0
                ? snakeCaseKeys(this.defaultServices)
                : undefined,
            variables: Object.entries(this.variables).length > 0 ? this.variables : undefined,
            workflow: snakeCaseKeys(this.workflow),
            stages: this.stages.length > 0 ? this.stages : undefined,
            // we do not want to change job names
            // as they can be hidden (https://docs.gitlab.com/ee/ci/jobs/index.html#hide-jobs)
            // or referenced in extends
            ...snakeCaseKeys(this.jobs, true),
        };
    }
    renderDefault() {
        const defaults = {
            afterScript: this.defaultAfterScript.length > 0
                ? this.defaultAfterScript
                : undefined,
            artifacts: this.defaultArtifacts,
            beforeScript: this.defaultBeforeScript.length > 0
                ? this.defaultBeforeScript
                : undefined,
            cache: this.defaultCache,
            image: this.defaultImage,
            interruptible: this.defaultInterruptible,
            retry: this.defaultRetry,
            services: this.defaultServices.length > 0 ? this.defaultServices : undefined,
            tags: this.defaultTags.length > 0 ? this.defaultTags : undefined,
            timeout: this.defaultTimeout,
        };
        return Object.values(defaults).filter((x) => x).length
            ? snakeCaseKeys(defaults)
            : undefined;
    }
}
exports.CiConfiguration = CiConfiguration;
_a = JSII_RTTI_SYMBOL_1;
CiConfiguration[_a] = { fqn: "projen.gitlab.CiConfiguration", version: "0.79.27" };
function snakeCaseKeys(obj, skipTopLevel = false) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map((o) => snakeCaseKeys(o));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === "object" &&
            v != null &&
            k !== "variables" &&
            k !== "idTokens") {
            v = snakeCaseKeys(v);
        }
        result[skipTopLevel ? k : (0, case_1.snake)(k)] = v;
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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