"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readJsiiManifest = exports.toProjectType = exports.resolveProjectType = exports.readManifest = exports.discover = void 0;
const fs = require("fs");
const path = require("path");
const zlib_1 = require("zlib");
const case_1 = require("case");
const PROJEN_MODULE_ROOT = path.join(__dirname, "..");
const PROJECT_BASE_FQN = "projen.Project";
/**
 * Returns a list of project types exported the modules defined in `moduleDirs`.
 * This list will always also include the built-in projen project types.
 * Modules without a .jsii manifest are skipped.
 *
 * @param moduleDirs A list of npm module directories
 */
function discover(...moduleDirs) {
    const jsii = discoverJsiiTypes(...moduleDirs);
    const result = new Array();
    for (const fqn of Object.keys(jsii)) {
        if (isProjectType(jsii, fqn)) {
            const p = toProjectType(jsii, fqn);
            result.push(p);
        }
    }
    return result.sort((r1, r2) => r1.pjid.localeCompare(r2.pjid));
}
exports.discover = discover;
function readManifest(dir) {
    const jsiiFile = path.join(dir, ".jsii");
    if (!fs.existsSync(jsiiFile)) {
        return undefined;
    } // no jsii manifest
    let manifest = JSON.parse(fs.readFileSync(jsiiFile, "utf-8"));
    if (manifest.schema === "jsii/file-redirect") {
        const compressedFile = path.join(dir, manifest.filename);
        if (!fs.existsSync(compressedFile)) {
            throw new Error(`${compressedFile} does not exist.`);
        }
        switch (manifest.compression) {
            case "gzip":
                manifest = JSON.parse((0, zlib_1.unzipSync)(fs.readFileSync(compressedFile)).toString());
                break;
            default:
                throw new Error(`Unsupported compression format: ${manifest.compression}`);
        }
    }
    return manifest;
}
exports.readManifest = readManifest;
/**
 * Resolve all jsii types from @modulesDirs.
 * When a jsii module is found it will recusively list the types from the dependant module as well
 *
 * @param moduleDirs
 * @returns
 */
function discoverJsiiTypes(...moduleDirs) {
    const jsii = {};
    const discoveredManifests = [];
    const discoverJsii = (dir) => {
        const manifest = readManifest(dir);
        if (!manifest) {
            return;
        }
        if (discoveredManifests.includes(manifest.fingerprint)) {
            return;
        }
        discoveredManifests.push(manifest.fingerprint);
        for (const [fqn, type] of Object.entries(manifest.types)) {
            jsii[fqn] = {
                ...type,
            };
        }
        // Also search recursively in nested project dependencies. If the requested module is an external module
        // this will also end-up in the projen module and add the projen types
        if (manifest.dependencies) {
            for (const dependency of Object.keys(manifest.dependencies)) {
                const nestedDependencyFolder = path.dirname(require.resolve(`${dependency}/package.json`, {
                    paths: [dir],
                }));
                if (fs.existsSync(nestedDependencyFolder)) {
                    discoverJsii(nestedDependencyFolder);
                }
            }
        }
    };
    // read all .jsii manifests from all requested modules and merge
    // them all into a single map of fqn->type.
    for (const dir of [...moduleDirs, PROJEN_MODULE_ROOT]) {
        discoverJsii(dir);
        // Read from scoped packages
        if (dir.includes("@") && fs.lstatSync(dir).isDirectory()) {
            const childDirs = fs.readdirSync(dir).map((file) => path.join(dir, file));
            for (const child of childDirs) {
                discoverJsii(child);
            }
        }
    }
    return jsii;
}
function resolveProjectType(projectFqn) {
    let [moduleName] = projectFqn.split(".");
    if (moduleName === "projen") {
        moduleName = PROJEN_MODULE_ROOT;
    }
    // try picking the manifest. We only need the base folder but this is directly a nice check if we request from a valid jsii package
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`, {
        paths: [process.cwd()],
    });
    const moduleFolder = path.dirname(jsiiManifestFile);
    // Read all jsii types that can be loaded from this project type
    const jsii = discoverJsiiTypes(moduleFolder);
    return toProjectType(jsii, projectFqn);
}
exports.resolveProjectType = resolveProjectType;
function toProjectType(jsii, fqn) {
    if (!isProjectType(jsii, fqn)) {
        throw new Error(`Fully qualified name "${fqn}" is not a valid project type.`);
    }
    const typeinfo = jsii[fqn];
    // projen.web.ReactProject -> web.ReactProject
    const typename = fqn.substring(fqn.indexOf(".") + 1);
    // projen.web.ReactProject -> web
    // projen.Project -> projen
    const readmeFileName = typename.includes(".")
        ? typename.split(".", 1)[0]
        : typeinfo.assembly;
    // * [java](https://projen.io/docs/api/java#javaproject-) - Java project.
    const docsurl = `https://projen.io/docs/api/${readmeFileName}#${typename
        .substring(typename.indexOf(".") + 1)
        .toLowerCase()}-`;
    let pjid = typeinfo.docs?.custom?.pjid ?? (0, case_1.snake)(typename).replace(/_project$/, "");
    return {
        moduleName: typeinfo.assembly,
        typename,
        pjid,
        fqn,
        options: discoverOptions(jsii, fqn),
        docs: typeinfo.docs?.summary,
        docsurl,
    };
}
exports.toProjectType = toProjectType;
function readJsiiManifest(jsiiFqn) {
    let [moduleName] = jsiiFqn.split(".");
    if (moduleName === "projen") {
        moduleName = PROJEN_MODULE_ROOT;
    }
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`);
    return JSON.parse(fs.readFileSync(jsiiManifestFile, "utf-8"));
}
exports.readJsiiManifest = readJsiiManifest;
function discoverOptions(jsii, fqn) {
    const options = {};
    const params = jsii[fqn]?.initializer?.parameters ?? [];
    const optionsParam = params[0];
    const optionsTypeFqn = optionsParam?.type?.fqn;
    if (params.length > 1 ||
        (params.length === 1 && optionsParam?.name !== "options")) {
        throw new Error(`constructor for project ${fqn} must have a single "options" argument of a struct type. got ${JSON.stringify(params)}`);
    }
    addOptions(optionsTypeFqn);
    const opts = Object.values(options);
    return opts.sort((a, b) => a.name.localeCompare(b.name));
    function addOptions(ofqn, basePath = [], optional = false) {
        if (!ofqn) {
            return;
        }
        const struct = jsii[ofqn];
        if (!struct) {
            throw new Error(`unable to find options type ${ofqn} for project ${fqn}`);
        }
        for (const prop of struct.properties ?? []) {
            const propPath = [...basePath, prop.name];
            // protect against double-booking
            if (prop.name in options) {
                throw new Error(`duplicate option "${prop.name}" in ${fqn} (already declared in ${options[prop.name].parent})`);
            }
            let jsiiKind;
            if (prop.type?.fqn) {
                jsiiKind = jsii[prop.type?.fqn].kind; // e.g. 'class', 'interface', 'enum'
            }
            const isOptional = optional || prop.optional;
            const defaultValue = sanitizeValue(prop.docs?.default);
            const pjnew = sanitizeValue(prop.docs?.custom?.pjnew);
            // if this is a mandatory option and we have a default value,
            // or the option is tagged to be rendered with an initial value,
            // the value has to be JSON-parsable to the correct type
            const initialValue = getInitialValue(defaultValue, pjnew, isOptional);
            if (initialValue) {
                checkDefaultIsParsable(prop.name, initialValue, prop.type);
            }
            options[prop.name] = filterUndefined({
                path: propPath,
                parent: struct.name,
                name: prop.name,
                fqn: prop.type?.fqn,
                docs: prop.docs.summary,
                simpleType: prop.type ? getSimpleTypeName(prop.type) : "unknown",
                fullType: prop.type,
                kind: jsiiKind,
                jsonLike: prop.type ? isJsonLike(jsii, prop.type) : undefined,
                switch: propPath.map((p) => (0, case_1.snake)(p).replace(/_/g, "-")).join("-"),
                default: defaultValue,
                initialValue: initialValue,
                optional: isOptional,
                featured: prop.docs?.custom?.featured === "true",
                deprecated: prop.docs.stability === "deprecated" ? true : undefined,
            });
        }
        for (const ifc of struct.interfaces ?? []) {
            addOptions(ifc);
        }
    }
}
function getInitialValue(defaultValue, pjnew, isOptional = false) {
    if (pjnew) {
        return pjnew;
    }
    if (!isOptional) {
        return defaultValue;
    }
    return undefined;
}
function sanitizeValue(val) {
    if (val === "undefined") {
        return undefined;
    }
    return val;
}
function getSimpleTypeName(type) {
    if (type?.primitive) {
        return type.primitive; // e.g. 'string', 'boolean', 'number'
    }
    else if (type?.fqn) {
        return type.fqn.split(".").pop(); // projen.NodeProjectOptions -> NodeProjectOptions
    }
    else {
        // any other types such as collection types
        return "unknown";
    }
}
/**
 * Whether a value of this type is serializable into JSON.
 */
function isJsonLike(jsii, type) {
    if (type.primitive) {
        // string, boolean, number, any
        return true;
    }
    else if (type.fqn) {
        const kind = jsii[type.fqn].kind;
        if (["interface", "enum"].includes(kind)) {
            // not 'class'
            return true;
        }
    }
    else if (type.collection) {
        return isJsonLike(jsii, type.collection.elementtype);
    }
    return false;
}
function filterUndefined(obj) {
    const ret = {};
    for (const [k, v] of Object.entries(obj)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
function isProjectType(jsii, fqn) {
    const type = jsii[fqn];
    if (!type) {
        throw new Error(`Could not find project type with fqn "${fqn}" in  .jsii file.`);
    }
    if (type.kind !== "class") {
        return false;
    }
    if (type.abstract) {
        return false;
    }
    if (type.docs?.deprecated) {
        return false;
    }
    let curr = type;
    while (true) {
        if (curr.fqn === PROJECT_BASE_FQN) {
            return true;
        }
        if (!curr.base) {
            return false;
        }
        curr = jsii[curr.base];
        if (!curr) {
            return false;
        }
    }
}
function isPrimitiveArray({ collection }) {
    return Boolean(collection?.kind === "array" && collection?.elementtype.primitive);
}
function isPrimitiveOrPrimitiveArray(type) {
    return Boolean(type?.primitive || isPrimitiveArray(type));
}
function checkDefaultIsParsable(prop, value, type) {
    if (!(type && isPrimitiveOrPrimitiveArray(type))) {
        throw new Error(`required option "${prop}" with a @default must use primitive types (string, number and boolean) or a primitive array. type found is: ${JSON.stringify(type)}`);
    }
    // macros are pass-through
    if (value.startsWith("$")) {
        return;
    }
    try {
        const parsed = JSON.parse(value);
        // Primitive type
        if (typeof parsed === type.primitive) {
            return;
        }
        // Primitive array
        if (Array.isArray(parsed) && isPrimitiveArray(type)) {
            // but empty (which is okay)
            if (parsed.length === 0) {
                return;
            }
            // if first element matches the type, assume it's correct
            if (typeof parsed[0] === type?.collection?.elementtype.primitive) {
                return;
            }
        }
        // Parsed value does not match type
        throw new Error(`cannot parse @default value for mandatory option ${prop} as a ${type}: ${parsed}`);
    }
    catch (e) {
        throw new Error(`unable to JSON.parse() value "${value}" specified as @default for mandatory option "${prop}": ${e.message}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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