"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pom = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const semver_1 = require("../util/semver");
const xmlfile_1 = require("../xmlfile");
const POM_XML_ATTRS = {
    "@xsi:schemaLocation": "http://maven.apache.org/POM/4.0.0 http://maven.apache.org/xsd/maven-4.0.0.xsd",
    "@xmlns": "http://maven.apache.org/POM/4.0.0",
    "@xmlns:xsi": "http://www.w3.org/2001/XMLSchema-instance",
};
/**
 * A Project Object Model or POM is the fundamental unit of work in Maven. It is
 * an XML file that contains information about the project and configuration
 * details used by Maven to build the project.
 */
class Pom extends component_1.Component {
    constructor(project, options) {
        super(project);
        this.properties = {};
        this.repositories = [];
        this.fileName = "pom.xml";
        this.groupId = options.groupId;
        this.artifactId = options.artifactId;
        this.version = options.version;
        this.packaging = options.packaging ?? "jar";
        this.name = project.name;
        this.description = options.description;
        this.url = options.url;
        this.parentPom = options.parentPom;
        new xmlfile_1.XmlFile(project, this.fileName, { obj: () => this.synthPom() });
    }
    /**
     * Adds a key/value property to the pom.
     * @param key the key
     * @param value the value
     */
    addProperty(key, value) {
        this.properties[key] = value;
    }
    /**
     * Adds a runtime dependency.
     *
     * @param spec Format `<groupId>/<artifactId>@<semver>`
     */
    addDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.RUNTIME);
    }
    /**
     * Adds a test dependency.
     *
     * @param spec Format `<groupId>/<artifactId>@<semver>`
     */
    addTestDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.TEST);
    }
    /**
     * Adds a build plugin to the pom.
     *
     * The plug in is also added as a BUILD dep to the project.
     *
     * @param spec dependency spec (`group/artifact@version`)
     * @param options plugin options
     */
    addPlugin(spec, options = {}) {
        for (const dep of options.dependencies ?? []) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
        }
        return this.project.deps.addDependency(spec, dependencies_1.DependencyType.BUILD, options);
    }
    /**
     * Adds a repository to the pom
     * @param repository the repository to add
     */
    addRepository(repository) {
        this.repositories.push(repository);
    }
    synthPom() {
        return (0, _resolve_1.resolve)({
            project: {
                ...POM_XML_ATTRS,
                modelVersion: "4.0.0",
                groupId: this.groupId,
                artifactId: this.artifactId,
                version: this.version,
                packaging: this.packaging,
                name: this.name,
                description: this.description,
                url: this.url,
                properties: this.properties,
                parent: this.parentPom,
                ...this.synthRepositories(),
                ...this.synthDependencies(),
            },
        }, { omitEmpty: true });
    }
    synthDependencies() {
        const deps = this.project.deps.all;
        if (deps.length === 0) {
            return;
        }
        const dependencies = [];
        const plugins = [];
        for (const dep of deps) {
            switch (dep.type) {
                case dependencies_1.DependencyType.PEER:
                case dependencies_1.DependencyType.RUNTIME:
                    dependencies.push(mavenCoords(dep));
                    break;
                case dependencies_1.DependencyType.TEST:
                    dependencies.push({
                        ...mavenCoords(dep),
                        scope: "test",
                    });
                    break;
                // build maps to plugins
                case dependencies_1.DependencyType.BUILD:
                    plugins.push({
                        ...mavenCoords(dep),
                        ...pluginConfig(dep.metadata),
                    });
                    break;
                default:
                    throw new Error(`unsupported dependency type: ${dep.type}`);
            }
        }
        return {
            build: { plugins: { plugin: plugins } },
            dependencies: { dependency: dependencies },
        };
    }
    synthRepositories() {
        if (this.repositories.length === 0) {
            return;
        }
        return {
            repositories: { repository: this.repositories },
        };
    }
}
exports.Pom = Pom;
_a = JSII_RTTI_SYMBOL_1;
Pom[_a] = { fqn: "projen.java.Pom", version: "0.79.27" };
/**
 * Parses maven groupId and artifactId from a dependency name.
 *
 *     name    <=> <groupId>/<artifactId>
 *     version <=> <version>
 */
function mavenCoords(dep) {
    const name = dep.name;
    const parts = name.split("/");
    if (parts.length !== 2) {
        throw new Error(`invalid maven coordinates in dependency named "${name}". format is "<groupId>/<artifactId>". For example "org.junit.jupiter/junit-jupiter-engine"`);
    }
    return {
        groupId: parts[0],
        artifactId: parts[1],
        version: dep.version ? (0, semver_1.toMavenVersionRange)(dep.version) : undefined,
    };
}
function pluginConfig(options = {}) {
    return {
        configuration: options.configuration,
        dependencies: options.dependencies && options.dependencies.length > 0
            ? {
                dependency: options.dependencies?.map((d) => mavenCoords(dependencies_1.Dependencies.parseDependency(d))),
            }
            : undefined,
        executions: options.executions?.map((e) => ({
            execution: {
                id: e.id,
                goals: e.goals.map((goal) => ({ goal })),
                phase: e.phase,
            },
        })),
    };
}
//# sourceMappingURL=data:application/json;base64,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