"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.getJavaImport = exports.generateJavaOptionNames = exports.Projenrc = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path_1 = require("path");
const common_1 = require("../common");
const dependencies_1 = require("../dependencies");
const inventory_1 = require("../inventory");
const projenrc_1 = require("../projenrc");
/**
 * Allows writing projenrc files in java.
 *
 * This will install `org.projen/projen` as a Maven dependency and will add a
 * `synth` task which will compile & execute `main()` from
 * `src/main/java/projenrc.java`.
 */
class Projenrc extends projenrc_1.ProjenrcFile {
    constructor(project, pom, options = {}) {
        super(project);
        const projenVersion = options.projenVersion ?? common_1.PROJEN_VERSION;
        this.className = options.className ?? "projenrc";
        this.testScope = options.testScope ?? true;
        const depType = this.testScope
            ? dependencies_1.DependencyType.TEST
            : dependencies_1.DependencyType.RUNTIME;
        const execOpts = this.testScope ? ' -Dexec.classpathScope="test"' : "";
        const compileGoal = this.testScope
            ? "compiler:testCompile"
            : "compiler:compile";
        project.deps.addDependency(`io.github.cdklabs/projen@${projenVersion}`, depType);
        pom.addPlugin("org.codehaus.mojo/exec-maven-plugin@3.0.0");
        // set up the "default" task which is the task executed when `projen` is executed for this project.
        project.defaultTask?.exec(`mvn ${compileGoal} --quiet`);
        project.defaultTask?.exec(`mvn exec:java --quiet -Dexec.mainClass=${this.className}${execOpts}`);
        // if this is a new project, generate a skeleton for projenrc.java
        this.generateProjenrc();
    }
    get javaClass() {
        const split = this.className.split(".");
        if (split.length === 1) {
            return split[0];
        }
        return split[split.length - 1];
    }
    get javaPackage() {
        const split = this.className.split(".");
        if (split.length === 1) {
            return [];
        }
        return split.slice(0, split.length - 2);
    }
    /**
     * The path of the projenrc file.
     */
    get filePath() {
        const dir = this.testScope ? "src/test/java" : "src/main/java";
        const javaFile = (0, path_1.join)(this.project.outdir, dir, ...this.javaPackage, this.javaClass + ".java");
        return (0, path_1.relative)(this.project.outdir, javaFile);
    }
    generateProjenrc() {
        const bootstrap = this.project.initProject;
        if (!bootstrap) {
            return;
        }
        const jsiiFqn = bootstrap.fqn;
        const jsiiManifest = (0, inventory_1.readJsiiManifest)(jsiiFqn);
        const jsiiType = jsiiManifest.types[jsiiFqn];
        const javaTarget = jsiiManifest.targets.java;
        const optionsTypeFqn = jsiiType.initializer?.parameters?.[0].type?.fqn;
        if (!optionsTypeFqn) {
            this.project.logger.warn("cannot determine jsii type for project options");
            return;
        }
        const jsiiOptionsType = jsiiManifest.types[optionsTypeFqn];
        if (!jsiiOptionsType) {
            this.project.logger.warn(`cannot find jsii type for project options: ${optionsTypeFqn}`);
            return;
        }
        // skip if file exists
        if ((0, fs_1.existsSync)(this.filePath)) {
            return;
        }
        const lines = new Array();
        let indent = 0;
        const emit = (line = "") => lines.push(" ".repeat(indent * 4) + line);
        const openBlock = (line = "") => {
            emit(line + " {");
            indent++;
        };
        const closeBlock = () => {
            indent--;
            emit("}");
        };
        const optionFqns = generateJavaOptionNames(bootstrap.type.options, jsiiManifest);
        if (this.javaPackage.length > 0) {
            emit(`package ${this.javaPackage.join(".")};`);
            emit();
        }
        const { renderedOptions, imports } = renderJavaOptions(2, jsiiOptionsType.name, optionFqns, bootstrap.args);
        emit(`import ${getJavaImport(jsiiType, jsiiManifest)};`);
        emit(`import ${getJavaImport(jsiiOptionsType, jsiiManifest)};`);
        for (const optionTypeName of imports) {
            emit(`import ${javaTarget.package}.${optionTypeName};`);
        }
        emit();
        openBlock(`public class ${this.javaClass}`);
        openBlock("public static void main(String[] args)");
        emit(`${jsiiType.name} project = new ${jsiiType.name}(${renderedOptions});`);
        emit("project.synth();");
        closeBlock();
        closeBlock();
        const filePath = (0, path_1.join)(this.project.outdir, this.filePath);
        (0, fs_1.mkdirSync)((0, path_1.dirname)(filePath), { recursive: true });
        (0, fs_1.writeFileSync)(filePath, lines.join("\n"));
        this.project.logger.info(`Project definition file was created at ${this.filePath}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.java.Projenrc", version: "0.79.27" };
function generateJavaOptionNames(options, jsiiManifest) {
    const optionFqns = {};
    for (const option of options) {
        if (option.fqn && jsiiManifest.types[option.fqn]) {
            optionFqns[option.name] = toJavaFullTypeName(jsiiManifest.types[option.fqn]);
        }
    }
    return optionFqns;
}
exports.generateJavaOptionNames = generateJavaOptionNames;
function renderJavaOptions(indent, optionsTypeName, optionFqns, initOptions) {
    const imports = new Set();
    if (!initOptions || Object.keys(initOptions).length === 0) {
        return { renderedOptions: "", imports }; // no options
    }
    const lines = [`${optionsTypeName}.builder()`];
    for (const [name, value] of Object.entries(initOptions)) {
        const { javaValue, importName } = toJavaValue(value, name, optionFqns);
        if (importName)
            imports.add(importName);
        lines.push(`.${toJavaProperty(name)}(${javaValue})`);
    }
    lines.push(".build()");
    const renderedOptions = lines.join(`\n${" ".repeat((indent + 1) * 4)}`);
    return { renderedOptions, imports };
}
function toJavaProperty(prop) {
    return prop;
}
function toJavaValue(value, name, optionFqns) {
    if (typeof value === "string" && optionFqns[name] !== undefined) {
        const parts = optionFqns[name].split(".");
        const base = parts[parts.length - 1];
        const choice = String(value).toUpperCase().replace(/-/g, "_");
        return { javaValue: `${base}.${choice}`, importName: optionFqns[name] };
    }
    else {
        return { javaValue: JSON.stringify(value) };
    }
}
function toJavaFullTypeName(jsiiType) {
    return [jsiiType.namespace, jsiiType.name].filter((x) => x).join(".");
}
function getJavaImport(jsiiType, jsiiManifest) {
    const packageName = jsiiManifest?.submodules?.[`${jsiiType.assembly}.${jsiiType?.namespace}`]
        ?.targets?.java?.package ||
        [jsiiManifest.targets.java.package, jsiiType.namespace]
            .filter((x) => x)
            .join(".");
    return `${packageName}.${jsiiType.name}`;
}
exports.getJavaImport = getJavaImport;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJvamVucmMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvamF2YS9wcm9qZW5yYy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDJCQUEwRDtBQUMxRCwrQkFBK0M7QUFFL0Msc0NBQTJDO0FBQzNDLGtEQUFpRDtBQUNqRCw0Q0FBK0Q7QUFFL0QsMENBQTJDO0FBaUMzQzs7Ozs7O0dBTUc7QUFDSCxNQUFhLFFBQVMsU0FBUSx1QkFBWTtJQVd4QyxZQUFZLE9BQWdCLEVBQUUsR0FBUSxFQUFFLFVBQTJCLEVBQUU7UUFDbkUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRWYsTUFBTSxhQUFhLEdBQUcsT0FBTyxDQUFDLGFBQWEsSUFBSSx1QkFBYyxDQUFDO1FBQzlELElBQUksQ0FBQyxTQUFTLEdBQUcsT0FBTyxDQUFDLFNBQVMsSUFBSSxVQUFVLENBQUM7UUFDakQsSUFBSSxDQUFDLFNBQVMsR0FBRyxPQUFPLENBQUMsU0FBUyxJQUFJLElBQUksQ0FBQztRQUUzQyxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsU0FBUztZQUM1QixDQUFDLENBQUMsNkJBQWMsQ0FBQyxJQUFJO1lBQ3JCLENBQUMsQ0FBQyw2QkFBYyxDQUFDLE9BQU8sQ0FBQztRQUMzQixNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBQ3ZFLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxTQUFTO1lBQ2hDLENBQUMsQ0FBQyxzQkFBc0I7WUFDeEIsQ0FBQyxDQUFDLGtCQUFrQixDQUFDO1FBRXZCLE9BQU8sQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUN4Qiw0QkFBNEIsYUFBYSxFQUFFLEVBQzNDLE9BQU8sQ0FDUixDQUFDO1FBQ0YsR0FBRyxDQUFDLFNBQVMsQ0FBQywyQ0FBMkMsQ0FBQyxDQUFDO1FBRTNELG1HQUFtRztRQUNuRyxPQUFPLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxPQUFPLFdBQVcsVUFBVSxDQUFDLENBQUM7UUFDeEQsT0FBTyxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQ3ZCLDBDQUEwQyxJQUFJLENBQUMsU0FBUyxHQUFHLFFBQVEsRUFBRSxDQUN0RSxDQUFDO1FBRUYsa0VBQWtFO1FBQ2xFLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDO0lBQzFCLENBQUM7SUFFRCxJQUFZLFNBQVM7UUFDbkIsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDeEMsSUFBSSxLQUFLLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUN0QixPQUFPLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUNqQjtRQUNELE9BQU8sS0FBSyxDQUFDLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVELElBQVksV0FBVztRQUNyQixNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUN4QyxJQUFJLEtBQUssQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3RCLE9BQU8sRUFBRSxDQUFDO1NBQ1g7UUFDRCxPQUFPLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOztPQUVHO0lBQ0gsSUFBVyxRQUFRO1FBQ2pCLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDO1FBRS9ELE1BQU0sUUFBUSxHQUFHLElBQUEsV0FBSSxFQUNuQixJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFDbkIsR0FBRyxFQUNILEdBQUcsSUFBSSxDQUFDLFdBQVcsRUFDbkIsSUFBSSxDQUFDLFNBQVMsR0FBRyxPQUFPLENBQ3pCLENBQUM7UUFFRixPQUFPLElBQUEsZUFBUSxFQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFTyxnQkFBZ0I7UUFDdEIsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUM7UUFDM0MsSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNkLE9BQU87U0FDUjtRQUNELE1BQU0sT0FBTyxHQUFHLFNBQVMsQ0FBQyxHQUFHLENBQUM7UUFDOUIsTUFBTSxZQUFZLEdBQUcsSUFBQSw0QkFBZ0IsRUFBQyxPQUFPLENBQUMsQ0FBQztRQUMvQyxNQUFNLFFBQVEsR0FBRyxZQUFZLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQzdDLE1BQU0sVUFBVSxHQUFHLFlBQVksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDO1FBQzdDLE1BQU0sY0FBYyxHQUFHLFFBQVEsQ0FBQyxXQUFXLEVBQUUsVUFBVSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQztRQUN2RSxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQ25CLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLElBQUksQ0FDdEIsZ0RBQWdELENBQ2pELENBQUM7WUFDRixPQUFPO1NBQ1I7UUFDRCxNQUFNLGVBQWUsR0FBRyxZQUFZLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQzNELElBQUksQ0FBQyxlQUFlLEVBQUU7WUFDcEIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUN0Qiw4Q0FBOEMsY0FBYyxFQUFFLENBQy9ELENBQUM7WUFDRixPQUFPO1NBQ1I7UUFDRCxzQkFBc0I7UUFDdEIsSUFBSSxJQUFBLGVBQVUsRUFBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUU7WUFDN0IsT0FBTztTQUNSO1FBRUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxLQUFLLEVBQVUsQ0FBQztRQUVsQyxJQUFJLE1BQU0sR0FBRyxDQUFDLENBQUM7UUFDZixNQUFNLElBQUksR0FBRyxDQUFDLE9BQWUsRUFBRSxFQUFFLEVBQUUsQ0FDakMsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsR0FBRyxJQUFJLENBQUMsQ0FBQztRQUM1QyxNQUFNLFNBQVMsR0FBRyxDQUFDLE9BQWUsRUFBRSxFQUFFLEVBQUU7WUFDdEMsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUMsQ0FBQztZQUNsQixNQUFNLEVBQUUsQ0FBQztRQUNYLENBQUMsQ0FBQztRQUNGLE1BQU0sVUFBVSxHQUFHLEdBQUcsRUFBRTtZQUN0QixNQUFNLEVBQUUsQ0FBQztZQUNULElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNaLENBQUMsQ0FBQztRQUVGLE1BQU0sVUFBVSxHQUEyQix1QkFBdUIsQ0FDaEUsU0FBUyxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQ3RCLFlBQVksQ0FDYixDQUFDO1FBRUYsSUFBSSxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDL0IsSUFBSSxDQUFDLFdBQVcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQy9DLElBQUksRUFBRSxDQUFDO1NBQ1I7UUFFRCxNQUFNLEVBQUUsZUFBZSxFQUFFLE9BQU8sRUFBRSxHQUFHLGlCQUFpQixDQUNwRCxDQUFDLEVBQ0QsZUFBZSxDQUFDLElBQUksRUFDcEIsVUFBVSxFQUNWLFNBQVMsQ0FBQyxJQUFJLENBQ2YsQ0FBQztRQUVGLElBQUksQ0FBQyxVQUFVLGFBQWEsQ0FBQyxRQUFRLEVBQUUsWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3pELElBQUksQ0FBQyxVQUFVLGFBQWEsQ0FBQyxlQUFlLEVBQUUsWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2hFLEtBQUssTUFBTSxjQUFjLElBQUksT0FBTyxFQUFFO1lBQ3BDLElBQUksQ0FBQyxVQUFVLFVBQVUsQ0FBQyxPQUFPLElBQUksY0FBYyxHQUFHLENBQUMsQ0FBQztTQUN6RDtRQUNELElBQUksRUFBRSxDQUFDO1FBQ1AsU0FBUyxDQUFDLGdCQUFnQixJQUFJLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQztRQUM1QyxTQUFTLENBQUMsd0NBQXdDLENBQUMsQ0FBQztRQUNwRCxJQUFJLENBQ0YsR0FBRyxRQUFRLENBQUMsSUFBSSxrQkFBa0IsUUFBUSxDQUFDLElBQUksSUFBSSxlQUFlLElBQUksQ0FDdkUsQ0FBQztRQUNGLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQ3pCLFVBQVUsRUFBRSxDQUFDO1FBQ2IsVUFBVSxFQUFFLENBQUM7UUFFYixNQUFNLFFBQVEsR0FBRyxJQUFBLFdBQUksRUFBQyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDMUQsSUFBQSxjQUFTLEVBQUMsSUFBQSxjQUFPLEVBQUMsUUFBUSxDQUFDLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUNsRCxJQUFBLGtCQUFhLEVBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUUxQyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQ3RCLDBDQUEwQyxJQUFJLENBQUMsUUFBUSxFQUFFLENBQzFELENBQUM7SUFDSixDQUFDOztBQTNKSCw0QkE0SkM7OztBQUVELFNBQWdCLHVCQUF1QixDQUNyQyxPQUF3QixFQUN4QixZQUFpQjtJQUVqQixNQUFNLFVBQVUsR0FBMkIsRUFBRSxDQUFDO0lBQzlDLEtBQUssTUFBTSxNQUFNLElBQUksT0FBTyxFQUFFO1FBQzVCLElBQUksTUFBTSxDQUFDLEdBQUcsSUFBSSxZQUFZLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUNoRCxVQUFVLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLGtCQUFrQixDQUMxQyxZQUFZLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FDL0IsQ0FBQztTQUNIO0tBQ0Y7SUFFRCxPQUFPLFVBQVUsQ0FBQztBQUNwQixDQUFDO0FBZEQsMERBY0M7QUFFRCxTQUFTLGlCQUFpQixDQUN4QixNQUFjLEVBQ2QsZUFBdUIsRUFDdkIsVUFBa0MsRUFDbEMsV0FBaUM7SUFFakMsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLEVBQVUsQ0FBQztJQUNsQyxJQUFJLENBQUMsV0FBVyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtRQUN6RCxPQUFPLEVBQUUsZUFBZSxFQUFFLEVBQUUsRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDLGFBQWE7S0FDdkQ7SUFFRCxNQUFNLEtBQUssR0FBRyxDQUFDLEdBQUcsZUFBZSxZQUFZLENBQUMsQ0FBQztJQUUvQyxLQUFLLE1BQU0sQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsRUFBRTtRQUN2RCxNQUFNLEVBQUUsU0FBUyxFQUFFLFVBQVUsRUFBRSxHQUFHLFdBQVcsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQ3ZFLElBQUksVUFBVTtZQUFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDeEMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLGNBQWMsQ0FBQyxJQUFJLENBQUMsSUFBSSxTQUFTLEdBQUcsQ0FBQyxDQUFDO0tBQ3REO0lBRUQsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUV2QixNQUFNLGVBQWUsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLEtBQUssR0FBRyxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDeEUsT0FBTyxFQUFFLGVBQWUsRUFBRSxPQUFPLEVBQUUsQ0FBQztBQUN0QyxDQUFDO0FBRUQsU0FBUyxjQUFjLENBQUMsSUFBWTtJQUNsQyxPQUFPLElBQUksQ0FBQztBQUNkLENBQUM7QUFFRCxTQUFTLFdBQVcsQ0FDbEIsS0FBVSxFQUNWLElBQVksRUFDWixVQUFrQztJQUVsQyxJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsSUFBSSxVQUFVLENBQUMsSUFBSSxDQUFDLEtBQUssU0FBUyxFQUFFO1FBQy9ELE1BQU0sS0FBSyxHQUFHLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDMUMsTUFBTSxJQUFJLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDckMsTUFBTSxNQUFNLEdBQUcsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDOUQsT0FBTyxFQUFFLFNBQVMsRUFBRSxHQUFHLElBQUksSUFBSSxNQUFNLEVBQUUsRUFBRSxVQUFVLEVBQUUsVUFBVSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUM7S0FDekU7U0FBTTtRQUNMLE9BQU8sRUFBRSxTQUFTLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDO0tBQzdDO0FBQ0gsQ0FBQztBQUVELFNBQVMsa0JBQWtCLENBQUMsUUFBYTtJQUN2QyxPQUFPLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7QUFDeEUsQ0FBQztBQUVELFNBQWdCLGFBQWEsQ0FBQyxRQUFhLEVBQUUsWUFBaUI7SUFDNUQsTUFBTSxXQUFXLEdBQ2YsWUFBWSxFQUFFLFVBQVUsRUFBRSxDQUFDLEdBQUcsUUFBUSxDQUFDLFFBQVEsSUFBSSxRQUFRLEVBQUUsU0FBUyxFQUFFLENBQUM7UUFDdkUsRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLE9BQU87UUFDMUIsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsUUFBUSxDQUFDLFNBQVMsQ0FBQzthQUNwRCxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQzthQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7SUFFZixPQUFPLEdBQUcsV0FBVyxJQUFJLFFBQVEsQ0FBQyxJQUFJLEVBQUUsQ0FBQztBQUMzQyxDQUFDO0FBVEQsc0NBU0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBleGlzdHNTeW5jLCBta2RpclN5bmMsIHdyaXRlRmlsZVN5bmMgfSBmcm9tIFwiZnNcIjtcbmltcG9ydCB7IGRpcm5hbWUsIGpvaW4sIHJlbGF0aXZlIH0gZnJvbSBcInBhdGhcIjtcbmltcG9ydCB7IFBvbSB9IGZyb20gXCIuL3BvbVwiO1xuaW1wb3J0IHsgUFJPSkVOX1ZFUlNJT04gfSBmcm9tIFwiLi4vY29tbW9uXCI7XG5pbXBvcnQgeyBEZXBlbmRlbmN5VHlwZSB9IGZyb20gXCIuLi9kZXBlbmRlbmNpZXNcIjtcbmltcG9ydCB7IFByb2plY3RPcHRpb24sIHJlYWRKc2lpTWFuaWZlc3QgfSBmcm9tIFwiLi4vaW52ZW50b3J5XCI7XG5pbXBvcnQgeyBQcm9qZWN0IH0gZnJvbSBcIi4uL3Byb2plY3RcIjtcbmltcG9ydCB7IFByb2plbnJjRmlsZSB9IGZyb20gXCIuLi9wcm9qZW5yY1wiO1xuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGBQcm9qZW5yY2AuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUHJvamVucmNPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBKYXZhIGNsYXNzIHdoaWNoIGNvbnRhaW5zIHRoZSBgbWFpbigpYCBtZXRob2QgZm9yIHByb2plbi5cbiAgICogQGRlZmF1bHQgXCJwcm9qZW5yY1wiXG4gICAqL1xuICByZWFkb25seSBjbGFzc05hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBwcm9qZW4gdmVyc2lvbiB0byB1c2VcbiAgICogQGRlZmF1bHQgLSBjdXJyZW50IHZlcnNpb25cbiAgICovXG4gIHJlYWRvbmx5IHByb2plblZlcnNpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIERlZmluZXMgcHJvamVucmMgdW5kZXIgdGhlIHRlc3Qgc2NvcGUgaW5zdGVhZCBvZiB0aGUgbWFpbiBzY29wZSwgd2hpY2ggaXNcbiAgICogcmVzZXJ2ZWQgdG8gdGhlIGFwcC4gVGhpcyBtZWFucyB0aGF0IHByb2plbnJjIHdpbGwgYmUgdW5kZXJcbiAgICogYHNyYy90ZXN0L2phdmEvcHJvamVucmMuamF2YWAgYW5kIHByb2plbiB3aWxsIGJlIGRlZmluZWQgYXMgYSB0ZXN0XG4gICAqIGRlcGVuZGVuY3kuIFRoaXMgZW5mb3JjZXMgdGhhdCBhcHBsaWNhdGlvbiBjb2RlIGRvZXMgbm90IHRha2UgYSBkZXBlbmRlbmN5XG4gICAqIG9uIHByb2plbiBjb2RlLlxuICAgKlxuICAgKiBJZiB0aGlzIGlzIGRpc2FibGVkLCBwcm9qZW5yYyBzaG91bGQgYmUgdW5kZXJcbiAgICogYHNyYy9tYWluL2phdmEvcHJvamVucmMuamF2YWAuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHRlc3RTY29wZT86IGJvb2xlYW47XG59XG5cbi8qKlxuICogQWxsb3dzIHdyaXRpbmcgcHJvamVucmMgZmlsZXMgaW4gamF2YS5cbiAqXG4gKiBUaGlzIHdpbGwgaW5zdGFsbCBgb3JnLnByb2plbi9wcm9qZW5gIGFzIGEgTWF2ZW4gZGVwZW5kZW5jeSBhbmQgd2lsbCBhZGQgYVxuICogYHN5bnRoYCB0YXNrIHdoaWNoIHdpbGwgY29tcGlsZSAmIGV4ZWN1dGUgYG1haW4oKWAgZnJvbVxuICogYHNyYy9tYWluL2phdmEvcHJvamVucmMuamF2YWAuXG4gKi9cbmV4cG9ydCBjbGFzcyBQcm9qZW5yYyBleHRlbmRzIFByb2plbnJjRmlsZSB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgamF2YSBjbGFzcyB0aGF0IGluY2x1ZGVzIHRoZSBwcm9qZW4gZW50cnlwb2ludC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjbGFzc05hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogV2hldGhlciB0aGUgY2xhc3MgaXMgaW4gdGhlIFwidGVzdFwiIG9yIFwibWFpblwiIHNjb3BlP1xuICAgKi9cbiAgcHJpdmF0ZSByZWFkb25seSB0ZXN0U2NvcGU6IGJvb2xlYW47XG5cbiAgY29uc3RydWN0b3IocHJvamVjdDogUHJvamVjdCwgcG9tOiBQb20sIG9wdGlvbnM6IFByb2plbnJjT3B0aW9ucyA9IHt9KSB7XG4gICAgc3VwZXIocHJvamVjdCk7XG5cbiAgICBjb25zdCBwcm9qZW5WZXJzaW9uID0gb3B0aW9ucy5wcm9qZW5WZXJzaW9uID8/IFBST0pFTl9WRVJTSU9OO1xuICAgIHRoaXMuY2xhc3NOYW1lID0gb3B0aW9ucy5jbGFzc05hbWUgPz8gXCJwcm9qZW5yY1wiO1xuICAgIHRoaXMudGVzdFNjb3BlID0gb3B0aW9ucy50ZXN0U2NvcGUgPz8gdHJ1ZTtcblxuICAgIGNvbnN0IGRlcFR5cGUgPSB0aGlzLnRlc3RTY29wZVxuICAgICAgPyBEZXBlbmRlbmN5VHlwZS5URVNUXG4gICAgICA6IERlcGVuZGVuY3lUeXBlLlJVTlRJTUU7XG4gICAgY29uc3QgZXhlY09wdHMgPSB0aGlzLnRlc3RTY29wZSA/ICcgLURleGVjLmNsYXNzcGF0aFNjb3BlPVwidGVzdFwiJyA6IFwiXCI7XG4gICAgY29uc3QgY29tcGlsZUdvYWwgPSB0aGlzLnRlc3RTY29wZVxuICAgICAgPyBcImNvbXBpbGVyOnRlc3RDb21waWxlXCJcbiAgICAgIDogXCJjb21waWxlcjpjb21waWxlXCI7XG5cbiAgICBwcm9qZWN0LmRlcHMuYWRkRGVwZW5kZW5jeShcbiAgICAgIGBpby5naXRodWIuY2RrbGFicy9wcm9qZW5AJHtwcm9qZW5WZXJzaW9ufWAsXG4gICAgICBkZXBUeXBlXG4gICAgKTtcbiAgICBwb20uYWRkUGx1Z2luKFwib3JnLmNvZGVoYXVzLm1vam8vZXhlYy1tYXZlbi1wbHVnaW5AMy4wLjBcIik7XG5cbiAgICAvLyBzZXQgdXAgdGhlIFwiZGVmYXVsdFwiIHRhc2sgd2hpY2ggaXMgdGhlIHRhc2sgZXhlY3V0ZWQgd2hlbiBgcHJvamVuYCBpcyBleGVjdXRlZCBmb3IgdGhpcyBwcm9qZWN0LlxuICAgIHByb2plY3QuZGVmYXVsdFRhc2s/LmV4ZWMoYG12biAke2NvbXBpbGVHb2FsfSAtLXF1aWV0YCk7XG4gICAgcHJvamVjdC5kZWZhdWx0VGFzaz8uZXhlYyhcbiAgICAgIGBtdm4gZXhlYzpqYXZhIC0tcXVpZXQgLURleGVjLm1haW5DbGFzcz0ke3RoaXMuY2xhc3NOYW1lfSR7ZXhlY09wdHN9YFxuICAgICk7XG5cbiAgICAvLyBpZiB0aGlzIGlzIGEgbmV3IHByb2plY3QsIGdlbmVyYXRlIGEgc2tlbGV0b24gZm9yIHByb2plbnJjLmphdmFcbiAgICB0aGlzLmdlbmVyYXRlUHJvamVucmMoKTtcbiAgfVxuXG4gIHByaXZhdGUgZ2V0IGphdmFDbGFzcygpOiBzdHJpbmcge1xuICAgIGNvbnN0IHNwbGl0ID0gdGhpcy5jbGFzc05hbWUuc3BsaXQoXCIuXCIpO1xuICAgIGlmIChzcGxpdC5sZW5ndGggPT09IDEpIHtcbiAgICAgIHJldHVybiBzcGxpdFswXTtcbiAgICB9XG4gICAgcmV0dXJuIHNwbGl0W3NwbGl0Lmxlbmd0aCAtIDFdO1xuICB9XG5cbiAgcHJpdmF0ZSBnZXQgamF2YVBhY2thZ2UoKTogc3RyaW5nW10ge1xuICAgIGNvbnN0IHNwbGl0ID0gdGhpcy5jbGFzc05hbWUuc3BsaXQoXCIuXCIpO1xuICAgIGlmIChzcGxpdC5sZW5ndGggPT09IDEpIHtcbiAgICAgIHJldHVybiBbXTtcbiAgICB9XG4gICAgcmV0dXJuIHNwbGl0LnNsaWNlKDAsIHNwbGl0Lmxlbmd0aCAtIDIpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBwYXRoIG9mIHRoZSBwcm9qZW5yYyBmaWxlLlxuICAgKi9cbiAgcHVibGljIGdldCBmaWxlUGF0aCgpOiBzdHJpbmcge1xuICAgIGNvbnN0IGRpciA9IHRoaXMudGVzdFNjb3BlID8gXCJzcmMvdGVzdC9qYXZhXCIgOiBcInNyYy9tYWluL2phdmFcIjtcblxuICAgIGNvbnN0IGphdmFGaWxlID0gam9pbihcbiAgICAgIHRoaXMucHJvamVjdC5vdXRkaXIsXG4gICAgICBkaXIsXG4gICAgICAuLi50aGlzLmphdmFQYWNrYWdlLFxuICAgICAgdGhpcy5qYXZhQ2xhc3MgKyBcIi5qYXZhXCJcbiAgICApO1xuXG4gICAgcmV0dXJuIHJlbGF0aXZlKHRoaXMucHJvamVjdC5vdXRkaXIsIGphdmFGaWxlKTtcbiAgfVxuXG4gIHByaXZhdGUgZ2VuZXJhdGVQcm9qZW5yYygpIHtcbiAgICBjb25zdCBib290c3RyYXAgPSB0aGlzLnByb2plY3QuaW5pdFByb2plY3Q7XG4gICAgaWYgKCFib290c3RyYXApIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgY29uc3QganNpaUZxbiA9IGJvb3RzdHJhcC5mcW47XG4gICAgY29uc3QganNpaU1hbmlmZXN0ID0gcmVhZEpzaWlNYW5pZmVzdChqc2lpRnFuKTtcbiAgICBjb25zdCBqc2lpVHlwZSA9IGpzaWlNYW5pZmVzdC50eXBlc1tqc2lpRnFuXTtcbiAgICBjb25zdCBqYXZhVGFyZ2V0ID0ganNpaU1hbmlmZXN0LnRhcmdldHMuamF2YTtcbiAgICBjb25zdCBvcHRpb25zVHlwZUZxbiA9IGpzaWlUeXBlLmluaXRpYWxpemVyPy5wYXJhbWV0ZXJzPy5bMF0udHlwZT8uZnFuO1xuICAgIGlmICghb3B0aW9uc1R5cGVGcW4pIHtcbiAgICAgIHRoaXMucHJvamVjdC5sb2dnZXIud2FybihcbiAgICAgICAgXCJjYW5ub3QgZGV0ZXJtaW5lIGpzaWkgdHlwZSBmb3IgcHJvamVjdCBvcHRpb25zXCJcbiAgICAgICk7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIGNvbnN0IGpzaWlPcHRpb25zVHlwZSA9IGpzaWlNYW5pZmVzdC50eXBlc1tvcHRpb25zVHlwZUZxbl07XG4gICAgaWYgKCFqc2lpT3B0aW9uc1R5cGUpIHtcbiAgICAgIHRoaXMucHJvamVjdC5sb2dnZXIud2FybihcbiAgICAgICAgYGNhbm5vdCBmaW5kIGpzaWkgdHlwZSBmb3IgcHJvamVjdCBvcHRpb25zOiAke29wdGlvbnNUeXBlRnFufWBcbiAgICAgICk7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIC8vIHNraXAgaWYgZmlsZSBleGlzdHNcbiAgICBpZiAoZXhpc3RzU3luYyh0aGlzLmZpbGVQYXRoKSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGNvbnN0IGxpbmVzID0gbmV3IEFycmF5PHN0cmluZz4oKTtcblxuICAgIGxldCBpbmRlbnQgPSAwO1xuICAgIGNvbnN0IGVtaXQgPSAobGluZTogc3RyaW5nID0gXCJcIikgPT5cbiAgICAgIGxpbmVzLnB1c2goXCIgXCIucmVwZWF0KGluZGVudCAqIDQpICsgbGluZSk7XG4gICAgY29uc3Qgb3BlbkJsb2NrID0gKGxpbmU6IHN0cmluZyA9IFwiXCIpID0+IHtcbiAgICAgIGVtaXQobGluZSArIFwiIHtcIik7XG4gICAgICBpbmRlbnQrKztcbiAgICB9O1xuICAgIGNvbnN0IGNsb3NlQmxvY2sgPSAoKSA9PiB7XG4gICAgICBpbmRlbnQtLTtcbiAgICAgIGVtaXQoXCJ9XCIpO1xuICAgIH07XG5cbiAgICBjb25zdCBvcHRpb25GcW5zOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+ID0gZ2VuZXJhdGVKYXZhT3B0aW9uTmFtZXMoXG4gICAgICBib290c3RyYXAudHlwZS5vcHRpb25zLFxuICAgICAganNpaU1hbmlmZXN0XG4gICAgKTtcblxuICAgIGlmICh0aGlzLmphdmFQYWNrYWdlLmxlbmd0aCA+IDApIHtcbiAgICAgIGVtaXQoYHBhY2thZ2UgJHt0aGlzLmphdmFQYWNrYWdlLmpvaW4oXCIuXCIpfTtgKTtcbiAgICAgIGVtaXQoKTtcbiAgICB9XG5cbiAgICBjb25zdCB7IHJlbmRlcmVkT3B0aW9ucywgaW1wb3J0cyB9ID0gcmVuZGVySmF2YU9wdGlvbnMoXG4gICAgICAyLFxuICAgICAganNpaU9wdGlvbnNUeXBlLm5hbWUsXG4gICAgICBvcHRpb25GcW5zLFxuICAgICAgYm9vdHN0cmFwLmFyZ3NcbiAgICApO1xuXG4gICAgZW1pdChgaW1wb3J0ICR7Z2V0SmF2YUltcG9ydChqc2lpVHlwZSwganNpaU1hbmlmZXN0KX07YCk7XG4gICAgZW1pdChgaW1wb3J0ICR7Z2V0SmF2YUltcG9ydChqc2lpT3B0aW9uc1R5cGUsIGpzaWlNYW5pZmVzdCl9O2ApO1xuICAgIGZvciAoY29uc3Qgb3B0aW9uVHlwZU5hbWUgb2YgaW1wb3J0cykge1xuICAgICAgZW1pdChgaW1wb3J0ICR7amF2YVRhcmdldC5wYWNrYWdlfS4ke29wdGlvblR5cGVOYW1lfTtgKTtcbiAgICB9XG4gICAgZW1pdCgpO1xuICAgIG9wZW5CbG9jayhgcHVibGljIGNsYXNzICR7dGhpcy5qYXZhQ2xhc3N9YCk7XG4gICAgb3BlbkJsb2NrKFwicHVibGljIHN0YXRpYyB2b2lkIG1haW4oU3RyaW5nW10gYXJncylcIik7XG4gICAgZW1pdChcbiAgICAgIGAke2pzaWlUeXBlLm5hbWV9IHByb2plY3QgPSBuZXcgJHtqc2lpVHlwZS5uYW1lfSgke3JlbmRlcmVkT3B0aW9uc30pO2BcbiAgICApO1xuICAgIGVtaXQoXCJwcm9qZWN0LnN5bnRoKCk7XCIpO1xuICAgIGNsb3NlQmxvY2soKTtcbiAgICBjbG9zZUJsb2NrKCk7XG5cbiAgICBjb25zdCBmaWxlUGF0aCA9IGpvaW4odGhpcy5wcm9qZWN0Lm91dGRpciwgdGhpcy5maWxlUGF0aCk7XG4gICAgbWtkaXJTeW5jKGRpcm5hbWUoZmlsZVBhdGgpLCB7IHJlY3Vyc2l2ZTogdHJ1ZSB9KTtcbiAgICB3cml0ZUZpbGVTeW5jKGZpbGVQYXRoLCBsaW5lcy5qb2luKFwiXFxuXCIpKTtcblxuICAgIHRoaXMucHJvamVjdC5sb2dnZXIuaW5mbyhcbiAgICAgIGBQcm9qZWN0IGRlZmluaXRpb24gZmlsZSB3YXMgY3JlYXRlZCBhdCAke3RoaXMuZmlsZVBhdGh9YFxuICAgICk7XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGdlbmVyYXRlSmF2YU9wdGlvbk5hbWVzKFxuICBvcHRpb25zOiBQcm9qZWN0T3B0aW9uW10sXG4gIGpzaWlNYW5pZmVzdDogYW55XG4pIHtcbiAgY29uc3Qgb3B0aW9uRnFuczogUmVjb3JkPHN0cmluZywgc3RyaW5nPiA9IHt9O1xuICBmb3IgKGNvbnN0IG9wdGlvbiBvZiBvcHRpb25zKSB7XG4gICAgaWYgKG9wdGlvbi5mcW4gJiYganNpaU1hbmlmZXN0LnR5cGVzW29wdGlvbi5mcW5dKSB7XG4gICAgICBvcHRpb25GcW5zW29wdGlvbi5uYW1lXSA9IHRvSmF2YUZ1bGxUeXBlTmFtZShcbiAgICAgICAganNpaU1hbmlmZXN0LnR5cGVzW29wdGlvbi5mcW5dXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIHJldHVybiBvcHRpb25GcW5zO1xufVxuXG5mdW5jdGlvbiByZW5kZXJKYXZhT3B0aW9ucyhcbiAgaW5kZW50OiBudW1iZXIsXG4gIG9wdGlvbnNUeXBlTmFtZTogc3RyaW5nLFxuICBvcHRpb25GcW5zOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+LFxuICBpbml0T3B0aW9ucz86IFJlY29yZDxzdHJpbmcsIGFueT5cbikge1xuICBjb25zdCBpbXBvcnRzID0gbmV3IFNldDxzdHJpbmc+KCk7XG4gIGlmICghaW5pdE9wdGlvbnMgfHwgT2JqZWN0LmtleXMoaW5pdE9wdGlvbnMpLmxlbmd0aCA9PT0gMCkge1xuICAgIHJldHVybiB7IHJlbmRlcmVkT3B0aW9uczogXCJcIiwgaW1wb3J0cyB9OyAvLyBubyBvcHRpb25zXG4gIH1cblxuICBjb25zdCBsaW5lcyA9IFtgJHtvcHRpb25zVHlwZU5hbWV9LmJ1aWxkZXIoKWBdO1xuXG4gIGZvciAoY29uc3QgW25hbWUsIHZhbHVlXSBvZiBPYmplY3QuZW50cmllcyhpbml0T3B0aW9ucykpIHtcbiAgICBjb25zdCB7IGphdmFWYWx1ZSwgaW1wb3J0TmFtZSB9ID0gdG9KYXZhVmFsdWUodmFsdWUsIG5hbWUsIG9wdGlvbkZxbnMpO1xuICAgIGlmIChpbXBvcnROYW1lKSBpbXBvcnRzLmFkZChpbXBvcnROYW1lKTtcbiAgICBsaW5lcy5wdXNoKGAuJHt0b0phdmFQcm9wZXJ0eShuYW1lKX0oJHtqYXZhVmFsdWV9KWApO1xuICB9XG5cbiAgbGluZXMucHVzaChcIi5idWlsZCgpXCIpO1xuXG4gIGNvbnN0IHJlbmRlcmVkT3B0aW9ucyA9IGxpbmVzLmpvaW4oYFxcbiR7XCIgXCIucmVwZWF0KChpbmRlbnQgKyAxKSAqIDQpfWApO1xuICByZXR1cm4geyByZW5kZXJlZE9wdGlvbnMsIGltcG9ydHMgfTtcbn1cblxuZnVuY3Rpb24gdG9KYXZhUHJvcGVydHkocHJvcDogc3RyaW5nKSB7XG4gIHJldHVybiBwcm9wO1xufVxuXG5mdW5jdGlvbiB0b0phdmFWYWx1ZShcbiAgdmFsdWU6IGFueSxcbiAgbmFtZTogc3RyaW5nLFxuICBvcHRpb25GcW5zOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+XG4pIHtcbiAgaWYgKHR5cGVvZiB2YWx1ZSA9PT0gXCJzdHJpbmdcIiAmJiBvcHRpb25GcW5zW25hbWVdICE9PSB1bmRlZmluZWQpIHtcbiAgICBjb25zdCBwYXJ0cyA9IG9wdGlvbkZxbnNbbmFtZV0uc3BsaXQoXCIuXCIpO1xuICAgIGNvbnN0IGJhc2UgPSBwYXJ0c1twYXJ0cy5sZW5ndGggLSAxXTtcbiAgICBjb25zdCBjaG9pY2UgPSBTdHJpbmcodmFsdWUpLnRvVXBwZXJDYXNlKCkucmVwbGFjZSgvLS9nLCBcIl9cIik7XG4gICAgcmV0dXJuIHsgamF2YVZhbHVlOiBgJHtiYXNlfS4ke2Nob2ljZX1gLCBpbXBvcnROYW1lOiBvcHRpb25GcW5zW25hbWVdIH07XG4gIH0gZWxzZSB7XG4gICAgcmV0dXJuIHsgamF2YVZhbHVlOiBKU09OLnN0cmluZ2lmeSh2YWx1ZSkgfTtcbiAgfVxufVxuXG5mdW5jdGlvbiB0b0phdmFGdWxsVHlwZU5hbWUoanNpaVR5cGU6IGFueSkge1xuICByZXR1cm4gW2pzaWlUeXBlLm5hbWVzcGFjZSwganNpaVR5cGUubmFtZV0uZmlsdGVyKCh4KSA9PiB4KS5qb2luKFwiLlwiKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGdldEphdmFJbXBvcnQoanNpaVR5cGU6IGFueSwganNpaU1hbmlmZXN0OiBhbnkpIHtcbiAgY29uc3QgcGFja2FnZU5hbWUgPVxuICAgIGpzaWlNYW5pZmVzdD8uc3VibW9kdWxlcz8uW2Ake2pzaWlUeXBlLmFzc2VtYmx5fS4ke2pzaWlUeXBlPy5uYW1lc3BhY2V9YF1cbiAgICAgID8udGFyZ2V0cz8uamF2YT8ucGFja2FnZSB8fFxuICAgIFtqc2lpTWFuaWZlc3QudGFyZ2V0cy5qYXZhLnBhY2thZ2UsIGpzaWlUeXBlLm5hbWVzcGFjZV1cbiAgICAgIC5maWx0ZXIoKHgpID0+IHgpXG4gICAgICAuam9pbihcIi5cIik7XG5cbiAgcmV0dXJuIGAke3BhY2thZ2VOYW1lfS4ke2pzaWlUeXBlLm5hbWV9YDtcbn1cbiJdfQ==