"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BundleLogLevel = exports.Charset = exports.SourceMapMode = exports.RunBundleTask = exports.Bundler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const path_1 = require("path");
const util_1 = require("./util");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
/**
 * Adds support for bundling JavaScript applications and dependencies into a
 * single file. In the future, this will also supports bundling websites.
 */
class Bundler extends component_1.Component {
    /**
     * Returns the `Bundler` instance associated with a project or `undefined` if
     * there is no Bundler.
     * @param project The project
     * @returns A bundler
     */
    static of(project) {
        const isBundler = (o) => o instanceof Bundler;
        return project.components.find(isBundler);
    }
    /**
     * Creates a `Bundler`.
     */
    constructor(project, options = {}) {
        super(project);
        this.esbuildVersion = options.esbuildVersion;
        this.bundledir = options.assetsDir ?? "assets";
        this.loaders = options.loaders;
        this.runBundleTask =
            options.runBundleTask ??
                (options.addToPreCompile === false
                    ? RunBundleTask.MANUAL
                    : RunBundleTask.PRE_COMPILE);
    }
    /**
     * Gets or creates the singleton "bundle" task of the project.
     *
     * If the project doesn't have a "bundle" task, it will be created and spawned
     * during the pre-compile phase.
     */
    get bundleTask() {
        if (!this._task) {
            this.addBundlingSupport();
            this._task = this.project.tasks.addTask("bundle", {
                description: "Prepare assets",
            });
            // install the bundle task into the pre-compile phase.
            if (this.runBundleTask === RunBundleTask.PRE_COMPILE) {
                this.project.preCompileTask.spawn(this._task);
            }
            else if (this.runBundleTask === RunBundleTask.POST_COMPILE) {
                this.project.postCompileTask.spawn(this._task);
            }
        }
        return this._task;
    }
    /**
     * Adds a task to the project which bundles a specific entrypoint and all of
     * its dependencies into a single javascript output file.
     *
     * @param entrypoint The relative path of the artifact within the project
     * @param options Bundling options
     */
    addBundle(entrypoint, options) {
        const name = (0, util_1.renderBundleName)(entrypoint);
        const outdir = path.posix.join(this.bundledir, name);
        const outfile = path.posix.join(outdir, options.outfile ?? "index.js");
        const args = [
            "esbuild",
            "--bundle",
            entrypoint,
            `--target="${options.target}"`,
            `--platform="${options.platform}"`,
            `--outfile="${outfile}"`,
        ];
        if (options.tsconfigPath) {
            args.push(`--tsconfig="${options.tsconfigPath}"`);
        }
        for (const x of options.externals ?? []) {
            args.push(`--external:${x}`);
        }
        if (options.sourcemap || options.sourceMapMode) {
            const sourceMapMode = options.sourceMapMode ?? SourceMapMode.DEFAULT;
            const sourceMapValue = sourceMapMode === SourceMapMode.DEFAULT
                ? ""
                : `=${options.sourceMapMode}`;
            args.push(`--sourcemap${sourceMapValue}`);
            if (options.sourcesContent === true) {
                args.push(`--sources-content=${options.sourcesContent}`);
            }
        }
        const format = options.format;
        if (format) {
            args.push(`--format=${format}`);
        }
        const loaders = options.loaders ?? false ? options.loaders : this.loaders ?? false;
        if (loaders) {
            for (let [extension, loader] of Object.entries(loaders)) {
                args.push(`--loader:.${extension}=${loader}`);
            }
        }
        const defines = Object.entries(options.define ?? {});
        for (const [key, value] of defines) {
            args.push(`--define:${key}=${JSON.stringify(value)}`);
        }
        if (options.minify) {
            args.push("--minify");
        }
        if (options.logLevel) {
            args.push(`--log-level=${options.logLevel}`);
        }
        if (options.keepNames) {
            args.push("--keep-names");
        }
        if (options.metafile) {
            args.push(`--metafile=${(0, path_1.join)(outdir, "index.meta.json")}`);
        }
        if (options.banner) {
            args.push(`--banner:js=${JSON.stringify(options.banner)}`);
        }
        if (options.footer) {
            args.push(`--footer:js=${JSON.stringify(options.footer)}`);
        }
        if (options.mainFields) {
            args.push(`--main-fields=${options.mainFields.join(",")}`);
        }
        if (options.inject) {
            args.push(...options.inject.map((i) => `--inject:${i}`));
        }
        if (options.esbuildArgs) {
            const subArgs = new Array();
            for (const [key, value] of Object.entries(options.esbuildArgs)) {
                if (value === true || value === "") {
                    subArgs.push(key);
                }
                else if (value) {
                    subArgs.push(`${key}="${value}"`);
                }
            }
            args.push(subArgs.join(" "));
        }
        const bundleTask = this.project.addTask(`bundle:${name}`, {
            description: `Create a JavaScript bundle from ${entrypoint}`,
            exec: args.join(" "),
        });
        this.bundleTask.spawn(bundleTask);
        if (options.executable ?? false) {
            bundleTask.exec(`chmod +x ${outfile}`);
        }
        let watchTask;
        const watch = options.watchTask ?? true;
        if (watch) {
            watchTask = this.project.addTask(`bundle:${name}:watch`, {
                description: `Continuously update the JavaScript bundle from ${entrypoint}`,
                exec: `${args.join(" ")} --watch`,
            });
        }
        return {
            bundleTask: bundleTask,
            watchTask: watchTask,
            outdir: outdir,
            outfile: outfile,
        };
    }
    /**
     * Add bundling support to a project. This is called implicitly when
     * `bundleTask` is referenced first. It adds the dependency on `esbuild`,
     * gitignore/npmignore, etc.
     */
    addBundlingSupport() {
        const ignoreEntry = `/${this.bundledir}/`;
        this.project.addGitIgnore(ignoreEntry);
        this.project.addPackageIgnore(`!${ignoreEntry}`); // include in tarball
        const dep = this.esbuildVersion
            ? `esbuild@${this.esbuildVersion}`
            : "esbuild";
        this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
    }
}
exports.Bundler = Bundler;
_a = JSII_RTTI_SYMBOL_1;
Bundler[_a] = { fqn: "projen.javascript.Bundler", version: "0.79.27" };
/**
 * Options for BundlerOptions.runBundleTask
 */
var RunBundleTask;
(function (RunBundleTask) {
    /**
     * Don't bundle automatically as part of the build.
     */
    RunBundleTask["MANUAL"] = "manual";
    /**
     * Bundle automatically before compilation.
     */
    RunBundleTask["PRE_COMPILE"] = "pre_compile";
    /**
     * Bundle automatically after compilation. This is useful if you want to
     * bundle the compiled results.
     *
     * Thus will run compilation tasks (using tsc, etc.) before running file
     * through bundling step.
     *
     * This is only required unless you are using new experimental features that
     * are not supported by `esbuild` but are supported by typescript's `tsc`
     * compiler. One example of such feature is `emitDecoratorMetadata`.
     *
     * ```typescript
     * // In a TypeScript project with output configured
     * // to go to the "lib" directory:
     * const project = new TypeScriptProject({
     *   name: "test",
     *   defaultReleaseBranch: "main",
     *   tsconfig: {
     *     compilerOptions: {
     *       outDir: "lib",
     *     },
     *   },
     *   bundlerOptions: {
     *     // ensure we compile with `tsc` before bundling
     *     runBundleTask: RunBundleTask.POST_COMPILE,
     *   },
     * });
     *
     * // Tell the bundler to bundle the compiled results (from the "lib" directory)
     * project.bundler.addBundle("./lib/index.js", {
     *   platform: "node",
     *   target: "node18",
     *   sourcemap: false,
     *   format: "esm",
     * });
     * ```
     **/
    RunBundleTask["POST_COMPILE"] = "post_compile";
})(RunBundleTask || (exports.RunBundleTask = RunBundleTask = {}));
/**
 * SourceMap mode for esbuild
 * @see https://esbuild.github.io/api/#sourcemap
 */
var SourceMapMode;
(function (SourceMapMode) {
    /**
     * Default sourceMap mode - will generate a .js.map file alongside any generated .js file and add a special //# sourceMappingURL=
     * comment to the bottom of the .js file pointing to the .js.map file
     */
    SourceMapMode["DEFAULT"] = "default";
    /**
     *  External sourceMap mode - If you want to omit the special //# sourceMappingURL= comment from the generated .js file but you still
     *  want to generate the .js.map files
     */
    SourceMapMode["EXTERNAL"] = "external";
    /**
     * Inline sourceMap mode - If you want to insert the entire source map into the .js file instead of generating a separate .js.map file
     */
    SourceMapMode["INLINE"] = "inline";
    /**
     * Both sourceMap mode - If you want to have the effect of both inline and external simultaneously
     */
    SourceMapMode["BOTH"] = "both";
})(SourceMapMode || (exports.SourceMapMode = SourceMapMode = {}));
/**
 * Charset for esbuild's output
 */
var Charset;
(function (Charset) {
    /**
     * ASCII
     *
     * Any non-ASCII characters are escaped using backslash escape sequences
     */
    Charset["ASCII"] = "ascii";
    /**
     * UTF-8
     *
     * Keep original characters without using escape sequences
     */
    Charset["UTF8"] = "utf8";
})(Charset || (exports.Charset = Charset = {}));
/**
 * Log levels for esbuild and package managers' install commands.
 */
var BundleLogLevel;
(function (BundleLogLevel) {
    /** Show everything */
    BundleLogLevel["VERBOSE"] = "verbose";
    /** Show everything from info and some additional messages for debugging */
    BundleLogLevel["DEBUG"] = "debug";
    /** Show warnings, errors, and an output file summary */
    BundleLogLevel["INFO"] = "info";
    /** Show warnings and errors */
    BundleLogLevel["WARNING"] = "warning";
    /** Show errors only */
    BundleLogLevel["ERROR"] = "error";
    /** Show nothing */
    BundleLogLevel["SILENT"] = "silent";
})(BundleLogLevel || (exports.BundleLogLevel = BundleLogLevel = {}));
//# sourceMappingURL=data:application/json;base64,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