"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Eslint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const prettier_1 = require("./prettier");
const common_1 = require("../common");
const component_1 = require("../component");
const json_1 = require("../json");
const yaml_1 = require("../yaml");
/**
 * Represents eslint configuration.
 */
class Eslint extends component_1.Component {
    /**
     * Returns the singletone Eslint component of a project or undefined if there is none.
     */
    static of(project) {
        const isEslint = (c) => c instanceof Eslint;
        return project.components.find(isEslint);
    }
    constructor(project, options) {
        super(project);
        /**
         * eslint overrides.
         */
        this.overrides = [];
        this._plugins = new Set();
        this._extends = new Set();
        this.nodeProject = project;
        project.addDevDeps("eslint@^8", "@typescript-eslint/eslint-plugin@^6", "@typescript-eslint/parser@^6", "eslint-import-resolver-typescript", "eslint-plugin-import");
        if (options.aliasMap) {
            project.addDevDeps("eslint-import-resolver-alias");
        }
        const lintProjenRc = options.lintProjenRc ?? true;
        const lintProjenRcFile = options.lintProjenRcFile ?? common_1.DEFAULT_PROJEN_RC_JS_FILENAME;
        const devdirs = options.devdirs ?? [];
        this._lintPatterns = new Set([
            ...options.dirs,
            ...devdirs,
            ...(lintProjenRc && lintProjenRcFile ? [lintProjenRcFile] : []),
        ]);
        this._fileExtensions = new Set(options.fileExtensions ?? [".ts"]);
        this._allowDevDeps = new Set((devdirs ?? []).map((dir) => `**/${dir}/**`));
        this.eslintTask = project.addTask("eslint", {
            description: "Runs eslint against the codebase",
        });
        this.updateTask();
        project.testTask.spawn(this.eslintTask);
        // exclude some files
        project.npmignore?.exclude("/.eslintrc.json");
        this._formattingRules = {
            // see https://github.com/typescript-eslint/typescript-eslint/blob/master/packages/eslint-plugin/docs/rules/indent.md
            indent: ["off"],
            "@typescript-eslint/indent": ["error", 2],
            // Style
            quotes: ["error", "single", { avoidEscape: true }],
            "comma-dangle": ["error", "always-multiline"],
            "comma-spacing": ["error", { before: false, after: true }],
            "no-multi-spaces": ["error", { ignoreEOLComments: false }],
            "array-bracket-spacing": ["error", "never"],
            "array-bracket-newline": ["error", "consistent"],
            "object-curly-spacing": ["error", "always"],
            "object-curly-newline": ["error", { multiline: true, consistent: true }],
            "object-property-newline": [
                "error",
                { allowAllPropertiesOnSameLine: true },
            ],
            "keyword-spacing": ["error"],
            "brace-style": ["error", "1tbs", { allowSingleLine: true }],
            "space-before-blocks": ["error"],
            curly: ["error", "multi-line", "consistent"],
            "@typescript-eslint/member-delimiter-style": ["error"],
            // Require semicolons
            semi: ["error", "always"],
            // Max line lengths
            "max-len": [
                "error",
                {
                    code: 150,
                    ignoreUrls: true,
                    ignoreStrings: true,
                    ignoreTemplateLiterals: true,
                    ignoreComments: true,
                    ignoreRegExpLiterals: true,
                },
            ],
            // Don't unnecessarily quote properties
            "quote-props": ["error", "consistent-as-needed"],
        };
        this.rules = {
            // Require use of the `import { foo } from 'bar';` form instead of `import foo = require('bar');`
            "@typescript-eslint/no-require-imports": ["error"],
            // Require all imported dependencies are actually declared in package.json
            "import/no-extraneous-dependencies": [
                "error",
                {
                    // Only allow importing devDependencies from "devdirs".
                    devDependencies: () => this.renderDevDepsAllowList(),
                    optionalDependencies: false,
                    peerDependencies: true, // Allow importing peer dependencies (that aren't also direct dependencies)
                },
            ],
            // Require all imported libraries actually resolve (!!required for import/no-extraneous-dependencies to work!!)
            "import/no-unresolved": ["error"],
            // Require an ordering on all imports
            "import/order": [
                "warn",
                {
                    groups: ["builtin", "external"],
                    alphabetize: { order: "asc", caseInsensitive: true },
                },
            ],
            // Cannot import from the same module twice
            "no-duplicate-imports": ["error"],
            // Cannot shadow names
            "no-shadow": ["off"],
            "@typescript-eslint/no-shadow": ["error"],
            // Required spacing in property declarations (copied from TSLint, defaults are good)
            "key-spacing": ["error"],
            // No multiple empty lines
            "no-multiple-empty-lines": ["error"],
            // One of the easiest mistakes to make
            "@typescript-eslint/no-floating-promises": ["error"],
            // Make sure that inside try/catch blocks, promises are 'return await'ed
            // (must disable the base rule as it can report incorrect errors)
            "no-return-await": ["off"],
            "@typescript-eslint/return-await": ["error"],
            // Useless diff results
            "no-trailing-spaces": ["error"],
            // Must use foo.bar instead of foo['bar'] if possible
            "dot-notation": ["error"],
            // Are you sure | is not a typo for || ?
            "no-bitwise": ["error"],
            // Member ordering
            "@typescript-eslint/member-ordering": [
                "error",
                {
                    default: [
                        "public-static-field",
                        "public-static-method",
                        "protected-static-field",
                        "protected-static-method",
                        "private-static-field",
                        "private-static-method",
                        "field",
                        // Constructors
                        "constructor",
                        // Methods
                        "method",
                    ],
                },
            ],
        };
        // Overrides for .projenrc.js
        // @deprecated
        if (lintProjenRc) {
            this.overrides = [
                {
                    files: [lintProjenRcFile || common_1.DEFAULT_PROJEN_RC_JS_FILENAME],
                    rules: {
                        "@typescript-eslint/no-require-imports": "off",
                        "import/no-extraneous-dependencies": "off",
                    },
                },
            ];
        }
        this.ignorePatterns = options.ignorePatterns ?? [
            "*.js",
            // @deprecated
            ...(lintProjenRc
                ? [`!${lintProjenRcFile || common_1.DEFAULT_PROJEN_RC_JS_FILENAME}`]
                : []),
            "*.d.ts",
            "node_modules/",
            "*.generated.ts",
            "coverage",
        ];
        const tsconfig = options.tsconfigPath ?? "./tsconfig.json";
        this.addPlugins("@typescript-eslint");
        this.addPlugins("import");
        this.addExtends("plugin:import/typescript");
        this.config = {
            env: {
                jest: true,
                node: true,
            },
            root: true,
            plugins: this._plugins,
            parser: "@typescript-eslint/parser",
            parserOptions: {
                ecmaVersion: 2018,
                sourceType: "module",
                project: tsconfig,
            },
            extends: this._extends,
            settings: {
                "import/parsers": {
                    "@typescript-eslint/parser": [".ts", ".tsx"],
                },
                "import/resolver": {
                    ...(options.aliasMap && {
                        alias: {
                            map: Object.entries(options.aliasMap).map(([k, v]) => [k, v]),
                            extensions: options.aliasExtensions,
                        },
                    }),
                    node: {},
                    typescript: {
                        project: tsconfig,
                        ...(options.tsAlwaysTryTypes !== false && { alwaysTryTypes: true }),
                    },
                },
            },
            ignorePatterns: this.ignorePatterns,
            rules: () => ({ ...this._formattingRules, ...this.rules }),
            overrides: this.overrides,
        };
        if (options.yaml) {
            new yaml_1.YamlFile(project, ".eslintrc.yml", {
                obj: this.config,
                marker: true,
            });
        }
        else {
            new json_1.JsonFile(project, ".eslintrc.json", {
                obj: this.config,
                // https://eslint.org/docs/latest/user-guide/configuring/configuration-files#comments-in-configuration-files
                marker: true,
                allowComments: true,
            });
        }
        // if the user enabled prettier explicitly _or_ if the project has a
        // `Prettier` component, we shall tweak our configuration accordingly.
        if (options.prettier || prettier_1.Prettier.of(project)) {
            this.enablePrettier();
        }
    }
    /**
     * Returns an immutable copy of the lintPatterns being used by this eslint configuration.
     */
    get lintPatterns() {
        if (this._lintPatterns && this._lintPatterns.size > 0) {
            return [...this._lintPatterns];
        }
        return [];
    }
    /**
     * Add a file, glob pattern or directory with source files to lint (e.g. [ "src" ])
     */
    addLintPattern(pattern) {
        this._lintPatterns.add(pattern);
        this.updateTask();
    }
    /**
     * Add an eslint rule.
     */
    addRules(rules) {
        for (const [k, v] of Object.entries(rules)) {
            this.rules[k] = v;
        }
    }
    /**
     * Adds an eslint plugin
     * @param plugins The names of plugins to add
     */
    addPlugins(...plugins) {
        for (const plugin of plugins) {
            this._plugins.add(plugin);
        }
    }
    /**
     * Add an eslint override.
     */
    addOverride(override) {
        this.overrides.push(override);
    }
    /**
     * Do not lint these files.
     */
    addIgnorePattern(pattern) {
        this.ignorePatterns.push(pattern);
    }
    /**
     * Adds an `extends` item to the eslint configuration.
     * @param extendList The list of "extends" to add.
     */
    addExtends(...extendList) {
        for (const extend of extendList) {
            this._extends.add(extend);
        }
    }
    /**
     * Add a glob file pattern which allows importing dev dependencies.
     * @param pattern glob pattern.
     */
    allowDevDeps(pattern) {
        this._allowDevDeps.add(pattern);
    }
    /**
     * Enables prettier for code formatting.
     */
    enablePrettier() {
        this.nodeProject.addDevDeps("prettier", "eslint-plugin-prettier", "eslint-config-prettier");
        this._formattingRules = {};
        this.addExtends("plugin:prettier/recommended");
    }
    renderDevDepsAllowList() {
        return Array.from(this._allowDevDeps);
    }
    /**
     * Update the task with the current list of lint patterns and file extensions
     */
    updateTask() {
        const taskExecCommand = "eslint";
        const argsSet = new Set();
        if (this._fileExtensions.size > 0) {
            argsSet.add(`--ext ${[...this._fileExtensions].join(",")}`);
        }
        argsSet.add("--fix");
        argsSet.add("--no-error-on-unmatched-pattern");
        argsSet.add("$@"); // External args go here
        for (const pattern of this._lintPatterns) {
            argsSet.add(pattern);
        }
        this.eslintTask.reset([taskExecCommand, ...argsSet].join(" "), this.buildTaskStepOptions(taskExecCommand));
    }
    /**
     * In case of external editing of the eslint task step, we preserve those changes.
     * Otherwise, we return the default task step options.
     *
     * @param taskExecCommand The command that the ESLint tasks executes
     * @returns Either the externally edited, or the default task step options
     */
    buildTaskStepOptions(taskExecCommand) {
        const currentEslintTaskStep = this.eslintTask?.steps?.find((step) => step?.exec?.startsWith?.(taskExecCommand));
        if (currentEslintTaskStep) {
            const { args, condition, cwd, env, name, receiveArgs } = currentEslintTaskStep;
            return {
                args,
                condition,
                cwd,
                env,
                name,
                receiveArgs,
            };
        }
        return {
            receiveArgs: true,
        };
    }
}
exports.Eslint = Eslint;
_a = JSII_RTTI_SYMBOL_1;
Eslint[_a] = { fqn: "projen.javascript.Eslint", version: "0.79.27" };
//# sourceMappingURL=data:application/json;base64,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