import { YarnrcOptions } from "./yarnrc";
import { Component } from "../component";
import { JsonFile } from "../json";
import { Project } from "../project";
import { Task } from "../task";
export interface NodePackageOptions {
    /**
     * The "name" in package.json
     * @default - defaults to project name
     * @featured
     */
    readonly packageName?: string;
    /**
     * The description is just a string that helps people understand the purpose of the package.
     * It can be used when searching for packages in a package manager as well.
     * See https://classic.yarnpkg.com/en/docs/package-json/#toc-description
     * @featured
     */
    readonly description?: string;
    /**
     * Runtime dependencies of this module.
     *
     * The recommendation is to only specify the module name here (e.g.
     * `express`). This will behave similar to `yarn add` or `npm install` in the
     * sense that it will add the module as a dependency to your `package.json`
     * file with the latest version (`^`). You can specify semver requirements in
     * the same syntax passed to `npm i` or `yarn add` (e.g. `express@^2`) and
     * this will be what you `package.json` will eventually include.
     *
     * @example [ 'express', 'lodash', 'foo@^2' ]
     * @default []
     * @featured
     */
    readonly deps?: string[];
    /**
     * Build dependencies for this module. These dependencies will only be
     * available in your build environment but will not be fetched when this
     * module is consumed.
     *
     * The recommendation is to only specify the module name here (e.g.
     * `express`). This will behave similar to `yarn add` or `npm install` in the
     * sense that it will add the module as a dependency to your `package.json`
     * file with the latest version (`^`). You can specify semver requirements in
     * the same syntax passed to `npm i` or `yarn add` (e.g. `express@^2`) and
     * this will be what you `package.json` will eventually include.
     *
     * @example [ 'typescript', '@types/express' ]
     * @default []
     * @featured
     */
    readonly devDeps?: string[];
    /**
     * Peer dependencies for this module. Dependencies listed here are required to
     * be installed (and satisfied) by the _consumer_ of this library. Using peer
     * dependencies allows you to ensure that only a single module of a certain
     * library exists in the `node_modules` tree of your consumers.
     *
     * Note that prior to npm@7, peer dependencies are _not_ automatically
     * installed, which means that adding peer dependencies to a library will be a
     * breaking change for your customers.
     *
     * Unless `peerDependencyOptions.pinnedDevDependency` is disabled (it is
     * enabled by default), projen will automatically add a dev dependency with a
     * pinned version for each peer dependency. This will ensure that you build &
     * test your module against the lowest peer version required.
     *
     * @default []
     */
    readonly peerDeps?: string[];
    /**
     * List of dependencies to bundle into this module. These modules will be
     * added both to the `dependencies` section and `bundledDependencies` section of
     * your `package.json`.
     *
     * The recommendation is to only specify the module name here (e.g.
     * `express`). This will behave similar to `yarn add` or `npm install` in the
     * sense that it will add the module as a dependency to your `package.json`
     * file with the latest version (`^`). You can specify semver requirements in
     * the same syntax passed to `npm i` or `yarn add` (e.g. `express@^2`) and
     * this will be what you `package.json` will eventually include.
     */
    readonly bundledDeps?: string[];
    /**
     * Options for `peerDeps`.
     */
    readonly peerDependencyOptions?: PeerDependencyOptions;
    /**
     * Allow the project to include `peerDependencies` and `bundledDependencies`.
     * This is normally only allowed for libraries. For apps, there's no meaning
     * for specifying these.
     *
     * @default true
     */
    readonly allowLibraryDependencies?: boolean;
    /**
     * Keywords to include in `package.json`.
     */
    readonly keywords?: string[];
    /**
     * Module entrypoint (`main` in `package.json`)
     *
     * Set to an empty string to not include `main` in your package.json
     *
     * @default "lib/index.js"
     */
    readonly entrypoint?: string;
    /**
     * Binary programs vended with your module.
     *
     * You can use this option to add/customize how binaries are represented in
     * your `package.json`, but unless `autoDetectBin` is `false`, every
     * executable file under `bin` will automatically be added to this section.
     */
    readonly bin?: Record<string, string>;
    /**
     * Automatically add all executables under the `bin` directory to your
     * `package.json` file under the `bin` section.
     *
     * @default true
     */
    readonly autoDetectBin?: boolean;
    /**
     * npm scripts to include. If a script has the same name as a standard script,
     * the standard script will be overwritten.
     * Also adds the script as a task.
     *
     * @default {}
     * @deprecated use `project.addTask()` or `package.setScript()`
     */
    readonly scripts?: {
        [name: string]: string;
    };
    /**
     * The Node Package Manager used to execute scripts
     *
     * @default NodePackageManager.YARN_CLASSIC
     */
    readonly packageManager?: NodePackageManager;
    /**
     * The repository is the location where the actual code for your package lives.
     * See https://classic.yarnpkg.com/en/docs/package-json/#toc-repository
     */
    readonly repository?: string;
    /**
     * If the package.json for your package is not in the root directory (for example if it is part of a monorepo),
     * you can specify the directory in which it lives.
     */
    readonly repositoryDirectory?: string;
    /**
     * Author's name
     */
    readonly authorName?: string;
    /**
     * Author's e-mail
     */
    readonly authorEmail?: string;
    /**
     * Author's URL / Website
     */
    readonly authorUrl?: string;
    /**
     * Is the author an organization
     */
    readonly authorOrganization?: boolean;
    /**
     * Package's Homepage / Website
     */
    readonly homepage?: string;
    /**
     * Package's Stability
     */
    readonly stability?: string;
    /**
     * Minimum Node.js version to require via package.json `engines` (inclusive).
     *
     * @default - no "engines" specified
     */
    readonly minNodeVersion?: string;
    /**
     * Minimum node.js version to require via `engines` (inclusive).
     *
     * @default - no max
     */
    readonly maxNodeVersion?: string;
    /**
     * The version of PNPM to use if using PNPM as a package manager.
     *
     * @default "7"
     */
    readonly pnpmVersion?: string;
    /**
     * License's SPDX identifier.
     * See https://github.com/projen/projen/tree/main/license-text for a list of supported licenses.
     * Use the `licensed` option if you want to no license to be specified.
     *
     * @default "Apache-2.0"
     */
    readonly license?: string;
    /**
     * Indicates if a license should be added.
     *
     * @default true
     */
    readonly licensed?: boolean;
    /**
     * The base URL of the npm package registry.
     *
     * Must be a URL (e.g. start with "https://" or "http://")
     *
     * @default "https://registry.npmjs.org"
     */
    readonly npmRegistryUrl?: string;
    /**
     * The host name of the npm registry to publish to. Cannot be set together with `npmRegistryUrl`.
     *
     * @deprecated use `npmRegistryUrl` instead
     */
    readonly npmRegistry?: string;
    /**
     * The url to your project's issue tracker.
     */
    readonly bugsUrl?: string;
    /**
     * The email address to which issues should be reported.
     */
    readonly bugsEmail?: string;
    /**
     * Access level of the npm package.
     *
     * @default - for scoped packages (e.g. `foo@bar`), the default is
     * `NpmAccess.RESTRICTED`, for non-scoped packages, the default is
     * `NpmAccess.PUBLIC`.
     */
    readonly npmAccess?: NpmAccess;
    /**
     * Should provenance statements be generated when the package is published.
     *
     * A supported package manager is required to publish a package with npm provenance statements and
     * you will need to use a supported CI/CD provider.
     *
     * Note that the projen `Release` and `Publisher` components are using `publib` to publish packages,
     * which is using npm internally and supports provenance statements independently of the package manager used.
     *
     * @see https://docs.npmjs.com/generating-provenance-statements
     * @default - true for public packages, false otherwise
     */
    readonly npmProvenance?: boolean;
    /**
     * GitHub secret which contains the NPM token to use when publishing packages.
     *
     * @default "NPM_TOKEN"
     */
    readonly npmTokenSecret?: string;
    /**
     * Options for npm packages using AWS CodeArtifact.
     * This is required if publishing packages to, or installing scoped packages from AWS CodeArtifact
     *
     * @default - undefined
     */
    readonly codeArtifactOptions?: CodeArtifactOptions;
    /**
     * Options for privately hosted scoped packages
     *
     * @default - fetch all scoped packages from the public npm registry
     */
    readonly scopedPackagesOptions?: ScopedPackagesOptions[];
    /**
     * Options for Yarn Berry
     *
     * @default - Yarn Berry v4 with all default options
     */
    readonly yarnBerryOptions?: YarnBerryOptions;
}
/**
 * Options for authorizing requests to a AWS CodeArtifact npm repository.
 */
export declare enum CodeArtifactAuthProvider {
    /**
     * Fixed credentials provided via Github secrets.
     */
    ACCESS_AND_SECRET_KEY_PAIR = "ACCESS_AND_SECRET_KEY_PAIR",
    /**
     * Ephemeral credentials provided via Github's OIDC integration with an IAM role.
     * See:
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html
     * https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
     */
    GITHUB_OIDC = "GITHUB_OIDC"
}
/**
 * Options for publishing npm packages to AWS CodeArtifact.
 */
export interface CodeArtifactOptions {
    /**
     * Provider to use for authorizing requests to AWS CodeArtifact.
     *
     * @default CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR
     */
    readonly authProvider?: CodeArtifactAuthProvider;
    /**
     * GitHub secret which contains the AWS access key ID to use when publishing packages to AWS CodeArtifact.
     * This property must be specified only when publishing to AWS CodeArtifact (`npmRegistryUrl` contains AWS CodeArtifact URL).
     *
     * @default - When the `authProvider` value is set to
     * `CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR`, the default is
     * "AWS_ACCESS_KEY_ID". For `CodeArtifactAuthProvider.GITHUB_OIDC`, this
     * value must be left undefined.
     */
    readonly accessKeyIdSecret?: string;
    /**
     * GitHub secret which contains the AWS secret access key to use when publishing packages to AWS CodeArtifact.
     * This property must be specified only when publishing to AWS CodeArtifact (`npmRegistryUrl` contains AWS CodeArtifact URL).
     *
     * @default - When the `authProvider` value is set to
     * `CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR`, the default is
     * "AWS_SECRET_ACCESS_KEY". For `CodeArtifactAuthProvider.GITHUB_OIDC`, this
     * value must be left undefined.
     */
    readonly secretAccessKeySecret?: string;
    /**
     * ARN of AWS role to be assumed prior to get authorization token from AWS CodeArtifact
     * This property must be specified only when publishing to AWS CodeArtifact (`registry` contains AWS CodeArtifact URL).
     * When using the `CodeArtifactAuthProvider.GITHUB_OIDC` auth provider, this value must be defined.
     *
     * @default undefined
     */
    readonly roleToAssume?: string;
}
/**
 * Options for scoped packages
 */
export interface ScopedPackagesOptions {
    /**
     * Scope of the packages
     *
     * @example "@angular"
     */
    readonly scope: string;
    /**
     * URL of the registry for scoped packages
     */
    readonly registryUrl: string;
}
/**
 * Represents the npm `package.json` file.
 */
export declare class NodePackage extends Component {
    /**
     * Returns the `NodePackage` instance associated with a project or `undefined` if
     * there is no NodePackage.
     * @param project The project
     * @returns A NodePackage, or undefined
     */
    static of(project: Project): NodePackage | undefined;
    /**
     * The name of the npm package.
     */
    readonly packageName: string;
    /**
     * The module's entrypoint (e.g. `lib/index.js`).
     */
    readonly entrypoint: string;
    /**
     * Allow project to take library dependencies.
     */
    readonly allowLibraryDependencies: boolean;
    /**
     * The package manager to use.
     */
    readonly packageManager: NodePackageManager;
    /**
     * @deprecated use `addField(x, y)`
     */
    readonly manifest: any;
    /**
     * Minimum node.js version required by this package.
     * @default - no minimum
     */
    readonly minNodeVersion?: string;
    /**
     * Maximum node version required by this package.
     * @default - no maximum.
     */
    readonly maxNodeVersion?: string;
    /**
     * The version of PNPM to use if using PNPM as a package manager.
     *
     * @default "7"
     */
    readonly pnpmVersion?: string;
    /**
     * The SPDX license of this module. `undefined` if this package is not licensed.
     */
    readonly license?: string;
    /**
     * npm registry (e.g. `https://registry.npmjs.org`). Use `npmRegistryHost` to get just the host name.
     */
    readonly npmRegistryUrl: string;
    /**
     * The npm registry host (e.g. `registry.npmjs.org`).
     */
    readonly npmRegistry: string;
    /**
     * GitHub secret which contains the NPM token to use when publishing packages.
     */
    readonly npmTokenSecret?: string;
    /**
     * Options for npm packages using AWS CodeArtifact.
     * This is required if publishing packages to, or installing scoped packages from AWS CodeArtifact
     *
     * @default - undefined
     */
    readonly codeArtifactOptions?: CodeArtifactOptions;
    /**
     * Options for privately hosted scoped packages
     *
     * @default undefined
     */
    readonly scopedPackagesOptions?: ScopedPackagesOptions[];
    /**
     * npm package access level.
     */
    readonly npmAccess: NpmAccess;
    /**
     * Should provenance statements be generated when package is published.
     */
    readonly npmProvenance: boolean;
    /**
     * The name of the lock file.
     */
    readonly lockFile: string;
    /**
     * The task for installing project dependencies (non-frozen)
     */
    readonly installTask: Task;
    /**
     * The task for installing project dependencies (frozen)
     */
    readonly installCiTask: Task;
    /**
     * The package.json file.
     */
    readonly file: JsonFile;
    private readonly scripts;
    private readonly scriptsToBeRemoved;
    private readonly keywords;
    private readonly bin;
    private readonly engines;
    private readonly peerDependencyOptions;
    private readonly _prev?;
    private _renderedDeps?;
    constructor(project: Project, options?: NodePackageOptions);
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps: string[]): void;
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps: string[]): void;
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps: string[]): void;
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps: string[]): void;
    /**
     * Adds an `engines` requirement to your package.
     * @param engine The engine (e.g. `node`)
     * @param version The semantic version requirement (e.g. `^10`)
     */
    addEngine(engine: string, version: string): void;
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords: string[]): void;
    addBin(bins: Record<string, string>): void;
    /**
     * Add a npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name: string, command: string): void;
    /**
     * Removes an npm script (always successful).
     *
     * @param name The name of the script.
     */
    removeScript(name: string): void;
    /**
     * Indicates if a script by the given name is defined.
     * @param name The name of the script
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name: string): boolean;
    /**
     * Directly set fields in `package.json`.
     * @escape
     * @param name field name
     * @param value field value
     */
    addField(name: string, value: any): void;
    /**
     * Sets the package version.
     * @param version Package version.
     */
    addVersion(version: string): void;
    /**
     * Defines resolutions for dependencies to change the normally resolved
     * version of a dependency to something else.
     *
     * @param resolutions Names resolutions to be added. Specify a version or
     * range with this syntax:
     * `module@^7`
     */
    addPackageResolutions(...resolutions: string[]): void;
    /**
     * Returns the command to execute in order to install all dependencies (always frozen).
     */
    get installCommand(): string;
    /**
     * Renders `yarn install` or `npm install` with lockfile update (not frozen)
     */
    get installAndUpdateLockfileCommand(): string;
    /**
     * Attempt to resolve the currently installed version for a given dependency.
     *
     * @remarks
     * This method will first look through the current project's dependencies.
     * If found and semantically valid (not '*'), that will be used.
     * Otherwise, it will fall back to locating a `package.json` manifest for the dependency
     * through node's internal resolution reading the version from there.
     *
     * @param dependencyName Dependency to resolve for.
     */
    tryResolveDependencyVersion(dependencyName: string): string | undefined;
    synthesize(): void;
    postSynthesize(): void;
    /**
     * The command which executes "projen".
     * @deprecated use `project.projenCommand` instead.
     */
    get projenCommand(): string;
    /**
     * Returns `true` if we are running within a CI build.
     */
    private get isAutomatedBuild();
    private determineVersion;
    /**
     * Returns `true` if this is a CI release build.
     */
    private get isReleaseBuild();
    private parseNpmOptions;
    private parseScopedPackagesOptions;
    private addCodeArtifactLoginScript;
    private addNodeEngine;
    private renderNpmRegistryPath;
    private renderInstallCommand;
    private processDeps;
    private renderDependencies;
    /**
     * Resolves any deps that do not have a specified version (e.g. `*`) and
     * update `package.json` if needed.
     *
     * @returns `true` if package.json was updated or `false` if not.
     */
    private resolveDepsAndWritePackageJson;
    private renderPackageResolutions;
    private renderPublishConfig;
    private renderKeywords;
    private renderEngines;
    private autoDiscoverBinaries;
    private renderAuthor;
    private renderBin;
    private renderScripts;
    private npmScriptForTask;
    private readPackageJson;
    private installDependencies;
    private configureYarnBerry;
    private checkForConflictingYarnOptions;
    /** See https://yarnpkg.com/getting-started/qa#which-files-should-be-gitignored */
    private configureYarnBerryGitignore;
}
export interface PeerDependencyOptions {
    /**
     * Automatically add a pinned dev dependency.
     * @default true
     */
    readonly pinnedDevDependency?: boolean;
}
/**
 * The node package manager to use.
 */
export declare enum NodePackageManager {
    /**
     * Use `yarn` as the package manager.
     *
     * @deprecated For `yarn` 1.x use `YARN_CLASSIC` for `yarn` >= 2 use `YARN_BERRY`. Currently, `NodePackageManager.YARN` means `YARN_CLASSIC`. In the future, we might repurpose it to mean `YARN_BERRY`.
     */
    YARN = "yarn",
    /**
     * Use `yarn` versions >= 2 as the package manager.
     *
     * @deprecated use YARN_BERRY instead
     */
    YARN2 = "yarn2",
    /**
     * Use `yarn` 1.x as the package manager.
     */
    YARN_CLASSIC = "yarn_classic",
    /**
     * Use `yarn` versions >= 2 as the package manager.
     */
    YARN_BERRY = "yarn_berry",
    /**
     * Use `npm` as the package manager.
     */
    NPM = "npm",
    /**
     * Use `pnpm` as the package manager.
     */
    PNPM = "pnpm",
    /**
     * Use `bun` as the package manager
     */
    BUN = "bun"
}
/**
 * Npm package access level
 */
export declare enum NpmAccess {
    /**
     * Package is public.
     */
    PUBLIC = "public",
    /**
     * Package can only be accessed with credentials.
     */
    RESTRICTED = "restricted"
}
/**
 * Configure Yarn Berry
 */
export interface YarnBerryOptions {
    /**
     * A fully specified version to use for yarn (e.g., x.x.x)
     *
     * @default - 4.0.1
     */
    readonly version?: string;
    /**
     * The yarnrc configuration.
     *
     * @default - a blank Yarn RC file
     */
    readonly yarnRcOptions?: YarnrcOptions;
    /**
     * Should zero-installs be enabled?
     * Learn more at: https://yarnpkg.com/features/caching#zero-installs
     *
     * @default false
     */
    readonly zeroInstalls?: boolean;
}
export declare function defaultNpmToken(npmToken: string | undefined, registry: string | undefined): string | undefined;
