"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.defaultNpmToken = exports.NpmAccess = exports.NodePackageManager = exports.NodePackage = exports.CodeArtifactAuthProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path_1 = require("path");
const semver = require("semver");
const util_1 = require("./util");
const yarnrc_1 = require("./yarnrc");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const json_1 = require("../json");
const release_1 = require("../release");
const task_runtime_1 = require("../task-runtime");
const util_2 = require("../util");
const UNLICENSED = "UNLICENSED";
const DEFAULT_NPM_REGISTRY_URL = "https://registry.npmjs.org/";
const GITHUB_PACKAGES_REGISTRY = "npm.pkg.github.com";
const DEFAULT_NPM_TOKEN_SECRET = "NPM_TOKEN";
const DEFAULT_GITHUB_TOKEN_SECRET = "GITHUB_TOKEN";
/**
 * Options for authorizing requests to a AWS CodeArtifact npm repository.
 */
var CodeArtifactAuthProvider;
(function (CodeArtifactAuthProvider) {
    /**
     * Fixed credentials provided via Github secrets.
     */
    CodeArtifactAuthProvider["ACCESS_AND_SECRET_KEY_PAIR"] = "ACCESS_AND_SECRET_KEY_PAIR";
    /**
     * Ephemeral credentials provided via Github's OIDC integration with an IAM role.
     * See:
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html
     * https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
     */
    CodeArtifactAuthProvider["GITHUB_OIDC"] = "GITHUB_OIDC";
})(CodeArtifactAuthProvider || (exports.CodeArtifactAuthProvider = CodeArtifactAuthProvider = {}));
/**
 * Represents the npm `package.json` file.
 */
class NodePackage extends component_1.Component {
    /**
     * Returns the `NodePackage` instance associated with a project or `undefined` if
     * there is no NodePackage.
     * @param project The project
     * @returns A NodePackage, or undefined
     */
    static of(project) {
        const isIt = (o) => o instanceof NodePackage;
        return project.components.find(isIt);
    }
    constructor(project, options = {}) {
        super(project);
        this.scripts = {};
        this.scriptsToBeRemoved = new Set();
        this.keywords = new Set();
        this.bin = {};
        this.engines = {};
        this.packageName = options.packageName ?? project.name;
        this.peerDependencyOptions = {
            pinnedDevDependency: true,
            ...options.peerDependencyOptions,
        };
        this.allowLibraryDependencies = options.allowLibraryDependencies ?? true;
        this.packageManager =
            options.packageManager ?? NodePackageManager.YARN_CLASSIC;
        this.entrypoint = options.entrypoint ?? "lib/index.js";
        this.lockFile = determineLockfile(this.packageManager);
        this.project.annotateGenerated(`/${this.lockFile}`);
        const { npmAccess, npmRegistry, npmRegistryUrl, npmTokenSecret, codeArtifactOptions, scopedPackagesOptions, npmProvenance, } = this.parseNpmOptions(options);
        this.npmAccess = npmAccess;
        this.npmRegistry = npmRegistry;
        this.npmRegistryUrl = npmRegistryUrl;
        this.npmTokenSecret = npmTokenSecret;
        this.codeArtifactOptions = codeArtifactOptions;
        this.scopedPackagesOptions = scopedPackagesOptions;
        this.npmProvenance = npmProvenance;
        this.processDeps(options);
        this._prev = this.readPackageJson();
        // empty objects are here to preserve order for backwards compatibility
        this.manifest = {
            name: this.packageName,
            description: options.description,
            repository: !options.repository
                ? undefined
                : {
                    type: "git",
                    url: options.repository,
                    directory: options.repositoryDirectory,
                },
            bin: () => this.renderBin(),
            scripts: () => this.renderScripts(),
            author: this.renderAuthor(options),
            devDependencies: {},
            peerDependencies: {},
            dependencies: {},
            bundledDependencies: [],
            ...this.renderPackageResolutions(),
            keywords: () => this.renderKeywords(),
            engines: () => this.renderEngines(),
            main: this.entrypoint !== "" ? this.entrypoint : undefined,
            license: () => this.license ?? UNLICENSED,
            homepage: options.homepage,
            publishConfig: () => this.renderPublishConfig(),
            typesVersions: this._prev?.typesVersions,
            // in release CI builds we bump the version before we run "build" so we want
            // to preserve the version number. otherwise, we always set it to 0.0.0
            version: this.determineVersion(this._prev?.version),
            bugs: options.bugsEmail || options.bugsUrl
                ? {
                    email: options.bugsEmail,
                    url: options.bugsUrl,
                }
                : undefined,
        };
        // Configure Yarn Berry if using
        if (this.packageManager === NodePackageManager.YARN_BERRY ||
            this.packageManager === NodePackageManager.YARN2) {
            this.configureYarnBerry(project, options);
        }
        // add tasks for scripts from options (if specified)
        // @deprecated
        for (const [cmdname, shell] of Object.entries(options.scripts ?? {})) {
            project.addTask(cmdname, { exec: shell });
        }
        this.file = new json_1.JsonFile(this, "package.json", {
            obj: this.manifest,
            readonly: false,
            newline: true,
            committed: true, // needs to be committed so users can install the dependencies
        });
        this.addKeywords(...(options.keywords ?? []));
        this.addBin(options.bin ?? {});
        // automatically add all executable files under "bin"
        if (options.autoDetectBin ?? true) {
            this.autoDiscoverBinaries();
        }
        // node version
        this.minNodeVersion = options.minNodeVersion;
        this.maxNodeVersion = options.maxNodeVersion;
        this.pnpmVersion = options.pnpmVersion ?? "7";
        this.addNodeEngine();
        this.addCodeArtifactLoginScript();
        // license
        if (options.licensed ?? true) {
            this.license = options.license ?? "Apache-2.0";
        }
        this.installTask = project.addTask("install", {
            description: "Install project dependencies and update lockfile (non-frozen)",
            exec: this.installAndUpdateLockfileCommand,
        });
        this.installCiTask = project.addTask("install:ci", {
            description: "Install project dependencies using frozen lockfile",
            exec: this.installCommand,
        });
    }
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.RUNTIME);
        }
    }
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
        }
    }
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps) {
        if (Object.keys(deps).length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add peer dependencies to an APP project: ${Object.keys(deps).join(",")}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.PEER);
        }
    }
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps) {
        if (deps.length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add bundled dependencies to an APP project: ${deps.join(",")}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUNDLED);
        }
    }
    /**
     * Adds an `engines` requirement to your package.
     * @param engine The engine (e.g. `node`)
     * @param version The semantic version requirement (e.g. `^10`)
     */
    addEngine(engine, version) {
        this.engines[engine] = version;
    }
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords) {
        for (const k of keywords) {
            this.keywords.add(k);
        }
    }
    addBin(bins) {
        for (const [k, v] of Object.entries(bins)) {
            this.bin[k] = v;
        }
    }
    /**
     * Add a npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name, command) {
        this.scripts[name] = command;
    }
    /**
     * Removes an npm script (always successful).
     *
     * @param name The name of the script.
     */
    removeScript(name) {
        // need to keep track in case there's a task of the same name
        this.scriptsToBeRemoved.add(name);
        delete this.scripts[name];
    }
    /**
     * Indicates if a script by the given name is defined.
     * @param name The name of the script
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name) {
        return this.project.tasks.tryFind(name) !== undefined;
    }
    /**
     * Directly set fields in `package.json`.
     * @escape
     * @param name field name
     * @param value field value
     */
    addField(name, value) {
        this.manifest[name] = value;
    }
    /**
     * Sets the package version.
     * @param version Package version.
     */
    addVersion(version) {
        this.manifest.version = version;
    }
    /**
     * Defines resolutions for dependencies to change the normally resolved
     * version of a dependency to something else.
     *
     * @param resolutions Names resolutions to be added. Specify a version or
     * range with this syntax:
     * `module@^7`
     */
    addPackageResolutions(...resolutions) {
        for (const resolution of resolutions) {
            this.project.deps.addDependency(resolution, dependencies_1.DependencyType.OVERRIDE);
        }
    }
    /**
     * Returns the command to execute in order to install all dependencies (always frozen).
     */
    get installCommand() {
        return this.renderInstallCommand(true);
    }
    /**
     * Renders `yarn install` or `npm install` with lockfile update (not frozen)
     */
    get installAndUpdateLockfileCommand() {
        return this.renderInstallCommand(false);
    }
    /**
     * Attempt to resolve the currently installed version for a given dependency.
     *
     * @remarks
     * This method will first look through the current project's dependencies.
     * If found and semantically valid (not '*'), that will be used.
     * Otherwise, it will fall back to locating a `package.json` manifest for the dependency
     * through node's internal resolution reading the version from there.
     *
     * @param dependencyName Dependency to resolve for.
     */
    tryResolveDependencyVersion(dependencyName) {
        try {
            const fromDeps = this.project.deps.tryGetDependency(dependencyName);
            const version = semver.coerce(fromDeps?.version, { loose: true });
            if (version) {
                return version.format();
            }
        }
        catch { }
        return (0, util_1.tryResolveDependencyVersion)(dependencyName, {
            paths: [this.project.outdir],
        });
    }
    // ---------------------------------------------------------------------------------------
    synthesize() {
        this._renderedDeps = this.renderDependencies();
        super.synthesize();
    }
    postSynthesize() {
        super.postSynthesize();
        // only run "install" if package.json has changed or if we don't have a
        // `node_modules` directory.
        if (this.file.changed ||
            !(0, fs_1.existsSync)((0, path_1.join)(this.project.outdir, "node_modules"))) {
            this.installDependencies();
        }
        // resolve "*" deps in package.json and update it. if it was changed,
        // install deps again so that lockfile is updated.
        if (this.resolveDepsAndWritePackageJson()) {
            this.installDependencies();
        }
    }
    /**
     * The command which executes "projen".
     * @deprecated use `project.projenCommand` instead.
     */
    get projenCommand() {
        return this.project.projenCommand;
    }
    /**
     * Returns `true` if we are running within a CI build.
     */
    get isAutomatedBuild() {
        return (0, util_2.isTruthy)(process.env.CI);
    }
    determineVersion(currVersion) {
        if (!this.isReleaseBuild) {
            return "0.0.0";
        }
        return currVersion ?? "0.0.0";
    }
    /**
     * Returns `true` if this is a CI release build.
     */
    get isReleaseBuild() {
        return (0, util_2.isTruthy)(process.env.RELEASE);
    }
    // -------------------------------------------------------------------------------------------
    parseNpmOptions(options) {
        let npmRegistryUrl = options.npmRegistryUrl;
        if (options.npmRegistry) {
            if (npmRegistryUrl) {
                throw new Error('cannot use the deprecated "npmRegistry" together with "npmRegistryUrl". please use "npmRegistryUrl" instead.');
            }
            npmRegistryUrl = `https://${options.npmRegistry}`;
        }
        const npmr = new URL(npmRegistryUrl ?? DEFAULT_NPM_REGISTRY_URL);
        if (!npmr || !npmr.hostname || !npmr.href) {
            throw new Error(`unable to determine npm registry host from url ${npmRegistryUrl}. Is this really a URL?`);
        }
        const npmAccess = options.npmAccess ?? defaultNpmAccess(this.packageName);
        if (!isScoped(this.packageName) && npmAccess === NpmAccess.RESTRICTED) {
            throw new Error(`"npmAccess" cannot be RESTRICTED for non-scoped npm package "${this.packageName}"`);
        }
        const npmProvenance = options.npmProvenance ?? npmAccess === NpmAccess.PUBLIC;
        if (npmProvenance && npmAccess !== NpmAccess.PUBLIC) {
            throw new Error(`"npmProvenance" can only be enabled for public packages`);
        }
        const isAwsCodeArtifact = (0, release_1.isAwsCodeArtifactRegistry)(npmRegistryUrl);
        const hasScopedPackage = options.scopedPackagesOptions &&
            options.scopedPackagesOptions.length !== 0;
        if (isAwsCodeArtifact) {
            if (options.npmTokenSecret) {
                throw new Error('"npmTokenSecret" must not be specified when publishing AWS CodeArtifact.');
            }
            else if (options.codeArtifactOptions?.authProvider ===
                CodeArtifactAuthProvider.GITHUB_OIDC) {
                if (options.codeArtifactOptions.accessKeyIdSecret ||
                    options.codeArtifactOptions.secretAccessKeySecret) {
                    throw new Error("access and secret key pair should not be provided when using GITHUB_OIDC auth provider for AWS CodeArtifact");
                }
                else if (!options.codeArtifactOptions.roleToAssume) {
                    throw new Error('"roleToAssume" property is required when using GITHUB_OIDC for AWS CodeArtifact options');
                }
            }
        }
        else {
            if ((options.codeArtifactOptions?.accessKeyIdSecret ||
                options.codeArtifactOptions?.secretAccessKeySecret ||
                options.codeArtifactOptions?.roleToAssume) &&
                !hasScopedPackage) {
                throw new Error("codeArtifactOptions must only be specified when publishing AWS CodeArtifact or used in scoped packages.");
            }
        }
        // apply defaults for AWS CodeArtifact
        let codeArtifactOptions;
        if (isAwsCodeArtifact || hasScopedPackage) {
            const authProvider = options.codeArtifactOptions?.authProvider ??
                CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR;
            const isAccessSecretKeyPairAuth = authProvider === CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR;
            codeArtifactOptions = {
                authProvider,
                accessKeyIdSecret: options.codeArtifactOptions?.accessKeyIdSecret ??
                    (isAccessSecretKeyPairAuth ? "AWS_ACCESS_KEY_ID" : undefined),
                secretAccessKeySecret: options.codeArtifactOptions?.secretAccessKeySecret ??
                    (isAccessSecretKeyPairAuth ? "AWS_SECRET_ACCESS_KEY" : undefined),
                roleToAssume: options.codeArtifactOptions?.roleToAssume,
            };
        }
        return {
            npmAccess,
            npmRegistry: npmr.hostname + this.renderNpmRegistryPath(npmr.pathname),
            npmRegistryUrl: npmr.href,
            npmTokenSecret: defaultNpmToken(options.npmTokenSecret, npmr.hostname),
            codeArtifactOptions,
            scopedPackagesOptions: this.parseScopedPackagesOptions(options.scopedPackagesOptions),
            npmProvenance,
        };
    }
    parseScopedPackagesOptions(scopedPackagesOptions) {
        if (!scopedPackagesOptions) {
            return undefined;
        }
        return scopedPackagesOptions.map((option) => {
            if (!isScoped(option.scope)) {
                throw new Error(`Scope must start with "@" in options, found ${option.scope}`);
            }
            if (!(0, release_1.isAwsCodeArtifactRegistry)(option.registryUrl)) {
                throw new Error(`Only AWS Code artifact scoped registry is supported for now, found ${option.registryUrl}`);
            }
            const result = {
                registryUrl: option.registryUrl,
                scope: option.scope,
            };
            return result;
        });
    }
    addCodeArtifactLoginScript() {
        if (!this.scopedPackagesOptions ||
            this.scopedPackagesOptions.length === 0) {
            return;
        }
        this.project.addTask("ca:login", {
            requiredEnv: ["AWS_ACCESS_KEY_ID", "AWS_SECRET_ACCESS_KEY"],
            steps: [
                { exec: "which aws" },
                ...this.scopedPackagesOptions.map((scopedPackagesOption) => {
                    const { registryUrl, scope } = scopedPackagesOption;
                    const { domain, region, accountId, registry } = (0, util_1.extractCodeArtifactDetails)(registryUrl);
                    // reference: https://docs.aws.amazon.com/codeartifact/latest/ug/npm-auth.html
                    const commands = [
                        `npm config set ${scope}:registry ${registryUrl}`,
                        `CODEARTIFACT_AUTH_TOKEN=$(aws codeartifact get-authorization-token --domain ${domain} --region ${region} --domain-owner ${accountId} --query authorizationToken --output text)`,
                        `npm config set //${registry}:_authToken=$CODEARTIFACT_AUTH_TOKEN`,
                    ];
                    if (!this.minNodeVersion || semver.major(this.minNodeVersion) <= 16)
                        commands.push(`npm config set //${registry}:always-auth=true`);
                    return {
                        exec: commands.join("; "),
                    };
                }),
            ],
        });
    }
    addNodeEngine() {
        if (!this.minNodeVersion && !this.maxNodeVersion) {
            return;
        }
        let nodeVersion = "";
        if (this.minNodeVersion) {
            nodeVersion += `>= ${this.minNodeVersion}`;
        }
        if (this.maxNodeVersion) {
            nodeVersion += ` <= ${this.maxNodeVersion}`;
        }
        this.addEngine("node", nodeVersion);
    }
    renderNpmRegistryPath(path) {
        if (!path || path == "/") {
            return "";
        }
        else {
            return path;
        }
    }
    renderInstallCommand(frozen) {
        switch (this.packageManager) {
            case NodePackageManager.YARN:
            case NodePackageManager.YARN_CLASSIC:
                return [
                    "yarn install",
                    "--check-files",
                    ...(frozen ? ["--frozen-lockfile"] : []),
                ].join(" ");
            case NodePackageManager.YARN2:
            case NodePackageManager.YARN_BERRY:
                return ["yarn install", ...(frozen ? ["--immutable"] : [])].join(" ");
            case NodePackageManager.NPM:
                return frozen ? "npm ci" : "npm install";
            case NodePackageManager.PNPM:
                return frozen
                    ? "pnpm i --frozen-lockfile"
                    : "pnpm i --no-frozen-lockfile";
            case NodePackageManager.BUN:
                return ["bun install", ...(frozen ? ["--frozen-lockfile"] : [])].join(" ");
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
    }
    processDeps(options) {
        this.addDeps(...(options.deps ?? []));
        this.addDevDeps(...(options.devDeps ?? []));
        this.addPeerDeps(...(options.peerDeps ?? []));
        this.addBundledDeps(...(options.bundledDeps ?? []));
    }
    renderDependencies() {
        const devDependencies = {};
        const peerDependencies = {};
        const dependencies = {};
        const bundledDependencies = new Array();
        // synthetic dependencies: add a pinned build dependency to ensure we are
        // testing against the minimum requirement of the peer.
        if (this.peerDependencyOptions.pinnedDevDependency) {
            for (const dep of this.project.deps.all.filter((d) => d.type === dependencies_1.DependencyType.PEER)) {
                let req = dep.name;
                // skip if we already have a runtime dependency on this peer
                if (this.project.deps.tryGetDependency(dep.name, dependencies_1.DependencyType.RUNTIME)) {
                    continue;
                }
                if (dep.version) {
                    const ver = (0, util_1.minVersion)(dep.version);
                    if (!ver) {
                        throw new Error(`unable to determine minimum semver for peer dependency ${dep.name}@${dep.version}`);
                    }
                    req += "@" + ver;
                }
                this.addDevDeps(req);
            }
        }
        for (const dep of this.project.deps.all) {
            let version = dep.version ?? "*";
            let name = dep.name;
            if (name.startsWith("file:")) {
                const localDependencyPath = name.substring(5);
                const depPackageJson = (0, path_1.resolve)(this.project.outdir, localDependencyPath, "package.json");
                const pkgFile = (0, fs_1.readFileSync)(depPackageJson, "utf8");
                const pkg = JSON.parse(pkgFile);
                version = localDependencyPath;
                name = pkg.name;
            }
            switch (dep.type) {
                case dependencies_1.DependencyType.BUNDLED:
                    bundledDependencies.push(name);
                    if (this.project.deps.all.find((d) => d.name === name && d.type === dependencies_1.DependencyType.PEER)) {
                        throw new Error(`unable to bundle "${name}". it cannot appear as a peer dependency`);
                    }
                    // also add as a runtime dependency
                    dependencies[name] = version;
                    break;
                case dependencies_1.DependencyType.PEER:
                    peerDependencies[name] = version;
                    break;
                case dependencies_1.DependencyType.RUNTIME:
                    dependencies[name] = version;
                    break;
                case dependencies_1.DependencyType.TEST:
                case dependencies_1.DependencyType.DEVENV:
                case dependencies_1.DependencyType.BUILD:
                    devDependencies[name] = version;
                    break;
            }
        }
        // returns a lazy value to normalize during synthesis
        const normalize = (obj) => () => (0, util_2.sorted)(obj);
        // update the manifest we are about to save into `package.json`
        this.manifest.devDependencies = normalize(devDependencies);
        this.manifest.peerDependencies = normalize(peerDependencies);
        this.manifest.dependencies = normalize(dependencies);
        this.manifest.bundledDependencies = (0, util_2.sorted)(bundledDependencies);
        // nothing further to do if package.json file does not exist
        if (!this._prev) {
            return { devDependencies, peerDependencies, dependencies };
        }
        const readDeps = (user, current = {}) => {
            for (const [name, userVersion] of Object.entries(user)) {
                const currentVersion = current[name];
                // respect user version if it's not '*' or if current version is undefined
                if (userVersion !== "*" || !currentVersion || currentVersion === "*") {
                    continue;
                }
                // memoize current version in memory so it is preserved when saving
                user[name] = currentVersion;
            }
            // report removals
            for (const name of Object.keys(current ?? {})) {
                if (!user[name]) {
                    this.project.logger.verbose(`${name}: removed`);
                }
            }
        };
        readDeps(devDependencies, this._prev.devDependencies);
        readDeps(dependencies, this._prev.dependencies);
        readDeps(peerDependencies, this._prev.peerDependencies);
        return { devDependencies, dependencies, peerDependencies };
    }
    /**
     * Resolves any deps that do not have a specified version (e.g. `*`) and
     * update `package.json` if needed.
     *
     * @returns `true` if package.json was updated or `false` if not.
     */
    resolveDepsAndWritePackageJson() {
        const outdir = this.project.outdir;
        const rootPackageJson = (0, path_1.join)(outdir, "package.json");
        const original = (0, fs_1.readFileSync)(rootPackageJson, "utf8");
        const pkg = JSON.parse(original);
        const resolveDeps = (current, user) => {
            const result = {};
            current = current ?? {};
            user = user ?? {};
            for (const [name, currentDefinition] of Object.entries(user)) {
                // find actual version from node_modules
                let desiredVersion = currentDefinition;
                if (currentDefinition === "*") {
                    // we already know we don't have the version in project `deps`,
                    // so skip straight to checking manifest.
                    const resolvedVersion = (0, util_1.tryResolveDependencyVersion)(name, {
                        paths: [this.project.outdir],
                    });
                    if (!resolvedVersion) {
                        this.project.logger.warn(`unable to resolve version for ${name} from installed modules`);
                        continue;
                    }
                    desiredVersion = `^${resolvedVersion}`;
                }
                if (currentDefinition !== desiredVersion) {
                    this.project.logger.verbose(`${name}: ${currentDefinition} => ${desiredVersion}`);
                }
                result[name] = desiredVersion;
            }
            // print removed packages
            for (const name of Object.keys(current)) {
                if (!result[name]) {
                    this.project.logger.verbose(`${name} removed`);
                }
            }
            return result;
        };
        const rendered = this._renderedDeps;
        if (!rendered) {
            throw new Error("assertion failed");
        }
        const deps = resolveDeps(pkg.dependencies, rendered.dependencies);
        const devDeps = resolveDeps(pkg.devDependencies, rendered.devDependencies);
        const peerDeps = resolveDeps(pkg.peerDependencies, rendered.peerDependencies);
        if (this.peerDependencyOptions.pinnedDevDependency) {
            for (const [name, version] of Object.entries(peerDeps)) {
                // Skip if we already have a runtime dependency on this peer
                // or if devDependency version is already set.
                // Relies on the "*" devDependency added in the presynth step
                if (deps[name] || rendered.devDependencies[name] !== "*") {
                    continue;
                }
                // Take version and pin as dev dependency
                const ver = (0, util_1.minVersion)(version);
                if (!ver) {
                    throw new Error(`unable to determine minimum semver for peer dependency ${name}@${version}`);
                }
                devDeps[name] = ver;
            }
        }
        pkg.dependencies = (0, util_2.sorted)(deps);
        pkg.devDependencies = (0, util_2.sorted)(devDeps);
        pkg.peerDependencies = (0, util_2.sorted)(peerDeps);
        const updated = JSON.stringify(pkg, undefined, 2) + "\n";
        if (original === updated) {
            return false;
        }
        (0, util_2.writeFile)(rootPackageJson, updated);
        return true;
    }
    renderPackageResolutions() {
        const render = () => {
            const overridingDependencies = this.project.deps.all.filter((dep) => dep.type === dependencies_1.DependencyType.OVERRIDE);
            if (!overridingDependencies.length) {
                return undefined;
            }
            return Object.fromEntries(overridingDependencies.map(({ name, version = "*" }) => [name, version]));
        };
        switch (this.packageManager) {
            case NodePackageManager.NPM:
                return { overrides: render };
            case NodePackageManager.PNPM:
                return this.project.parent
                    ? undefined
                    : { pnpm: { overrides: render } };
            case NodePackageManager.YARN:
            case NodePackageManager.YARN2:
            case NodePackageManager.YARN_CLASSIC:
            case NodePackageManager.YARN_BERRY:
            case NodePackageManager.BUN:
            default:
                return { resolutions: render };
        }
    }
    renderPublishConfig() {
        // omit values if they are the same as the npm defaults
        return (0, _resolve_1.resolve)({
            registry: this.npmRegistryUrl !== DEFAULT_NPM_REGISTRY_URL
                ? this.npmRegistryUrl
                : undefined,
            access: this.npmAccess !== defaultNpmAccess(this.packageName)
                ? this.npmAccess
                : undefined,
        }, { omitEmpty: true });
    }
    renderKeywords() {
        const kwds = Array.from(this.keywords);
        return (0, util_2.sorted)(kwds.sort());
    }
    renderEngines() {
        return (0, util_2.sorted)(this.engines);
    }
    autoDiscoverBinaries() {
        const binrel = "bin";
        const bindir = (0, path_1.join)(this.project.outdir, binrel);
        if ((0, fs_1.existsSync)(bindir)) {
            for (const file of (0, fs_1.readdirSync)(bindir)) {
                try {
                    (0, fs_1.accessSync)((0, path_1.join)(bindir, file), fs_1.constants.X_OK);
                    const binPath = (0, path_1.join)(binrel, file);
                    const normalizedPath = (0, util_2.normalizePersistedPath)(binPath);
                    this.bin[file] = normalizedPath;
                }
                catch (e) {
                    // not executable, skip
                }
            }
        }
    }
    renderAuthor(options) {
        let author;
        if (options.authorName) {
            author = {
                name: options.authorName,
                email: options.authorEmail,
                url: options.authorUrl,
                organization: options.authorOrganization ?? false,
            };
        }
        else {
            if (options.authorEmail ||
                options.authorUrl ||
                options.authorOrganization !== undefined) {
                throw new Error('"authorName" is required if specifying "authorEmail" or "authorUrl"');
            }
        }
        return author;
    }
    renderBin() {
        return (0, util_2.sorted)(this.bin);
    }
    renderScripts() {
        const result = {};
        const tasks = this.project.tasks.all
            .filter((t) => 
        // Must remove to prevent overriding built-in npm command (which would loop)
        t.name !== this.installTask.name && t.name !== this.installCiTask.name)
            .sort((x, y) => x.name.localeCompare(y.name));
        for (const task of tasks) {
            if (this.scriptsToBeRemoved.has(task.name)) {
                continue;
            }
            result[task.name] = this.npmScriptForTask(task);
        }
        return {
            ...result,
            ...this.scripts,
        };
    }
    npmScriptForTask(task) {
        return `${this.projenCommand} ${task.name}`;
    }
    readPackageJson() {
        const file = (0, path_1.join)(this.project.outdir, "package.json");
        if (!(0, fs_1.existsSync)(file)) {
            return undefined;
        }
        return JSON.parse((0, fs_1.readFileSync)(file, "utf-8"));
    }
    installDependencies() {
        this.project.logger.info("Installing dependencies...");
        const runtime = new task_runtime_1.TaskRuntime(this.project.outdir);
        const taskToRun = this.isAutomatedBuild
            ? this.installCiTask
            : this.installTask;
        runtime.runTask(taskToRun.name);
    }
    configureYarnBerry(project, options) {
        const { version = "4.0.1", yarnRcOptions = {}, zeroInstalls = false, } = options.yarnBerryOptions || {};
        this.checkForConflictingYarnOptions(yarnRcOptions);
        // Set the `packageManager` field in `package.json` to the version specified. This tells `corepack` which version
        // of `yarn` to use.
        this.addField("packageManager", `yarn@${version}`);
        this.configureYarnBerryGitignore(zeroInstalls);
        new yarnrc_1.Yarnrc(project, version, yarnRcOptions);
    }
    checkForConflictingYarnOptions(yarnRcOptions) {
        if (this.npmAccess &&
            yarnRcOptions.npmPublishAccess &&
            this.npmAccess.toString() !== yarnRcOptions.npmPublishAccess.toString()) {
            throw new Error(`Cannot set npmAccess (${this.npmAccess}) and yarnRcOptions.npmPublishAccess (${yarnRcOptions.npmPublishAccess}) to different values.`);
        }
        if (this.npmRegistryUrl &&
            yarnRcOptions.npmRegistryServer &&
            this.npmRegistryUrl !== yarnRcOptions.npmRegistryServer) {
            throw new Error(`Cannot set npmRegistryUrl (${this.npmRegistryUrl}) and yarnRcOptions.npmRegistryServer (${yarnRcOptions.npmRegistryServer}) to different values.`);
        }
    }
    /** See https://yarnpkg.com/getting-started/qa#which-files-should-be-gitignored */
    configureYarnBerryGitignore(zeroInstalls) {
        const { gitignore } = this.project;
        // These patterns are the same whether or not you're using zero-installs
        gitignore.exclude(".yarn/*");
        gitignore.include(".yarn/patches", ".yarn/plugins", ".yarn/releases", ".yarn/sdks", ".yarn/versions");
        if (zeroInstalls) {
            gitignore.include("!.yarn/cache");
        }
        else {
            gitignore.exclude(".pnp.*");
        }
    }
}
exports.NodePackage = NodePackage;
_a = JSII_RTTI_SYMBOL_1;
NodePackage[_a] = { fqn: "projen.javascript.NodePackage", version: "0.79.27" };
/**
 * The node package manager to use.
 */
var NodePackageManager;
(function (NodePackageManager) {
    /**
     * Use `yarn` as the package manager.
     *
     * @deprecated For `yarn` 1.x use `YARN_CLASSIC` for `yarn` >= 2 use `YARN_BERRY`. Currently, `NodePackageManager.YARN` means `YARN_CLASSIC`. In the future, we might repurpose it to mean `YARN_BERRY`.
     */
    NodePackageManager["YARN"] = "yarn";
    /**
     * Use `yarn` versions >= 2 as the package manager.
     *
     * @deprecated use YARN_BERRY instead
     */
    NodePackageManager["YARN2"] = "yarn2";
    /**
     * Use `yarn` 1.x as the package manager.
     */
    NodePackageManager["YARN_CLASSIC"] = "yarn_classic";
    /**
     * Use `yarn` versions >= 2 as the package manager.
     */
    NodePackageManager["YARN_BERRY"] = "yarn_berry";
    /**
     * Use `npm` as the package manager.
     */
    NodePackageManager["NPM"] = "npm";
    /**
     * Use `pnpm` as the package manager.
     */
    NodePackageManager["PNPM"] = "pnpm";
    /**
     * Use `bun` as the package manager
     */
    NodePackageManager["BUN"] = "bun";
})(NodePackageManager || (exports.NodePackageManager = NodePackageManager = {}));
/**
 * Npm package access level
 */
var NpmAccess;
(function (NpmAccess) {
    /**
     * Package is public.
     */
    NpmAccess["PUBLIC"] = "public";
    /**
     * Package can only be accessed with credentials.
     */
    NpmAccess["RESTRICTED"] = "restricted";
})(NpmAccess || (exports.NpmAccess = NpmAccess = {}));
/**
 * Determines if an npm package is "scoped" (i.e. it starts with "xxx@").
 */
function isScoped(packageName) {
    return packageName.includes("@");
}
function defaultNpmAccess(packageName) {
    return isScoped(packageName) ? NpmAccess.RESTRICTED : NpmAccess.PUBLIC;
}
function defaultNpmToken(npmToken, registry) {
    // if we are publishing to AWS CdodeArtifact, no NPM_TOKEN used (will be requested using AWS CLI later).
    if ((0, release_1.isAwsCodeArtifactRegistry)(registry)) {
        return undefined;
    }
    // if we are publishing to GitHub Packages, default to GITHUB_TOKEN.
    const isGitHubPackages = registry === GITHUB_PACKAGES_REGISTRY;
    return (npmToken ??
        (isGitHubPackages ? DEFAULT_GITHUB_TOKEN_SECRET : DEFAULT_NPM_TOKEN_SECRET));
}
exports.defaultNpmToken = defaultNpmToken;
function determineLockfile(packageManager) {
    if (packageManager === NodePackageManager.YARN ||
        packageManager === NodePackageManager.YARN2 ||
        packageManager === NodePackageManager.YARN_CLASSIC ||
        packageManager === NodePackageManager.YARN_BERRY) {
        return "yarn.lock";
    }
    else if (packageManager === NodePackageManager.NPM) {
        return "package-lock.json";
    }
    else if (packageManager === NodePackageManager.PNPM) {
        return "pnpm-lock.yaml";
    }
    else if (packageManager === NodePackageManager.BUN) {
        return "bun.lockb";
    }
    throw new Error(`unsupported package manager ${packageManager}`);
}
//# sourceMappingURL=data:application/json;base64,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