import { Bundler, BundlerOptions } from "./bundler";
import { Jest, JestOptions } from "./jest";
import { LicenseCheckerOptions } from "./license-checker";
import { NodePackage, NodePackageManager, NodePackageOptions } from "./node-package";
import { ProjenrcOptions } from "./projenrc";
import { BuildWorkflow } from "../build";
import { AutoMerge, DependabotOptions, GitHubProject, GitHubProjectOptions, GitIdentity } from "../github";
import { JobStep, JobStepConfiguration, Triggers } from "../github/workflows-model";
import { IgnoreFile, IgnoreFileOptions } from "../ignore-file";
import { NpmConfig, Prettier, PrettierOptions, UpgradeDependencies, UpgradeDependenciesOptions } from "../javascript";
import { Publisher, Release, ReleaseProjectOptions } from "../release";
import { Task } from "../task";
export interface NodeProjectOptions extends GitHubProjectOptions, NodePackageOptions, ReleaseProjectOptions {
    /**
     * License copyright owner.
     *
     * @default - defaults to the value of authorName or "" if `authorName` is undefined.
     */
    readonly copyrightOwner?: string;
    /**
     * The copyright years to put in the LICENSE file.
     *
     * @default - current year
     */
    readonly copyrightPeriod?: string;
    /**
     * Version of projen to install.
     *
     * @default - Defaults to the latest version.
     */
    readonly projenVersion?: string;
    /**
     * Indicates of "projen" should be installed as a devDependency.
     *
     * @default true
     */
    readonly projenDevDependency?: boolean;
    /**
     * Define a GitHub workflow for building PRs.
     * @default - true if not a subproject
     */
    readonly buildWorkflow?: boolean;
    /**
     * Automatically update files modified during builds to pull-request branches. This means
     * that any files synthesized by projen or e.g. test snapshots will always be up-to-date
     * before a PR is merged.
     *
     * Implies that PR builds do not have anti-tamper checks.
     *
     * @default true
     */
    readonly mutableBuild?: boolean;
    /**
     * Define a GitHub workflow step for sending code coverage metrics to https://codecov.io/
     * Uses codecov/codecov-action@v3
     * A secret is required for private repos. Configured with `@codeCovTokenSecret`
     * @default false
     */
    readonly codeCov?: boolean;
    /**
     * Define the secret name for a specified https://codecov.io/ token
     * A secret is required to send coverage for private repositories
     * @default - if this option is not specified, only public repositories are supported
     */
    readonly codeCovTokenSecret?: string;
    /**
     * DEPRECATED: renamed to `release`.
     *
     * @default - true if not a subproject
     * @deprecated see `release`.
     */
    readonly releaseWorkflow?: boolean;
    /**
     * Add release management to this project.
     *
     * @default - true (false for subprojects)
     */
    readonly release?: boolean;
    /**
     * The name of the main release branch.
     *
     * @default "main"
     */
    readonly defaultReleaseBranch: string;
    /**
     * Workflow steps to use in order to bootstrap this repo.
     *
     * @default "yarn install --frozen-lockfile && yarn projen"
     */
    readonly workflowBootstrapSteps?: JobStep[];
    /**
     * The git identity to use in workflows.
     *
     * @default - GitHub Actions
     */
    readonly workflowGitIdentity?: GitIdentity;
    /**
     * Automatically release to npm when new versions are introduced.
     * @default false
     */
    readonly releaseToNpm?: boolean;
    /**
     * The node version to use in GitHub workflows.
     *
     * @default - same as `minNodeVersion`
     */
    readonly workflowNodeVersion?: string;
    /**
     * Enable Node.js package cache in GitHub workflows.
     *
     * @default false
     */
    readonly workflowPackageCache?: boolean;
    /**
     * Use dependabot to handle dependency upgrades.
     * Cannot be used in conjunction with `depsUpgrade`.
     *
     * @default false
     */
    readonly dependabot?: boolean;
    /**
     * Options for dependabot.
     *
     * @default - default options
     */
    readonly dependabotOptions?: DependabotOptions;
    /**
     * Use tasks and github workflows to handle dependency upgrades.
     * Cannot be used in conjunction with `dependabot`.
     *
     * @default true
     */
    readonly depsUpgrade?: boolean;
    /**
     * Options for `UpgradeDependencies`.
     *
     * @default - default options
     */
    readonly depsUpgradeOptions?: UpgradeDependenciesOptions;
    /**
     * Automatically approve deps upgrade PRs, allowing them to be
     * merged by mergify (if configued).
     *
     * Throw if set to true but `autoApproveOptions` are not defined.
     *
     * @default - true
     */
    readonly autoApproveUpgrades?: boolean;
    /**
     * Defines an .npmignore file. Normally this is only needed for libraries that
     * are packaged as tarballs.
     *
     * @default true
     */
    readonly npmignoreEnabled?: boolean;
    /**
     * Configuration options for .npmignore file
     */
    readonly npmIgnoreOptions?: IgnoreFileOptions;
    /**
     * Additional entries to .npmignore.
     * @deprecated - use `project.addPackageIgnore`
     */
    readonly npmignore?: string[];
    /**
     * Include a GitHub pull request template.
     *
     * @default true
     */
    readonly pullRequestTemplate?: boolean;
    /**
     * The contents of the pull request template.
     *
     * @default - default content
     */
    readonly pullRequestTemplateContents?: string[];
    /**
     * Setup prettier.
     *
     * @default false
     */
    readonly prettier?: boolean;
    /**
     * Prettier options
     * @default - default options
     */
    readonly prettierOptions?: PrettierOptions;
    /**
     * Additional entries to .gitignore
     */
    readonly gitignore?: string[];
    /**
     * Setup jest unit tests
     * @default true
     */
    readonly jest?: boolean;
    /**
     * Jest options
     * @default - default options
     */
    readonly jestOptions?: JestOptions;
    /**
     * Generate (once) .projenrc.js (in JavaScript). Set to `false` in order to disable
     * .projenrc.js generation.
     *
     * @default - true if projenrcJson is false
     */
    readonly projenrcJs?: boolean;
    /**
     * Options for .projenrc.js
     * @default - default options
     */
    readonly projenrcJsOptions?: ProjenrcOptions;
    /**
     * Options for `Bundler`.
     */
    readonly bundlerOptions?: BundlerOptions;
    /**
     * A directory which will contain build artifacts.
     *
     * @default "dist"
     */
    readonly artifactsDirectory?: string;
    /**
     * Defines a `package` task that will produce an npm tarball under the
     * artifacts directory (e.g. `dist`).
     *
     * @default true
     */
    readonly package?: boolean;
    /**
     * Build workflow triggers
     * @default "{ pullRequest: {}, workflowDispatch: {} }"
     */
    readonly buildWorkflowTriggers?: Triggers;
    /**
     * Configure which licenses should be deemed acceptable for use by dependencies
     *
     * This setting will cause the build to fail, if any prohibited or not allowed licenses ares encountered.
     *
     * @default - no license checks are run during the build and all licenses will be accepted
     */
    readonly checkLicenses?: LicenseCheckerOptions;
}
/**
 * Automatic bump modes
 */
export declare enum AutoRelease {
    /**
     * Automatically bump & release a new version for every commit to "main"
     */
    EVERY_COMMIT = 0,
    /**
     * Automatically bump & release a new version on a daily basis.
     */
    DAILY = 1
}
/**
 * Node.js project.
 *
 * @pjid node
 */
export declare class NodeProject extends GitHubProject {
    /**
     * API for managing the node package.
     */
    readonly package: NodePackage;
    /**
     * The .npmignore file.
     */
    readonly npmignore?: IgnoreFile;
    /**
     * The .npmrc file
     */
    get npmrc(): NpmConfig;
    private _npmrc?;
    /**
     * @deprecated use `package.allowLibraryDependencies`
     */
    get allowLibraryDependencies(): boolean;
    /**
     * @deprecated use `package.entrypoint`
     */
    get entrypoint(): string;
    /**
     * Component that sets up mergify for merging approved pull requests.
     */
    readonly autoMerge?: AutoMerge;
    /**
     * The PR build GitHub workflow. `undefined` if `buildWorkflow` is disabled.
     */
    readonly buildWorkflow?: BuildWorkflow;
    /**
     * Package publisher. This will be `undefined` if the project does not have a
     * release workflow.
     *
     * @deprecated use `release.publisher`.
     */
    readonly publisher?: Publisher;
    /**
     * Release management.
     */
    readonly release?: Release;
    /**
     * Minimum node.js version required by this package.
     */
    get minNodeVersion(): string | undefined;
    /**
     * Maximum node version required by this package.
     */
    get maxNodeVersion(): string | undefined;
    protected readonly nodeVersion?: string;
    /**
     * The package manager to use.
     *
     * @deprecated use `package.packageManager`
     */
    get packageManager(): NodePackageManager;
    /**
     * The command to use to run scripts (e.g. `yarn run` or `npm run` depends on the package manager).
     */
    readonly runScriptCommand: string;
    /**
     * The Jest configuration (if enabled)
     */
    readonly jest?: Jest;
    /**
     * @deprecated use `package.addField(x, y)`
     */
    get manifest(): any;
    readonly bundler: Bundler;
    /**
     * The build output directory. An npm tarball will be created under the `js`
     * subdirectory. For example, if this is set to `dist` (the default), the npm
     * tarball will be placed under `dist/js/boom-boom-1.2.3.tg`.
     */
    readonly artifactsDirectory: string;
    /**
     * The location of the npm tarball after build (`${artifactsDirectory}/js`).
     */
    readonly artifactsJavascriptDirectory: string;
    /**
     * The upgrade workflow.
     */
    readonly upgradeWorkflow?: UpgradeDependencies;
    protected readonly workflowBootstrapSteps: JobStep[];
    private readonly workflowGitIdentity;
    protected readonly workflowPackageCache: boolean;
    readonly prettier?: Prettier;
    constructor(options: NodeProjectOptions);
    private determineInstallWorkingDirectory;
    private renderUploadCoverageJobStep;
    addBins(bins: Record<string, string>): void;
    /**
     * Replaces the contents of an npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name: string, command: string): void;
    /**
     * Replaces the contents of multiple npm package.json scripts.
     * @param scripts The scripts to set
     */
    addScripts(scripts: {
        [name: string]: string;
    }): void;
    /**
     * Removes the npm script (always successful).
     * @param name The name of the script.
     */
    removeScript(name: string): void;
    /**
     * Indicates if a script by the name name is defined.
     * @param name The name of the script
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name: string): boolean;
    /**
     * DEPRECATED
     * @deprecated use `project.compileTask.exec()`
     */
    addCompileCommand(...commands: string[]): void;
    /**
     * DEPRECATED
     * @deprecated use `project.testTask.exec()`
     */
    addTestCommand(...commands: string[]): void;
    /**
     * Directly set fields in `package.json`.
     * @param fields The fields to set
     */
    addFields(fields: {
        [name: string]: any;
    }): void;
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords: string[]): void;
    /**
     * Get steps for scoped package access
     *
     * @param codeArtifactOptions Details of logging in to AWS
     * @returns array of job steps required for each private scoped packages
     */
    private getScopedPackageSteps;
    /**
     * Returns the set of workflow steps which should be executed to bootstrap a
     * workflow.
     *
     * @param options Options.
     * @returns Job steps
     */
    renderWorkflowSetup(options?: RenderWorkflowSetupOptions): JobStep[];
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps: string[]): void;
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps: string[]): void;
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps: string[]): void;
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps: string[]): void;
    /**
     * Adds patterns to be ignored by npm.
     *
     * @param pattern The pattern to ignore.
     *
     * @remarks
     * If you are having trouble getting an ignore to populate, try using your construct or component's preSynthesize method to properly delay calling this method.
     */
    addPackageIgnore(pattern: string): void;
    private addLicense;
    private addDefaultGitIgnore;
    /**
     * Returns the shell command to execute in order to run a task. This will
     * typically be `npx projen TASK`.
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task: Task): string;
    /**
     * The job ID of the build workflow.
     */
    get buildWorkflowJobId(): string | undefined;
}
/**
 * Options for `renderWorkflowSetup()`.
 */
export interface RenderWorkflowSetupOptions {
    /**
     * Configure the install step in the workflow setup.
     *
     * @default - `{ name: "Install dependencies" }`
     *
     * @example - { workingDirectory: "rootproject-dir" } for subprojects installing from root.
     * @example - { env: { NPM_TOKEN: "token" }} for installing from private npm registry.
     */
    readonly installStepConfiguration?: JobStepConfiguration;
    /**
     * Should the package lockfile be updated?
     * @default false
     */
    readonly mutable?: boolean;
}
