"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeProject = exports.AutoRelease = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const bundler_1 = require("./bundler");
const jest_1 = require("./jest");
const license_checker_1 = require("./license-checker");
const node_package_1 = require("./node-package");
const projenrc_1 = require("./projenrc");
const build_1 = require("../build");
const common_1 = require("../common");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const util_1 = require("../github/util");
const workflows_model_1 = require("../github/workflows-model");
const ignore_file_1 = require("../ignore-file");
const javascript_1 = require("../javascript");
const license_1 = require("../license");
const projenrc_json_1 = require("../projenrc-json");
const release_1 = require("../release");
const runner_options_1 = require("../runner-options");
const util_2 = require("../util");
const path_2 = require("../util/path");
const version_1 = require("../version");
const PROJEN_SCRIPT = "projen";
/**
 * Automatic bump modes
 */
var AutoRelease;
(function (AutoRelease) {
    /**
     * Automatically bump & release a new version for every commit to "main"
     */
    AutoRelease[AutoRelease["EVERY_COMMIT"] = 0] = "EVERY_COMMIT";
    /**
     * Automatically bump & release a new version on a daily basis.
     */
    AutoRelease[AutoRelease["DAILY"] = 1] = "DAILY";
})(AutoRelease || (exports.AutoRelease = AutoRelease = {}));
/**
 * Node.js project.
 *
 * @pjid node
 */
class NodeProject extends github_1.GitHubProject {
    /**
     * The .npmrc file
     */
    get npmrc() {
        if (!this._npmrc) {
            this._npmrc = new javascript_1.NpmConfig(this, { omitEmpty: true });
        }
        return this._npmrc;
    }
    /**
     * @deprecated use `package.allowLibraryDependencies`
     */
    get allowLibraryDependencies() {
        return this.package.allowLibraryDependencies;
    }
    /**
     * @deprecated use `package.entrypoint`
     */
    get entrypoint() {
        return this.package.entrypoint;
    }
    /**
     * Minimum node.js version required by this package.
     */
    get minNodeVersion() {
        return this.package.minNodeVersion;
    }
    /**
     * Maximum node version required by this package.
     */
    get maxNodeVersion() {
        return this.package.maxNodeVersion;
    }
    /**
     * The package manager to use.
     *
     * @deprecated use `package.packageManager`
     */
    get packageManager() {
        return this.package.packageManager;
    }
    /**
     * @deprecated use `package.addField(x, y)`
     */
    get manifest() {
        return this.package.manifest;
    }
    constructor(options) {
        super({
            ...options,
            // Node projects have the specific projen version locked via lockfile, so we can skip the @<VERSION> part of the top-level project
            projenCommand: options.projenCommand ?? "npx projen",
        });
        this.package = new node_package_1.NodePackage(this, options);
        this.workflowBootstrapSteps = options.workflowBootstrapSteps ?? [];
        this.workflowGitIdentity =
            options.workflowGitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.workflowPackageCache = options.workflowPackageCache ?? false;
        this.artifactsDirectory = options.artifactsDirectory ?? "dist";
        this.artifactsJavascriptDirectory = (0, path_1.join)(this.artifactsDirectory, "js");
        this.runScriptCommand = (() => {
            switch (this.packageManager) {
                case node_package_1.NodePackageManager.NPM:
                    return "npm run";
                case node_package_1.NodePackageManager.YARN:
                case node_package_1.NodePackageManager.YARN2:
                case node_package_1.NodePackageManager.YARN_CLASSIC:
                case node_package_1.NodePackageManager.YARN_BERRY:
                    return "yarn run";
                case node_package_1.NodePackageManager.PNPM:
                    return "pnpm run";
                case node_package_1.NodePackageManager.BUN:
                    return "bun run";
                default:
                    throw new Error(`unexpected package manager ${this.packageManager}`);
            }
        })();
        const envCommand = (() => {
            switch (this.packageManager) {
                case node_package_1.NodePackageManager.PNPM:
                    return '$(pnpm -c exec "node --print process.env.PATH")';
                default:
                    return '$(npx -c "node --print process.env.PATH")';
            }
        })();
        this.nodeVersion =
            options.workflowNodeVersion ?? this.package.minNodeVersion;
        // add PATH for all tasks which includes the project's npm .bin list
        this.tasks.addEnvironment("PATH", envCommand);
        this.addLicense(options);
        if (options.npmignoreEnabled ?? true) {
            this.npmignore = new ignore_file_1.IgnoreFile(this, ".npmignore", options.npmIgnoreOptions);
        }
        this.addDefaultGitIgnore();
        if (options.gitignore?.length) {
            for (const i of options.gitignore) {
                this.gitignore.exclude(i);
            }
        }
        if (options.npmignore?.length) {
            if (!this.npmignore) {
                throw new Error('.npmignore is not defined for an APP project type. Add "npmIgnore: true" to override this');
            }
            for (const i of options.npmignore) {
                this.npmignore.exclude(i);
            }
        }
        if (!this.ejected) {
            this.setScript(PROJEN_SCRIPT, this.package.projenCommand);
        }
        this.npmignore?.exclude(`/${common_1.PROJEN_DIR}/`);
        const projen = options.projenDevDependency ?? true;
        if (projen && !this.ejected) {
            const postfix = options.projenVersion ? `@${options.projenVersion}` : "";
            this.addDevDeps(`projen${postfix}`);
            this.addDevDeps(`constructs@^10.0.0`);
        }
        if (!options.defaultReleaseBranch) {
            throw new Error('"defaultReleaseBranch" is temporarily a required option while we migrate its default value from "master" to "main"');
        }
        const buildEnabled = options.buildWorkflow ?? (this.parent ? false : true);
        // configure jest if enabled
        // must be before the build/release workflows
        if (options.jest ?? true) {
            this.jest = new jest_1.Jest(this, options.jestOptions);
        }
        const requiresIdTokenPermission = (options.scopedPackagesOptions ?? []).length > 0 &&
            options.codeArtifactOptions?.authProvider ===
                release_1.CodeArtifactAuthProvider.GITHUB_OIDC;
        const workflowPermissions = {
            idToken: requiresIdTokenPermission ? workflows_model_1.JobPermission.WRITE : undefined,
        };
        if (buildEnabled && this.github) {
            this.buildWorkflow = new build_1.BuildWorkflow(this, {
                buildTask: this.buildTask,
                artifactsDirectory: this.artifactsDirectory,
                containerImage: options.workflowContainerImage,
                gitIdentity: this.workflowGitIdentity,
                mutableBuild: options.mutableBuild,
                preBuildSteps: this.renderWorkflowSetup({
                    mutable: options.mutableBuild ?? true,
                }),
                postBuildSteps: options.postBuildSteps,
                ...(0, runner_options_1.filteredRunsOnOptions)(options.workflowRunsOn, options.workflowRunsOnGroup),
                workflowTriggers: options.buildWorkflowTriggers,
                permissions: workflowPermissions,
            });
            this.buildWorkflow.addPostBuildSteps(...this.renderUploadCoverageJobStep(options));
        }
        const release = options.release ??
            options.releaseWorkflow ??
            (this.parent ? false : true);
        if (release) {
            this.addDevDeps(version_1.Version.STANDARD_VERSION);
            this.release = new release_1.Release(this, {
                versionFile: "package.json",
                task: this.buildTask,
                branch: options.defaultReleaseBranch ?? "main",
                ...options,
                artifactsDirectory: this.artifactsDirectory,
                releaseWorkflowSetupSteps: [
                    ...this.renderWorkflowSetup({
                        installStepConfiguration: {
                            workingDirectory: this.determineInstallWorkingDirectory(),
                        },
                        mutable: false,
                    }),
                    ...(options.releaseWorkflowSetupSteps ?? []),
                ],
                postBuildSteps: [
                    ...(options.postBuildSteps ?? []),
                    ...this.renderUploadCoverageJobStep(options),
                ],
                workflowNodeVersion: this.nodeVersion,
                workflowPermissions,
            });
            this.publisher = this.release.publisher;
            const nodePackageToReleaseCodeArtifactAuthProviderMapping = {
                [node_package_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR]: release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                [node_package_1.CodeArtifactAuthProvider.GITHUB_OIDC]: release_1.CodeArtifactAuthProvider.GITHUB_OIDC,
            };
            if (options.releaseToNpm ?? false) {
                const codeArtifactOptions = (0, release_1.isAwsCodeArtifactRegistry)(this.package.npmRegistry)
                    ? {
                        accessKeyIdSecret: options.codeArtifactOptions?.accessKeyIdSecret,
                        secretAccessKeySecret: options.codeArtifactOptions?.secretAccessKeySecret,
                        roleToAssume: options.codeArtifactOptions?.roleToAssume,
                        authProvider: options.codeArtifactOptions?.authProvider
                            ? nodePackageToReleaseCodeArtifactAuthProviderMapping[options.codeArtifactOptions.authProvider]
                            : release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                    }
                    : {};
                this.release.publisher.publishToNpm({
                    registry: this.package.npmRegistry,
                    npmTokenSecret: this.package.npmTokenSecret,
                    npmProvenance: this.package.npmProvenance,
                    codeArtifactOptions,
                });
            }
        }
        else {
            // validate that no release options are selected if the release workflow is disabled.
            if (options.releaseToNpm) {
                throw new Error('"releaseToNpm" is not supported if "release" is not set');
            }
            if (options.releaseEveryCommit) {
                throw new Error('"releaseEveryCommit" is not supported if "release" is not set');
            }
            if (options.releaseSchedule) {
                throw new Error('"releaseSchedule" is not supported if "release" is not set');
            }
        }
        if ((options.autoMerge ?? true) &&
            this.github?.mergify &&
            this.buildWorkflow?.buildJobIds) {
            this.autoMerge = new github_1.AutoMerge(this.github, options.autoMergeOptions);
            this.autoMerge.addConditionsLater({
                render: () => this.buildWorkflow?.buildJobIds.map((id) => `status-success=${id}`) ??
                    [],
            });
        }
        const dependabot = options.dependabot ?? false;
        const depsUpgrade = options.depsUpgrade ?? !dependabot;
        if (dependabot && depsUpgrade) {
            throw new Error("'dependabot' cannot be configured together with 'depsUpgrade'");
        }
        const depsAutoApprove = options.autoApproveUpgrades ?? false;
        if (depsAutoApprove && !this.autoApprove && this.github) {
            throw new Error("Automatic approval of dependencies upgrades requires configuring `autoApproveOptions`");
        }
        const autoApproveLabel = (condition) => condition && this.autoApprove?.label
            ? [this.autoApprove.label]
            : undefined;
        if (dependabot) {
            const defaultOptions = {
                labels: autoApproveLabel(depsAutoApprove),
            };
            this.github?.addDependabot((0, util_2.deepMerge)([defaultOptions, options.dependabotOptions ?? {}]));
        }
        if (depsUpgrade) {
            const defaultOptions = {
                workflowOptions: {
                    container: options.workflowContainerImage
                        ? {
                            image: options.workflowContainerImage,
                        }
                        : undefined,
                    labels: autoApproveLabel(depsAutoApprove),
                    gitIdentity: this.workflowGitIdentity,
                    permissions: workflowPermissions,
                },
            };
            this.upgradeWorkflow = new javascript_1.UpgradeDependencies(this, (0, util_2.deepMerge)([defaultOptions, options.depsUpgradeOptions ?? {}]));
        }
        if (options.pullRequestTemplate ?? true) {
            this.github?.addPullRequestTemplate(...(options.pullRequestTemplateContents ?? []));
        }
        const projenrcJs = options.projenrcJs ?? !options.projenrcJson;
        if (!this.parent && projenrcJs) {
            const projenrcJsFile = new projenrc_1.Projenrc(this, options.projenrcJsOptions);
            this.npmignore?.exclude(`/${projenrcJsFile.filePath}`);
        }
        else if (options.projenrcJson) {
            const projenrcJsonFile = projenrc_json_1.ProjenrcJson.of(this);
            if (projenrcJsonFile) {
                this.npmignore?.exclude(`/${projenrcJsonFile.filePath}`);
            }
        }
        // add a bundler component - this enables things like Lambda bundling and in the future web bundling.
        this.bundler = new bundler_1.Bundler(this, options.bundlerOptions);
        if (options.package ?? true) {
            this.packageTask.exec(`mkdir -p ${this.artifactsJavascriptDirectory}`);
            const pkgMgr = this.package.packageManager === node_package_1.NodePackageManager.PNPM
                ? "pnpm"
                : "npm"; // sadly we cannot use --pack-destination because it is not supported by older npm
            this.packageTask.exec(`mv $(${pkgMgr} pack) ${this.artifactsJavascriptDirectory}/`);
        }
        if (options.prettier ?? false) {
            this.prettier = new javascript_1.Prettier(this, { ...options.prettierOptions });
        }
        // For PNPM, the default resolution mode is "lowest", which leads to any non-versioned (ie '*') dependencies being
        // resolved to the lowest available version, which is unlikely to be expected behaviour for users. We set resolution
        // mode to "highest" to match the behaviour of yarn and npm.
        if (this.package.packageManager === node_package_1.NodePackageManager.PNPM) {
            this.npmrc.addConfig("resolution-mode", "highest");
        }
        if (options.checkLicenses) {
            new license_checker_1.LicenseChecker(this, options.checkLicenses);
        }
    }
    determineInstallWorkingDirectory() {
        if (this.parent) {
            return (0, path_2.ensureRelativePathStartsWithDot)((0, path_1.relative)(".", this.root.outdir));
        }
        return;
    }
    renderUploadCoverageJobStep(options) {
        // run codecov if enabled or a secret token name is passed in
        // AND jest must be configured
        if ((options.codeCov || options.codeCovTokenSecret) && this.jest?.config) {
            return [
                {
                    name: "Upload coverage to Codecov",
                    uses: "codecov/codecov-action@v3",
                    with: options.codeCovTokenSecret
                        ? {
                            token: `\${{ secrets.${options.codeCovTokenSecret} }}`,
                            directory: this.jest.config.coverageDirectory,
                        }
                        : {
                            directory: this.jest.config.coverageDirectory,
                        },
                },
            ];
        }
        else {
            return [];
        }
    }
    addBins(bins) {
        this.package.addBin(bins);
    }
    /**
     * Replaces the contents of an npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name, command) {
        this.package.setScript(name, command);
    }
    /**
     * Replaces the contents of multiple npm package.json scripts.
     * @param scripts The scripts to set
     */
    addScripts(scripts) {
        for (const [name, command] of Object.entries(scripts)) {
            this.package.setScript(name, command);
        }
    }
    /**
     * Removes the npm script (always successful).
     * @param name The name of the script.
     */
    removeScript(name) {
        this.package.removeScript(name);
    }
    /**
     * Indicates if a script by the name name is defined.
     * @param name The name of the script
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name) {
        return this.package.hasScript(name);
    }
    /**
     * DEPRECATED
     * @deprecated use `project.compileTask.exec()`
     */
    addCompileCommand(...commands) {
        for (const c of commands) {
            this.compileTask.exec(c);
        }
    }
    /**
     * DEPRECATED
     * @deprecated use `project.testTask.exec()`
     */
    addTestCommand(...commands) {
        for (const c of commands) {
            this.testTask.exec(c);
        }
    }
    /**
     * Directly set fields in `package.json`.
     * @param fields The fields to set
     */
    addFields(fields) {
        for (const [name, value] of Object.entries(fields)) {
            this.package.addField(name, value);
        }
    }
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords) {
        this.package.addKeywords(...keywords);
    }
    /**
     * Get steps for scoped package access
     *
     * @param codeArtifactOptions Details of logging in to AWS
     * @returns array of job steps required for each private scoped packages
     */
    getScopedPackageSteps(codeArtifactOptions) {
        const parsedCodeArtifactOptions = {
            accessKeyIdSecret: codeArtifactOptions?.accessKeyIdSecret ?? "AWS_ACCESS_KEY_ID",
            secretAccessKeySecret: codeArtifactOptions?.secretAccessKeySecret ?? "AWS_SECRET_ACCESS_KEY",
            roleToAssume: codeArtifactOptions?.roleToAssume,
            authProvider: codeArtifactOptions?.authProvider,
        };
        if (parsedCodeArtifactOptions.authProvider ===
            node_package_1.CodeArtifactAuthProvider.GITHUB_OIDC) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v4",
                    with: {
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        if (parsedCodeArtifactOptions.roleToAssume) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v4",
                    with: {
                        "aws-access-key-id": (0, util_1.secretToString)(parsedCodeArtifactOptions.accessKeyIdSecret),
                        "aws-secret-access-key": (0, util_1.secretToString)(parsedCodeArtifactOptions.secretAccessKeySecret),
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        return [
            {
                name: "AWS CodeArtifact Login",
                run: `${this.runScriptCommand} ca:login`,
                env: {
                    AWS_ACCESS_KEY_ID: (0, util_1.secretToString)(parsedCodeArtifactOptions.accessKeyIdSecret),
                    AWS_SECRET_ACCESS_KEY: (0, util_1.secretToString)(parsedCodeArtifactOptions.secretAccessKeySecret),
                },
            },
        ];
    }
    /**
     * Returns the set of workflow steps which should be executed to bootstrap a
     * workflow.
     *
     * @param options Options.
     * @returns Job steps
     */
    renderWorkflowSetup(options = {}) {
        const install = new Array();
        // first run the workflow bootstrap steps
        install.push(...this.workflowBootstrapSteps);
        if (this.package.packageManager === node_package_1.NodePackageManager.PNPM) {
            install.push({
                name: "Setup pnpm",
                uses: "pnpm/action-setup@v3",
                with: { version: this.package.pnpmVersion },
            });
        }
        if (this.nodeVersion || this.workflowPackageCache) {
            const cache = this.package.packageManager === node_package_1.NodePackageManager.YARN
                ? "yarn"
                : this.package.packageManager === node_package_1.NodePackageManager.YARN2
                    ? "yarn"
                    : this.package.packageManager === node_package_1.NodePackageManager.YARN_CLASSIC
                        ? "yarn"
                        : this.package.packageManager === node_package_1.NodePackageManager.YARN_BERRY
                            ? "yarn"
                            : this.packageManager === node_package_1.NodePackageManager.BUN
                                ? "bun"
                                : this.package.packageManager === node_package_1.NodePackageManager.PNPM
                                    ? "pnpm"
                                    : "npm";
            install.push({
                name: "Setup Node.js",
                uses: "actions/setup-node@v4",
                with: {
                    ...(this.nodeVersion && {
                        "node-version": this.nodeVersion,
                    }),
                    ...(this.workflowPackageCache && {
                        cache,
                    }),
                },
            });
        }
        const mutable = options.mutable ?? false;
        if (this.package.scopedPackagesOptions) {
            install.push(...this.getScopedPackageSteps(this.package.codeArtifactOptions));
        }
        install.push({
            name: "Install dependencies",
            run: mutable
                ? this.package.installAndUpdateLockfileCommand
                : this.package.installCommand,
            ...(options.installStepConfiguration ?? {}),
        });
        return install;
    }
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps) {
        return this.package.addDeps(...deps);
    }
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps) {
        return this.package.addDevDeps(...deps);
    }
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps) {
        return this.package.addPeerDeps(...deps);
    }
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps) {
        return this.package.addBundledDeps(...deps);
    }
    /**
     * Adds patterns to be ignored by npm.
     *
     * @param pattern The pattern to ignore.
     *
     * @remarks
     * If you are having trouble getting an ignore to populate, try using your construct or component's preSynthesize method to properly delay calling this method.
     */
    addPackageIgnore(pattern) {
        this.npmignore?.addPatterns(pattern);
    }
    addLicense(options) {
        if (this.package.license) {
            new license_1.License(this, {
                spdx: this.package.license,
                copyrightOwner: options.copyrightOwner ?? options.authorName,
                copyrightPeriod: options.copyrightPeriod,
            });
        }
    }
    addDefaultGitIgnore() {
        this.gitignore.exclude("# Logs", "logs", "*.log", "npm-debug.log*", "yarn-debug.log*", "yarn-error.log*", "lerna-debug.log*", "# Diagnostic reports (https://nodejs.org/api/report.html)", "report.[0-9]*.[0-9]*.[0-9]*.[0-9]*.json", "# Runtime data", "pids", "*.pid", "*.seed", "*.pid.lock", "# Directory for instrumented libs generated by jscoverage/JSCover", "lib-cov", "# Coverage directory used by tools like istanbul", "coverage", "*.lcov", "# nyc test coverage", ".nyc_output", "# Compiled binary addons (https://nodejs.org/api/addons.html)", "build/Release", "# Dependency directories", "node_modules/", "jspm_packages/", "# TypeScript cache", "*.tsbuildinfo", "# Optional eslint cache", ".eslintcache", "# Output of 'npm pack'", "*.tgz", "# Yarn Integrity file", ".yarn-integrity", "# parcel-bundler cache (https://parceljs.org/)", ".cache");
    }
    /**
     * Returns the shell command to execute in order to run a task. This will
     * typically be `npx projen TASK`.
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        return `${this.package.projenCommand} ${task.name}`;
    }
    /**
     * The job ID of the build workflow.
     */
    get buildWorkflowJobId() {
        return this.buildWorkflow?.buildJobIds[0];
    }
}
exports.NodeProject = NodeProject;
_a = JSII_RTTI_SYMBOL_1;
NodeProject[_a] = { fqn: "projen.javascript.NodeProject", version: "0.79.27" };
//# sourceMappingURL=data:application/json;base64,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