"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypescriptConfig = exports.TypescriptConfigExtends = exports.TypeScriptJsxMode = exports.TypeScriptImportsNotUsedAsValues = exports.TypeScriptModuleResolution = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const semver = require("semver");
const _1 = require(".");
const component_1 = require("../component");
const json_1 = require("../json");
/**
 * Determines how modules get resolved.
 *
 * @see https://www.typescriptlang.org/docs/handbook/module-resolution.html
 */
var TypeScriptModuleResolution;
(function (TypeScriptModuleResolution) {
    /**
     * TypeScript's former default resolution strategy.
     *
     * @see https://www.typescriptlang.org/docs/handbook/module-resolution.html#classic
     */
    TypeScriptModuleResolution["CLASSIC"] = "classic";
    /**
     * Resolution strategy which attempts to mimic the Node.js module resolution strategy at runtime.
     *
     * @see https://www.typescriptlang.org/docs/handbook/module-resolution.html#node
     */
    TypeScriptModuleResolution["NODE"] = "node";
    /**
     * Node.js’ ECMAScript Module Support from TypeScript 4.7 onwards
     *
     * @see https://www.typescriptlang.org/tsconfig#moduleResolution
     */
    TypeScriptModuleResolution["NODE16"] = "node16";
    /**
     * Node.js’ ECMAScript Module Support from TypeScript 4.7 onwards
     *
     * @see https://www.typescriptlang.org/tsconfig#moduleResolution
     */
    TypeScriptModuleResolution["NODE_NEXT"] = "nodenext";
    /**
     * Resolution strategy which attempts to mimic resolution patterns of modern bundlers; from TypeScript 5.0 onwards.
     *
     * @see https://www.typescriptlang.org/tsconfig#moduleResolution
     */
    TypeScriptModuleResolution["BUNDLER"] = "bundler";
})(TypeScriptModuleResolution || (exports.TypeScriptModuleResolution = TypeScriptModuleResolution = {}));
/**
 * This flag controls how `import` works, there are 3 different options.
 *
 * @see https://www.typescriptlang.org/tsconfig#importsNotUsedAsValues
 */
var TypeScriptImportsNotUsedAsValues;
(function (TypeScriptImportsNotUsedAsValues) {
    /**
     * The default behavior of dropping `import` statements which only reference types.
     */
    TypeScriptImportsNotUsedAsValues["REMOVE"] = "remove";
    /**
     * Preserves all `import` statements whose values or types are never used. This can cause imports/side-effects to be preserved.
     */
    TypeScriptImportsNotUsedAsValues["PRESERVE"] = "preserve";
    /**
     * This preserves all imports (the same as the preserve option), but will error when a value import is only used as a type.
     * This might be useful if you want to ensure no values are being accidentally imported, but still make side-effect imports explicit.
     */
    TypeScriptImportsNotUsedAsValues["ERROR"] = "error";
})(TypeScriptImportsNotUsedAsValues || (exports.TypeScriptImportsNotUsedAsValues = TypeScriptImportsNotUsedAsValues = {}));
/**
 * Determines how JSX should get transformed into valid JavaScript.
 *
 * @see https://www.typescriptlang.org/docs/handbook/jsx.html
 */
var TypeScriptJsxMode;
(function (TypeScriptJsxMode) {
    /**
     * Keeps the JSX as part of the output to be further consumed by another transform step (e.g. Babel).
     */
    TypeScriptJsxMode["PRESERVE"] = "preserve";
    /**
     * Converts JSX syntax into React.createElement, does not need to go through a JSX transformation before use, and the output will have a .js file extension.
     */
    TypeScriptJsxMode["REACT"] = "react";
    /**
     * Keeps all JSX like 'preserve' mode, but output will have a .js extension.
     */
    TypeScriptJsxMode["REACT_NATIVE"] = "react-native";
    /**
     * Passes `key` separately from props and always passes `children` as props (since React 17).
     *
     * @see https://www.typescriptlang.org/docs/handbook/release-notes/typescript-4-1.html#react-17-jsx-factories
     */
    TypeScriptJsxMode["REACT_JSX"] = "react-jsx";
    /**
     * Same as `REACT_JSX` with additional debug data.
     */
    TypeScriptJsxMode["REACT_JSXDEV"] = "react-jsxdev";
})(TypeScriptJsxMode || (exports.TypeScriptJsxMode = TypeScriptJsxMode = {}));
/**
 * Container for `TypescriptConfig` `tsconfig.json` base configuration(s).
 * Extending from more than one base config file requires TypeScript 5.0+.
 */
class TypescriptConfigExtends {
    /**
     * Factory for creation from array of file paths.
     *
     * @remarks
     * TypeScript 5.0+ is required to specify more than one value in `paths`.
     *
     * @param paths Absolute or relative paths to base `tsconfig.json` files.
     */
    static fromPaths(paths) {
        return new TypescriptConfigExtends(paths);
    }
    /**
     * Factory for creation from array of other `TypescriptConfig` instances.
     *
     * @remarks
     * TypeScript 5.0+ is required to specify more than on value in `configs`.
     *
     * @param configs Base `TypescriptConfig` instances.
     */
    static fromTypescriptConfigs(configs) {
        const paths = configs.map((config) => config.file.absolutePath);
        return TypescriptConfigExtends.fromPaths(paths);
    }
    constructor(bases) {
        this.bases = bases;
    }
    toJSON() {
        return this.bases;
    }
}
exports.TypescriptConfigExtends = TypescriptConfigExtends;
_a = JSII_RTTI_SYMBOL_1;
TypescriptConfigExtends[_a] = { fqn: "projen.javascript.TypescriptConfigExtends", version: "0.79.27" };
class TypescriptConfig extends component_1.Component {
    constructor(project, options) {
        super(project);
        const fileName = options.fileName ?? "tsconfig.json";
        this._extends = options.extends ?? TypescriptConfigExtends.fromPaths([]);
        this.include = options.include ?? ["**/*.ts"];
        this.exclude = options.exclude ?? ["node_modules"];
        this.fileName = fileName;
        this.compilerOptions = options.compilerOptions;
        this.file = new json_1.JsonFile(project, fileName, {
            allowComments: true,
            obj: {
                extends: () => this.renderExtends(),
                compilerOptions: this.compilerOptions,
                include: () => this.include,
                exclude: () => this.exclude,
            },
        });
        if (project instanceof _1.NodeProject) {
            project.npmignore?.exclude(`/${fileName}`);
        }
    }
    /**
     * Render appropriate value for `extends` field.
     * @private
     */
    renderExtends() {
        if (this.extends.length <= 1) {
            // render string value when only one extension (TS<5.0);
            // omit if no extensions.
            return this.extends[0];
        }
        // render many extensions as array (TS>=5.0)
        return this.extends;
    }
    /**
     * Resolve valid TypeScript extends paths relative to this config.
     *
     * @remarks
     * This will only resolve the relative path from this config to another given
     * an absolute path as input. Any non-absolute path or other string will be returned as is.
     * This is to preserve manually specified relative paths as well as npm import paths.
     *
     * @param configPath Path to resolve against.
     */
    resolveExtendsPath(configPath) {
        // if not absolute assume user-defined path (or npm package).
        if (!path.isAbsolute(configPath))
            return configPath;
        const relativeConfig = path.relative(path.dirname(this.file.absolutePath), configPath);
        // ensure immediate sibling files are prefixed with './'
        // typescript figures this out, but some tools seemingly break without it (i.e, eslint).
        const { dir, ...pathParts } = path.parse(relativeConfig);
        const configDir = dir
            ? path.format({ dir: dir.startsWith("..") ? "" : ".", base: dir })
            : ".";
        return path.format({ ...pathParts, dir: configDir });
    }
    /**
     * Validate usage of `extends` against current TypeScript version.
     * @private
     */
    validateExtends() {
        const project = this.project;
        const hasOneOrNoneExtends = this.extends.length <= 1;
        const isNodeProject = project instanceof _1.NodeProject;
        if (hasOneOrNoneExtends || !isNodeProject) {
            // always accept no extends or singular extends.
            return;
        }
        const tsVersion = semver.coerce(project.package.tryResolveDependencyVersion("typescript"), { loose: true });
        if (tsVersion && tsVersion.major < 5) {
            this.project.logger.warn("TypeScript < 5.0.0 can only extend from a single base config.", `TypeScript Version: ${tsVersion.format()}`, `File: ${this.file.absolutePath}`, `Extends: ${this.extends}`);
        }
    }
    /**
     * Array of base `tsconfig.json` paths.
     * Any absolute paths are resolved relative to this instance,
     * while any relative paths are used as is.
     */
    get extends() {
        return this._extends
            .toJSON()
            .map((value) => this.resolveExtendsPath(value));
    }
    /**
     * Extend from base `TypescriptConfig` instance.
     *
     * @remarks
     * TypeScript 5.0+ is required to extend from more than one base `TypescriptConfig`.
     *
     * @param value Base `TypescriptConfig` instance.
     */
    addExtends(value) {
        this._extends = TypescriptConfigExtends.fromPaths([
            ...this._extends.toJSON(),
            value.file.absolutePath,
        ]);
    }
    addInclude(pattern) {
        this.include.push(pattern);
    }
    addExclude(pattern) {
        this.exclude.push(pattern);
    }
    preSynthesize() {
        super.preSynthesize();
        this.validateExtends();
    }
}
exports.TypescriptConfig = TypescriptConfig;
_b = JSII_RTTI_SYMBOL_1;
TypescriptConfig[_b] = { fqn: "projen.javascript.TypescriptConfig", version: "0.79.27" };
//# sourceMappingURL=data:application/json;base64,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