"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpgradeDependenciesSchedule = exports.UpgradeDependencies = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflow_actions_1 = require("../github/workflow-actions");
const workflows_model_1 = require("../github/workflows-model");
const javascript_1 = require("../javascript");
const release_1 = require("../release");
const runner_options_1 = require("../runner-options");
const CREATE_PATCH_STEP_ID = "create_patch";
const PATCH_CREATED_OUTPUT = "patch_created";
/**
 * Upgrade node project dependencies.
 */
class UpgradeDependencies extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        /**
         * The workflows that execute the upgrades. One workflow per branch.
         */
        this.workflows = [];
        this._project = project;
        this.options = options;
        this.depTypes = this.options.types ?? [
            dependencies_1.DependencyType.BUILD,
            dependencies_1.DependencyType.BUNDLED,
            dependencies_1.DependencyType.DEVENV,
            dependencies_1.DependencyType.PEER,
            dependencies_1.DependencyType.RUNTIME,
            dependencies_1.DependencyType.TEST,
            dependencies_1.DependencyType.OPTIONAL,
        ];
        this.upgradeTarget = this.options.target ?? "minor";
        this.satisfyPeerDependencies = this.options.satisfyPeerDependencies ?? true;
        this.pullRequestTitle = options.pullRequestTitle ?? "upgrade dependencies";
        this.gitIdentity =
            options.workflowOptions?.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.permissions = {
            contents: workflows_model_1.JobPermission.READ,
            ...options.workflowOptions?.permissions,
        };
        this.postBuildSteps = [];
        this.containerOptions = options.workflowOptions?.container;
        this.postUpgradeTask =
            project.tasks.tryFind("post-upgrade") ??
                project.tasks.addTask("post-upgrade", {
                    description: "Runs after upgrading dependencies",
                });
        this.upgradeTask = project.addTask(options.taskName ?? "upgrade", {
            // this task should not run in CI mode because its designed to
            // update package.json and lock files.
            env: { CI: "0" },
            description: this.pullRequestTitle,
            steps: { toJSON: () => this.renderTaskSteps() },
        });
        this.upgradeTask.lock(); // this task is a lazy value, so make it readonly
        if (this.upgradeTask && project.github && (options.workflow ?? true)) {
            if (options.workflowOptions?.branches) {
                for (const branch of options.workflowOptions.branches) {
                    this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, branch));
                }
            }
            else if (release_1.Release.of(project)) {
                const release = release_1.Release.of(project);
                release._forEachBranch((branch) => {
                    this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, branch));
                });
            }
            else {
                // represents the default repository branch.
                // just like not specifying anything.
                const defaultBranch = undefined;
                this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, defaultBranch));
            }
        }
    }
    /**
     * Add steps to execute a successful build.
     * @param steps workflow steps
     */
    addPostBuildSteps(...steps) {
        this.postBuildSteps.push(...steps);
    }
    renderTaskSteps() {
        function executeCommand(packageManager) {
            switch (packageManager) {
                case javascript_1.NodePackageManager.NPM:
                case javascript_1.NodePackageManager.YARN:
                case javascript_1.NodePackageManager.YARN_CLASSIC:
                    return "npx";
                case javascript_1.NodePackageManager.PNPM:
                    return "pnpm dlx";
                case javascript_1.NodePackageManager.YARN2:
                case javascript_1.NodePackageManager.YARN_BERRY:
                    return "yarn dlx";
                case javascript_1.NodePackageManager.BUN:
                    return "bunx";
            }
        }
        const steps = new Array();
        // Package Manager upgrade should always include all deps
        const includeForPackageManagerUpgrade = this.buildDependencyList(true);
        if (includeForPackageManagerUpgrade.length === 0) {
            return [{ exec: "echo No dependencies to upgrade." }];
        }
        // Removing `npm-check-updates` from our dependency tree because it depends on a package
        // that uses an npm-specific feature that causes an invalid dependency tree when using Yarn 1.
        // See https://github.com/projen/projen/pull/3136 for more details.
        const includeForNcu = this.buildDependencyList(false);
        const ncuCommand = [
            `${executeCommand(this._project.package.packageManager)} npm-check-updates@16`,
            "--upgrade",
            `--target=${this.upgradeTarget}`,
            `--${this.satisfyPeerDependencies ? "peer" : "no-peer"}`,
            `--dep=${this.renderNcuDependencyTypes(this.depTypes)}`,
            `--filter=${includeForNcu.join(",")}`,
        ];
        // bump versions in package.json
        if (includeForNcu.length) {
            steps.push({ exec: ncuCommand.join(" ") });
        }
        // run "yarn/npm install" to update the lockfile and install any deps (such as projen)
        steps.push({ exec: this._project.package.installAndUpdateLockfileCommand });
        // run upgrade command to upgrade transitive deps as well
        steps.push({
            exec: this.renderUpgradePackagesCommand(includeForPackageManagerUpgrade),
        });
        // run "projen" to give projen a chance to update dependencies (it will also run "yarn install")
        steps.push({ exec: this._project.projenCommand });
        steps.push({ spawn: this.postUpgradeTask.name });
        return steps;
    }
    /**
     * Render projen dependencies types to a list of ncu compatible types
     */
    renderNcuDependencyTypes(types) {
        return Array.from(new Set(types
            .map((type) => {
            switch (type) {
                case dependencies_1.DependencyType.PEER:
                    return "peer";
                case dependencies_1.DependencyType.RUNTIME:
                    return "prod";
                case dependencies_1.DependencyType.OPTIONAL:
                    return "optional";
                case dependencies_1.DependencyType.TEST:
                case dependencies_1.DependencyType.DEVENV:
                case dependencies_1.DependencyType.BUILD:
                    return "dev";
                case dependencies_1.DependencyType.BUNDLED:
                default:
                    return false;
            }
        })
            .filter((type) => Boolean(type)))).join(",");
    }
    /**
     * Render a package manager specific command to upgrade all requested dependencies.
     */
    renderUpgradePackagesCommand(include) {
        function upgradePackages(command) {
            return () => {
                return `${command} ${include.join(" ")}`;
            };
        }
        const packageManager = this._project.package.packageManager;
        let lazy = undefined;
        switch (packageManager) {
            case javascript_1.NodePackageManager.YARN:
            case javascript_1.NodePackageManager.YARN_CLASSIC:
                lazy = upgradePackages("yarn upgrade");
                break;
            case javascript_1.NodePackageManager.YARN2:
            case javascript_1.NodePackageManager.YARN_BERRY:
                lazy = upgradePackages("yarn up");
                break;
            case javascript_1.NodePackageManager.NPM:
                lazy = upgradePackages("npm update");
                break;
            case javascript_1.NodePackageManager.PNPM:
                lazy = upgradePackages("pnpm update");
                break;
            case javascript_1.NodePackageManager.BUN:
                lazy = upgradePackages("bun update");
                break;
            default:
                throw new Error(`unexpected package manager ${packageManager}`);
        }
        // return a lazy function so that dependencies include ones that were
        // added post project instantiation (i.e using project.addDeps)
        return lazy;
    }
    buildDependencyList(includeDependenciesWithConstraint) {
        return Array.from(new Set(this.options.include ??
            this.filterDependencies(includeDependenciesWithConstraint)));
    }
    filterDependencies(includeConstraint) {
        const dependencies = [];
        const deps = this.project.deps.all
            // remove those that have a pinned version
            .filter((d) => includeConstraint || !(d.version && d.version[0] === "^"))
            // remove override dependencies
            .filter((d) => d.type !== dependencies_1.DependencyType.OVERRIDE);
        for (const type of this.depTypes) {
            dependencies.push(...deps
                .filter((d) => d.type === type)
                .filter((d) => !(this.options.exclude ?? []).includes(d.name)));
        }
        return dependencies.map((d) => d.name);
    }
    createWorkflow(task, github, branch) {
        const schedule = this.options.workflowOptions?.schedule ??
            UpgradeDependenciesSchedule.DAILY;
        const workflowName = `${task.name}${branch ? `-${branch.replace(/\//g, "-")}` : ""}`;
        const workflow = github.addWorkflow(workflowName);
        const triggers = {
            workflowDispatch: {},
            schedule: schedule.cron.length > 0
                ? schedule.cron.map((e) => ({ cron: e }))
                : undefined,
        };
        workflow.on(triggers);
        const upgrade = this.createUpgrade(task, github, branch);
        const pr = this.createPr(workflow, upgrade);
        const jobs = {};
        jobs[upgrade.jobId] = upgrade.job;
        jobs[pr.jobId] = pr.job;
        workflow.addJobs(jobs);
        return workflow;
    }
    createUpgrade(task, github, branch) {
        const with_ = {
            ...(branch ? { ref: branch } : {}),
            ...(github.downloadLfs ? { lfs: true } : {}),
        };
        const steps = [
            github_1.WorkflowSteps.checkout({ with: with_ }),
            ...this._project.renderWorkflowSetup({ mutable: false }),
            {
                name: "Upgrade dependencies",
                run: this._project.runTaskCommand(task),
            },
        ];
        steps.push(...this.postBuildSteps);
        steps.push(...workflow_actions_1.WorkflowActions.uploadGitPatch({
            stepId: CREATE_PATCH_STEP_ID,
            outputName: PATCH_CREATED_OUTPUT,
        }));
        return {
            job: {
                name: "Upgrade",
                container: this.containerOptions,
                permissions: this.permissions,
                ...(0, runner_options_1.filteredRunsOnOptions)(this.options.workflowOptions?.runsOn, this.options.workflowOptions?.runsOnGroup),
                steps: steps,
                outputs: {
                    [PATCH_CREATED_OUTPUT]: {
                        stepId: CREATE_PATCH_STEP_ID,
                        outputName: PATCH_CREATED_OUTPUT,
                    },
                },
            },
            jobId: "upgrade",
            ref: branch,
        };
    }
    createPr(workflow, upgrade) {
        const credentials = this.options.workflowOptions?.projenCredentials ??
            workflow.projenCredentials;
        const semanticCommit = this.options.semanticCommit ?? "chore";
        return {
            job: github_1.WorkflowJobs.pullRequestFromPatch({
                patch: {
                    jobId: upgrade.jobId,
                    outputName: PATCH_CREATED_OUTPUT,
                    ref: upgrade.ref,
                },
                workflowName: workflow.name,
                credentials,
                ...(0, runner_options_1.filteredRunsOnOptions)(this.options.workflowOptions?.runsOn, this.options.workflowOptions?.runsOnGroup),
                pullRequestTitle: `${semanticCommit}(deps): ${this.pullRequestTitle}`,
                pullRequestDescription: "Upgrades project dependencies.",
                gitIdentity: this.gitIdentity,
                assignees: this.options.workflowOptions?.assignees,
                labels: this.options.workflowOptions?.labels,
                signoff: this.options.signoff,
            }),
            jobId: "pr",
        };
    }
}
exports.UpgradeDependencies = UpgradeDependencies;
_a = JSII_RTTI_SYMBOL_1;
UpgradeDependencies[_a] = { fqn: "projen.javascript.UpgradeDependencies", version: "0.79.27" };
/**
 * How often to check for new versions and raise pull requests for version upgrades.
 */
class UpgradeDependenciesSchedule {
    /**
     * Create a schedule from a raw cron expression.
     */
    static expressions(cron) {
        return new UpgradeDependenciesSchedule(cron);
    }
    constructor(cron) {
        this.cron = cron;
    }
}
exports.UpgradeDependenciesSchedule = UpgradeDependenciesSchedule;
_b = JSII_RTTI_SYMBOL_1;
UpgradeDependenciesSchedule[_b] = { fqn: "projen.javascript.UpgradeDependenciesSchedule", version: "0.79.27" };
/**
 * Disables automatic upgrades.
 */
UpgradeDependenciesSchedule.NEVER = new UpgradeDependenciesSchedule([]);
/**
 * At 00:00.
 */
UpgradeDependenciesSchedule.DAILY = new UpgradeDependenciesSchedule(["0 0 * * *"]);
/**
 * At 00:00 on every day-of-week from Monday through Friday.
 */
UpgradeDependenciesSchedule.WEEKDAY = new UpgradeDependenciesSchedule([
    "0 0 * * 1-5",
]);
/**
 * At 00:00 on Monday.
 */
UpgradeDependenciesSchedule.WEEKLY = new UpgradeDependenciesSchedule([
    "0 0 * * 1",
]);
/**
 * At 00:00 on day-of-month 1.
 */
UpgradeDependenciesSchedule.MONTHLY = new UpgradeDependenciesSchedule([
    "0 0 1 * *",
]);
//# sourceMappingURL=data:application/json;base64,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