"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonPatch = exports.TestFailureBehavior = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// inspired by https://github.com/cdk8s-team/cdk8s-core/blob/2.x/src/json-patch.ts
// under Apache 2.0 license
const fast_json_patch_1 = require("fast-json-patch");
var TestFailureBehavior;
(function (TestFailureBehavior) {
    /**
     * Skip the current patch operation and continue with the next operation.
     */
    TestFailureBehavior["SKIP"] = "skip";
    /**
     * Fail the whole file synthesis.
     */
    TestFailureBehavior["FAIL_SYNTHESIS"] = "fail";
})(TestFailureBehavior || (exports.TestFailureBehavior = TestFailureBehavior = {}));
const TEST_FAILURE_BEHAVIOR_SYMBOL = Symbol.for("testFailureBehavior");
/**
 * Utility for applying RFC-6902 JSON-Patch to a document.
 *
 * Use the the `JsonPatch.apply(doc, ...ops)` function to apply a set of
 * operations to a JSON document and return the result.
 *
 * Operations can be created using the factory methods `JsonPatch.add()`,
 * `JsonPatch.remove()`, etc.
 *
 * @example
 *
 *const output = JsonPatch.apply(input,
 *  JsonPatch.replace('/world/hi/there', 'goodbye'),
 *  JsonPatch.add('/world/foo/', 'boom'),
 *  JsonPatch.remove('/hello'));
 *
 */
class JsonPatch {
    /**
     * Applies a set of JSON-Patch (RFC-6902) operations to `document` and returns the result.
     * @param document The document to patch
     * @param ops The operations to apply
     * @returns The result document
     */
    static apply(document, ...ops) {
        try {
            const result = (0, fast_json_patch_1.applyPatch)(document, (0, fast_json_patch_1.deepClone)(ops.map((o) => o._toJson())));
            return result.newDocument;
        }
        catch (e) {
            if (e instanceof fast_json_patch_1.JsonPatchError && e.name === "TEST_OPERATION_FAILED") {
                const op = ops[e.index];
                if (TEST_FAILURE_BEHAVIOR_SYMBOL in op) {
                    const failureBehavior = op[TEST_FAILURE_BEHAVIOR_SYMBOL];
                    if (failureBehavior === TestFailureBehavior.SKIP) {
                        return document;
                    }
                }
            }
            throw e;
        }
    }
    /**
     * Adds a value to an object or inserts it into an array. In the case of an
     * array, the value is inserted before the given index. The - character can be
     * used instead of an index to insert at the end of an array.
     *
     * @example JsonPatch.add('/biscuits/1', { "name": "Ginger Nut" })
     */
    static add(path, value) {
        return new JsonPatch({
            op: "add",
            path,
            value,
        });
    }
    /**
     * Removes a value from an object or array.
     *
     * @example JsonPatch.remove('/biscuits')
     * @example JsonPatch.remove('/biscuits/0')
     */
    static remove(path) {
        return new JsonPatch({ op: "remove", path });
    }
    /**
     * Replaces a value. Equivalent to a “remove” followed by an “add”.
     *
     * @example JsonPatch.replace('/biscuits/0/name', 'Chocolate Digestive')
     */
    static replace(path, value) {
        return new JsonPatch({
            op: "replace",
            path,
            value,
        });
    }
    /**
     * Copies a value from one location to another within the JSON document. Both
     * from and path are JSON Pointers.
     *
     * @example JsonPatch.copy('/biscuits/0', '/best_biscuit')
     */
    static copy(from, path) {
        return new JsonPatch({ op: "copy", from, path });
    }
    /**
     * Moves a value from one location to the other. Both from and path are JSON Pointers.
     *
     * @example JsonPatch.move('/biscuits', '/cookies')
     */
    static move(from, path) {
        return new JsonPatch({ op: "move", from, path });
    }
    /**
     * Tests that the specified value is set in the document. If the test fails,
     * then the patch as a whole should not apply.
     *
     * @example JsonPatch.test('/best_biscuit/name', 'Choco Leibniz')
     */
    static test(path, value, failureBehavior = TestFailureBehavior.SKIP) {
        const patch = new JsonPatch({
            op: "test",
            path,
            value,
        });
        Object.defineProperty(patch, TEST_FAILURE_BEHAVIOR_SYMBOL, {
            writable: false,
            value: failureBehavior,
        });
        return patch;
    }
    /**
     * Escapes a json pointer path
     * @param path The raw pointer
     * @return the Escaped path
     */
    static escapePath(path) {
        return (0, fast_json_patch_1.escapePathComponent)(path);
    }
    constructor(operation) {
        this.operation = operation;
    }
    /**
     * Returns the JSON representation of this JSON patch operation.
     *
     * @internal
     */
    _toJson() {
        return this.operation;
    }
}
exports.JsonPatch = JsonPatch;
_a = JSII_RTTI_SYMBOL_1;
JsonPatch[_a] = { fqn: "projen.JsonPatch", version: "0.79.27" };
//# sourceMappingURL=data:application/json;base64,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