"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogLevel = exports.Logger = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const chalk = require("chalk");
const common_1 = require("./common");
const component_1 = require("./component");
const logging_1 = require("./logging");
const util_1 = require("./util");
const constructs_1 = require("./util/constructs");
/**
 * Project-level logging utilities.
 */
class Logger extends component_1.Component {
    constructor(scope, options = {}) {
        const project = (0, constructs_1.findClosestProject)(scope);
        super(scope, `${new.target.name}#${project.name}`);
        // if we are running inside a test, default to no logs
        const defaultLevel = common_1.IS_TEST_RUN ? LogLevel.OFF : LogLevel.INFO;
        this.level = options.level ?? defaultLevel;
        this.usePrefix = options.usePrefix ?? false;
    }
    /**
     * Log a message to stderr with a given logging level. The message will be
     * printed as long as `logger.level` is set to the message's severity or higher.
     *
     * @param level Logging verbosity
     * @param text strings or objects to print
     */
    log(level, ...text) {
        if (level === LogLevel.OFF) {
            throw new Error("Cannot log a message with level LogLevel.OFF");
        }
        let maxLevel = this.level;
        // logging level can be overridden through the --debug CLI option
        if ((0, util_1.isTruthy)(process.env.DEBUG) && maxLevel < LogLevel.DEBUG) {
            maxLevel = LogLevel.DEBUG;
        }
        if (level <= maxLevel) {
            const color = this.colorForLogLevel(level) ??
                ((...values) => values.join(" "));
            const prefix = this.usePrefix ? `[${this.project.name}] ` : "";
            console.error(`${logging_1.ICON} ${prefix}${color(...text)}`);
        }
    }
    /**
     * Log a message to stderr with VERBOSE severity
     * @param text strings or objects to print
     */
    verbose(...text) {
        this.log(LogLevel.VERBOSE, ...text);
    }
    /**
     * Log a message to stderr with DEBUG severity
     * @param text strings or objects to print
     */
    debug(...text) {
        this.log(LogLevel.DEBUG, ...text);
    }
    /**
     * Log a message to stderr with INFO severity
     * @param text strings or objects to print
     */
    info(...text) {
        this.log(LogLevel.INFO, ...text);
    }
    /**
     * Log a message to stderr with WARN severity
     * @param text strings or objects to print
     */
    warn(...text) {
        this.log(LogLevel.WARN, ...text);
    }
    /**
     * Log a message to stderr with ERROR severity
     * @param text strings or objects to print
     */
    error(...text) {
        this.log(LogLevel.ERROR, ...text);
    }
    colorForLogLevel(level) {
        switch (level) {
            case LogLevel.ERROR:
                return chalk.red;
            case LogLevel.WARN:
                return chalk.yellow;
            case LogLevel.INFO:
                return chalk.cyan;
            case LogLevel.DEBUG:
                return chalk.gray;
            case LogLevel.VERBOSE:
                return chalk.white;
            default:
                return chalk.white;
        }
    }
}
exports.Logger = Logger;
_a = JSII_RTTI_SYMBOL_1;
Logger[_a] = { fqn: "projen.Logger", version: "0.79.27" };
/**
 * Logging verbosity.
 */
var LogLevel;
(function (LogLevel) {
    LogLevel["OFF"] = "00.off";
    LogLevel["ERROR"] = "10.error";
    LogLevel["WARN"] = "20.warn";
    LogLevel["INFO"] = "30.info";
    LogLevel["DEBUG"] = "40.debug";
    LogLevel["VERBOSE"] = "50.verbose";
})(LogLevel || (exports.LogLevel = LogLevel = {}));
//# sourceMappingURL=data:application/json;base64,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