import { IConstruct } from "constructs";
import { FileBase, FileBaseOptions, IResolver } from "./file";
import { JsonPatch } from "./json-patch";
/**
 * Options for `ObjectFile`.
 */
export interface ObjectFileOptions extends FileBaseOptions {
    /**
     * The object that will be serialized. You can modify the object's contents
     * before synthesis.
     *
     * Serialization of the object is similar to JSON.stringify with few enhancements:
     * - values that are functions will be called during synthesis and the result will be serialized - this allow to have lazy values.
     * - `Set` will be converted to array
     * - `Map` will be converted to a plain object ({ key: value, ... }})
     * - `RegExp` without flags will be converted to string representation of the source
     *
     *  @default {} an empty object (use `file.obj` to mutate).
     */
    readonly obj?: any;
    /**
     * Omits empty objects and arrays.
     * @default false
     */
    readonly omitEmpty?: boolean;
}
/**
 * Represents an Object file.
 */
export declare abstract class ObjectFile extends FileBase {
    /**
     * The output object. This object can be mutated until the project is
     * synthesized.
     */
    private readonly obj;
    /**
     * An object to be merged on top of `obj` after the resolver is called
     */
    private readonly rawOverrides;
    /**
     * Indicates if empty objects and arrays are omitted from the output object.
     */
    readonly omitEmpty: boolean;
    /**
     * patches to be applied to `obj` after the resolver is called
     */
    private readonly patchOperations;
    constructor(scope: IConstruct, filePath: string, options: ObjectFileOptions);
    /**
     * Adds an override to the synthesized object file.
     *
     * If the override is nested, separate each nested level using a dot (.) in the path parameter.
     * If there is an array as part of the nesting, specify the index in the path.
     *
     * To include a literal `.` in the property name, prefix with a `\`. In most
     * programming languages you will need to write this as `"\\."` because the
     * `\` itself will need to be escaped.
     *
     * For example,
     * ```typescript
     * project.tsconfig.file.addOverride('compilerOptions.alwaysStrict', true);
     * project.tsconfig.file.addOverride('compilerOptions.lib', ['dom', 'dom.iterable', 'esnext']);
     * ```
     * would add the overrides
     * ```json
     * "compilerOptions": {
     *   "alwaysStrict": true,
     *   "lib": [
     *     "dom",
     *     "dom.iterable",
     *     "esnext"
     *   ]
     *   ...
     * }
     * ...
     * ```
     *
     * @param path - The path of the property, you can use dot notation to
     *        override values in complex types. Any intermediate keys
     *        will be created as needed.
     * @param value - The value. Could be primitive or complex.
     */
    addOverride(path: string, value: any): void;
    /**
     * Adds to an array in the synthesized object file.
     *
     * If the array is nested, separate each nested level using a dot (.) in the path parameter.
     * If there is an array as part of the nesting, specify the index in the path.
     *
     * To include a literal `.` in the property name, prefix with a `\`. In most
     * programming languages you will need to write this as `"\\."` because the
     * `\` itself will need to be escaped.
     *
     * For example, with the following object file
     * ```json
     * "compilerOptions": {
     *   "exclude": ["node_modules"],
     *   "lib": ["es2019"]
     *   ...
     * }
     * ...
     * ```
     *
     * ```typescript
     * project.tsconfig.file.addToArray('compilerOptions.exclude', 'coverage');
     * project.tsconfig.file.addToArray('compilerOptions.lib', 'dom', 'dom.iterable', 'esnext');
     * ```
     * would result in the following object file
     * ```json
     * "compilerOptions": {
     *   "exclude": ["node_modules", "coverage"],
     *   "lib": ["es2019", "dom", "dom.iterable", "esnext"]
     *   ...
     * }
     * ...
     * ```
     *
     * @param path - The path of the property, you can use dot notation to
     *        att to arrays in complex types. Any intermediate keys
     *        will be created as needed.
     * @param values - The values to add. Could be primitive or complex.
     */
    addToArray(path: string, ...values: any): void;
    /**
     * Applies an RFC 6902 JSON-patch to the synthesized object file.
     * See https://datatracker.ietf.org/doc/html/rfc6902 for more information.
     *
     * For example, with the following object file
     * ```json
     * "compilerOptions": {
     *   "exclude": ["node_modules"],
     *   "lib": ["es2019"]
     *   ...
     * }
     * ...
     * ```
     *
     * ```typescript
     * project.tsconfig.file.patch(JsonPatch.add("/compilerOptions/exclude/-", "coverage"));
     * project.tsconfig.file.patch(JsonPatch.replace("/compilerOptions/lib", ["dom", "dom.iterable", "esnext"]));
     * ```
     * would result in the following object file
     * ```json
     * "compilerOptions": {
     *   "exclude": ["node_modules", "coverage"],
     *   "lib": ["dom", "dom.iterable", "esnext"]
     *   ...
     * }
     * ...
     * ```
     *
     * @param patches - The patch operations to apply
     */
    patch(...patches: JsonPatch[]): void;
    /**
     * Syntactic sugar for `addOverride(path, undefined)`.
     * @param path The path of the value to delete
     */
    addDeletionOverride(path: string): void;
    protected synthesizeContent(resolver: IResolver): string | undefined;
}
