"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectType = exports.Project = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const os_1 = require("os");
const path = require("path");
const constructs_1 = require("constructs");
const glob = require("glob");
const cleanup_1 = require("./cleanup");
const common_1 = require("./common");
const dependencies_1 = require("./dependencies");
const file_1 = require("./file");
const gitattributes_1 = require("./gitattributes");
const ignore_file_1 = require("./ignore-file");
const render_options_1 = require("./javascript/render-options");
const json_1 = require("./json");
const logger_1 = require("./logger");
const object_file_1 = require("./object-file");
const project_build_1 = require("./project-build");
const projenrc_json_1 = require("./projenrc-json");
const renovatebot_1 = require("./renovatebot");
const tasks_1 = require("./tasks");
const util_1 = require("./util");
const constructs_2 = require("./util/constructs");
/**
 * The default output directory for a project if none is specified.
 */
const DEFAULT_OUTDIR = ".";
/**
 * Base project
 */
class Project extends constructs_1.Construct {
    /**
     * Test whether the given construct is a project.
     */
    static isProject(x) {
        return (0, constructs_2.isProject)(x);
    }
    /**
     * Find the closest ancestor project for given construct.
     * When given a project, this it the project itself.
     *
     * @throws when no project is found in the path to the root
     */
    static of(construct) {
        return (0, constructs_2.findClosestProject)(construct);
    }
    /**
     * The command to use in order to run the projen CLI.
     */
    get projenCommand() {
        return this._projenCommand ?? "npx projen";
    }
    constructor(options) {
        const outdir = determineOutdir(options.parent, options.outdir);
        const autoId = `${new.target.name}#${options.name}@${path.normalize(options.outdir ?? "<root>")}`;
        if (options.parent?.subprojects.find((p) => p.outdir === outdir)) {
            throw new Error(`There is already a subproject with "outdir": ${outdir}`);
        }
        super(options.parent, autoId);
        this.tips = new Array();
        (0, constructs_2.tagAsProject)(this);
        this.node.addMetadata("type", "project");
        this.node.addMetadata("construct", new.target.name);
        this.initProject = (0, render_options_1.resolveInitProject)(options);
        this.name = options.name;
        this.parent = options.parent;
        this.excludeFromCleanup = [];
        this._ejected = (0, util_1.isTruthy)(process.env.PROJEN_EJECTING);
        this._projenCommand = options.projenCommand;
        if (this.ejected) {
            this._projenCommand = "scripts/run-task";
        }
        this.outdir = outdir;
        // ------------------------------------------------------------------------
        this.gitattributes = new gitattributes_1.GitAttributesFile(this);
        this.annotateGenerated("/.projen/**"); // contents  of the .projen/ directory are generated by projen
        this.annotateGenerated(`/${this.gitattributes.path}`); // the .gitattributes file itself is generated
        if (options.gitOptions?.lfsPatterns) {
            for (const pattern of options.gitOptions.lfsPatterns) {
                this.gitattributes.addAttributes(pattern, "filter=lfs", "diff=lfs", "merge=lfs", "-text");
            }
        }
        this.gitignore = new ignore_file_1.IgnoreFile(this, ".gitignore", options.gitIgnoreOptions);
        this.gitignore.exclude("node_modules/"); // created by running `npx projen`
        this.gitignore.include(`/${this.gitattributes.path}`);
        // oh no: tasks depends on gitignore so it has to be initialized after
        // smells like dep injection but god forbid.
        this.tasks = new tasks_1.Tasks(this);
        if (!this.ejected) {
            this.defaultTask = this.tasks.addTask(Project.DEFAULT_TASK, {
                description: "Synthesize project files",
            });
            // Subtasks should call the root task for synth
            if (this.parent) {
                this.defaultTask.exec(`${this.projenCommand} ${Project.DEFAULT_TASK}`, {
                    cwd: path.relative(this.outdir, this.root.outdir),
                });
            }
            if (!this.parent) {
                this.ejectTask = this.tasks.addTask("eject", {
                    description: "Remove projen from the project",
                    env: {
                        PROJEN_EJECTING: "true",
                    },
                });
                this.ejectTask.spawn(this.defaultTask);
            }
        }
        this.projectBuild = new project_build_1.ProjectBuild(this);
        this.deps = new dependencies_1.Dependencies(this);
        this.logger = new logger_1.Logger(this, options.logging);
        const projenrcJson = options.projenrcJson ?? false;
        if (!this.parent && projenrcJson) {
            new projenrc_json_1.ProjenrcJson(this, options.projenrcJsonOptions);
        }
        if (options.renovatebot) {
            new renovatebot_1.Renovatebot(this, options.renovatebotOptions);
        }
        this.commitGenerated = options.commitGenerated ?? true;
        if (!this.ejected) {
            new json_1.JsonFile(this, cleanup_1.FILE_MANIFEST, {
                omitEmpty: true,
                obj: () => ({
                    // replace `\` with `/` to ensure paths match across platforms
                    files: this.files
                        .filter((f) => f.readonly)
                        .map((f) => (0, util_1.normalizePersistedPath)(f.path)),
                }),
                // This file is used by projen to track the generated files, so must be committed.
                committed: true,
            });
        }
    }
    /**
     * The root project.
     */
    get root() {
        return (0, constructs_2.isProject)(this.node.root) ? this.node.root : this;
    }
    /**
     * Returns all the components within this project.
     */
    get components() {
        return this.node
            .findAll()
            .filter((c) => (0, constructs_2.isComponent)(c) && c.project.node.path === this.node.path);
    }
    /**
     * Returns all the subprojects within this project.
     */
    get subprojects() {
        return this.node.children.filter(constructs_2.isProject);
    }
    /**
     * All files in this project.
     */
    get files() {
        return this.components
            .filter(isFile)
            .sort((f1, f2) => f1.path.localeCompare(f2.path));
    }
    /**
     * Adds a new task to this project. This will fail if the project already has
     * a task with this name.
     *
     * @param name The task name to add
     * @param props Task properties
     */
    addTask(name, props = {}) {
        return this.tasks.addTask(name, props);
    }
    /**
     * Removes a task from a project.
     *
     * @param name The name of the task to remove.
     *
     * @returns The `Task` that was removed, otherwise `undefined`.
     */
    removeTask(name) {
        return this.tasks.removeTask(name);
    }
    get buildTask() {
        return this.projectBuild.buildTask;
    }
    get compileTask() {
        return this.projectBuild.compileTask;
    }
    get testTask() {
        return this.projectBuild.testTask;
    }
    get preCompileTask() {
        return this.projectBuild.preCompileTask;
    }
    get postCompileTask() {
        return this.projectBuild.postCompileTask;
    }
    get packageTask() {
        return this.projectBuild.packageTask;
    }
    /**
     * Finds a file at the specified relative path within this project and all
     * its subprojects.
     *
     * @param filePath The file path. If this path is relative, it will be resolved
     * from the root of _this_ project.
     * @returns a `FileBase` or undefined if there is no file in that path
     */
    tryFindFile(filePath) {
        const absolute = path.isAbsolute(filePath)
            ? filePath
            : path.resolve(this.outdir, filePath);
        const candidate = this.node
            .findAll()
            .find((c) => (0, constructs_2.isComponent)(c) && isFile(c) && c.absolutePath === absolute);
        return candidate;
    }
    /**
     * Finds a json file by name.
     * @param filePath The file path.
     * @deprecated use `tryFindObjectFile`
     */
    tryFindJsonFile(filePath) {
        const file = this.tryFindObjectFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof json_1.JsonFile)) {
            throw new Error(`found file ${filePath} but it is not a JsonFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Finds an object file (like JsonFile, YamlFile, etc.) by name.
     * @param filePath The file path.
     */
    tryFindObjectFile(filePath) {
        const file = this.tryFindFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof object_file_1.ObjectFile)) {
            throw new Error(`found file ${filePath} but it is not a ObjectFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Finds a file at the specified relative path within this project and removes
     * it.
     *
     * @param filePath The file path. If this path is relative, it will be
     * resolved from the root of _this_ project.
     * @returns a `FileBase` if the file was found and removed, or undefined if
     * the file was not found.
     */
    tryRemoveFile(filePath) {
        const candidate = this.tryFindFile(filePath);
        if (candidate) {
            candidate.node.scope?.node.tryRemoveChild(candidate.node.id);
            return candidate;
        }
        return undefined;
    }
    /**
     * Prints a "tip" message during synthesis.
     * @param message The message
     * @deprecated - use `project.logger.info(message)` to show messages during synthesis
     */
    addTip(message) {
        this.tips.push(message);
    }
    /**
     * Exclude the matching files from pre-synth cleanup. Can be used when, for example, some
     * source files include the projen marker and we don't want them to be erased during synth.
     *
     * @param globs The glob patterns to match
     */
    addExcludeFromCleanup(...globs) {
        this.excludeFromCleanup.push(...globs);
    }
    /**
     * Returns the shell command to execute in order to run a task.
     *
     * By default, this is `npx projen@<version> <task>`
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        const pj = this._projenCommand ?? `npx projen@${common_1.PROJEN_VERSION}`;
        return `${pj} ${task.name}`;
    }
    /**
     * Exclude these files from the bundled package. Implemented by project types based on the
     * packaging mechanism. For example, `NodeProject` delegates this to `.npmignore`.
     *
     * @param _pattern The glob pattern to exclude
     */
    addPackageIgnore(_pattern) {
        // nothing to do at the abstract level
    }
    /**
     * Adds a .gitignore pattern.
     * @param pattern The glob pattern to ignore.
     */
    addGitIgnore(pattern) {
        this.gitignore.addPatterns(pattern);
    }
    /**
     * Consider a set of files as "generated". This method is implemented by
     * derived classes and used for example, to add git attributes to tell GitHub
     * that certain files are generated.
     *
     * @param _glob the glob pattern to match (could be a file path).
     */
    annotateGenerated(_glob) {
        // nothing to do at the abstract level
    }
    /**
     * Synthesize all project files into `outdir`.
     *
     * 1. Call "this.preSynthesize()"
     * 2. Delete all generated files
     * 3. Synthesize all subprojects
     * 4. Synthesize all components of this project
     * 5. Call "postSynthesize()" for all components of this project
     * 6. Call "this.postSynthesize()"
     */
    synth() {
        const outdir = this.outdir;
        this.logger.debug("Synthesizing project...");
        this.preSynthesize();
        for (const comp of this.components) {
            comp.preSynthesize();
        }
        // we exclude all subproject directories to ensure that when subproject.synth()
        // gets called below after cleanup(), subproject generated files are left intact
        for (const subproject of this.subprojects) {
            this.addExcludeFromCleanup(subproject.outdir + "/**");
        }
        // delete orphaned files before we start synthesizing new ones
        (0, cleanup_1.cleanup)(outdir, this.files.map((f) => (0, util_1.normalizePersistedPath)(f.path)), this.excludeFromCleanup);
        for (const subproject of this.subprojects) {
            subproject.synth();
        }
        for (const comp of this.components) {
            comp.synthesize();
        }
        if (!(0, util_1.isTruthy)(process.env.PROJEN_DISABLE_POST)) {
            for (const comp of this.components) {
                comp.postSynthesize();
            }
            // project-level hook
            this.postSynthesize();
        }
        if (this.ejected) {
            this.logger.debug("Ejecting project...");
            // Backup projenrc files
            const files = glob.sync(".projenrc.*", {
                cwd: this.outdir,
                dot: true,
                nodir: true,
                absolute: true,
            });
            for (const file of files) {
                (0, fs_1.renameSync)(file, `${file}.bak`);
            }
        }
        this.logger.debug("Synthesis complete");
    }
    /**
     * Whether or not the project is being ejected.
     */
    get ejected() {
        return this._ejected;
    }
    /**
     * Called before all components are synthesized.
     */
    preSynthesize() { }
    /**
     * Called after all components are synthesized. Order is *not* guaranteed.
     */
    postSynthesize() { }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "projen.Project", version: "0.79.27" };
/**
 * The name of the default task (the task executed when `projen` is run without arguments). Normally
 * this task should synthesize the project files.
 */
Project.DEFAULT_TASK = "default";
/**
 * Which type of project this is.
 *
 * @deprecated no longer supported at the base project level
 */
var ProjectType;
(function (ProjectType) {
    /**
     * This module may be a either a library or an app.
     */
    ProjectType["UNKNOWN"] = "unknown";
    /**
     * This is a library, intended to be published to a package manager and
     * consumed by other projects.
     */
    ProjectType["LIB"] = "lib";
    /**
     * This is an app (service, tool, website, etc). Its artifacts are intended to
     * be deployed or published for end-user consumption.
     */
    ProjectType["APP"] = "app";
})(ProjectType || (exports.ProjectType = ProjectType = {}));
/**
 * Resolves the project's output directory.
 */
function determineOutdir(parent, outdirOption) {
    if (parent && outdirOption && path.isAbsolute(outdirOption)) {
        throw new Error('"outdir" must be a relative path');
    }
    // if this is a subproject, it is relative to the parent
    if (parent) {
        if (!outdirOption) {
            throw new Error('"outdir" must be specified for subprojects');
        }
        return path.resolve(parent.outdir, outdirOption);
    }
    // if this is running inside a test and outdir is not explicitly set
    // use a temp directory (unless cwd is already under tmp)
    if (common_1.IS_TEST_RUN && !outdirOption) {
        const realCwd = (0, fs_1.realpathSync)(process.cwd());
        const realTmp = (0, fs_1.realpathSync)((0, os_1.tmpdir)());
        if (realCwd.startsWith(realTmp)) {
            return path.resolve(realCwd, outdirOption ?? DEFAULT_OUTDIR);
        }
        return (0, fs_1.mkdtempSync)(path.join((0, os_1.tmpdir)(), "projen."));
    }
    return path.resolve(outdirOption ?? DEFAULT_OUTDIR);
}
function isFile(c) {
    return c instanceof file_1.FileBase;
}
//# sourceMappingURL=data:application/json;base64,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