"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Projects = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const vm = require("vm");
const inventory_1 = require("./inventory");
const render_options_1 = require("./javascript/render-options");
const option_hints_1 = require("./option-hints");
/**
 * Programmatic API for projen.
 */
class Projects {
    /**
     * Creates a new project with defaults.
     *
     * This function creates the project type in-process (with in VM) and calls
     * `.synth()` on it (if `options.synth` is not `false`).
     *
     * At the moment, it also generates a `.projenrc.js` file with the same code
     * that was just executed. In the future, this will also be done by the project
     * type, so we can easily support multiple languages of projenrc.
     *
     * An environment variable (PROJEN_CREATE_PROJECT=true) is set within the VM
     * so that custom project types can detect whether the current synthesis is the
     * result of a new project creation (and take additional steps accordingly)
     */
    static createProject(options) {
        createProject(options);
    }
    constructor() { }
}
exports.Projects = Projects;
_a = JSII_RTTI_SYMBOL_1;
Projects[_a] = { fqn: "projen.Projects", version: "0.79.27" };
function resolveModulePath(moduleName) {
    // Default project resolution location
    if (moduleName === "projen") {
        return "./index";
    }
    // External projects need to load the module from the modules directory
    try {
        return path.dirname(require.resolve(path.join(moduleName, "package.json"), {
            paths: [process.cwd()],
        }));
    }
    catch (err) {
        throw new Error(`External project module '${moduleName}' could not be resolved.`);
    }
}
function createProject(opts) {
    const projectType = (0, inventory_1.resolveProjectType)(opts.projectFqn);
    const mod = resolveModulePath(projectType.moduleName);
    // "dir" is exposed as a top-level option to require users to specify a value for it
    opts.projectOptions.outdir = opts.dir;
    // Generated a random name space for imports used by options
    // This is so we can keep the top-level namespace as clean as possible
    const optionsImports = "_options" + Math.random().toString(36).slice(2);
    // pass the FQN of the project type to the project initializer so it can
    // generate the projenrc file.
    const { renderedOptions, imports } = (0, render_options_1.renderJavaScriptOptions)({
        bootstrap: true,
        comments: opts.optionHints ?? option_hints_1.InitProjectOptionHints.FEATURED,
        type: projectType,
        args: opts.projectOptions,
        omitFromBootstrap: ["outdir"],
        prefixImports: optionsImports,
    });
    const initProjectCode = new Array();
    // generate a random variable name because jest tests appear to share
    // VM contexts, causing
    //
    // > SyntaxError: Identifier 'project' has already been declared
    //
    // errors if this isn't unique
    const varName = "project" + Math.random().toString(36).slice(2);
    initProjectCode.push(`const ${varName} = new ${projectType.typename}(${renderedOptions});`);
    if (opts.synth ?? true) {
        initProjectCode.push(`${varName}.synth();`);
    }
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mainModule = require(mod);
    const ctx = vm.createContext({
        ...mainModule,
        [optionsImports]: {
            ...imports.modules.reduce((optionsContext, currentModule) => ({
                ...optionsContext,
                // eslint-disable-next-line @typescript-eslint/no-require-imports
                [currentModule]: require(resolveModulePath(currentModule)),
            }), {}),
        },
    });
    const postSynth = opts.post ?? true;
    process.env.PROJEN_DISABLE_POST = (!postSynth).toString();
    process.env.PROJEN_CREATE_PROJECT = "true";
    vm.runInContext(initProjectCode.join("\n"), ctx);
}
//# sourceMappingURL=data:application/json;base64,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